import os
import click
import requests

from google.protobuf import text_format
from six.moves import reduce

from sandbox.projects.release_machine.components.configs import all as all_cfg
from sandbox.projects.release_machine.components.config_core import config_serializer


@click.group()
def main():
    pass


@main.command()
@click.argument("component_name")
@click.option("--token", default=None)
def serialize(component_name, token):
    cfg = all_cfg.get_config(component_name)
    serialized = config_serializer.config2proto(cfg, token=token)
    click.echo(text_format.MessageToString(serialized))


@main.command()
def touch_all():
    for component_name in all_cfg.get_all_names():
        click.echo("Touching {}".format(component_name))
        all_cfg.get_config(component_name)
        click.echo("... touched!")
    click.echo("=" * 100)
    click.echo("It's done. You've touched each and every config in Release Machine. Are you happy now?")


@main.command()
@click.argument("component_name")
@click.argument("parameter")
def get_setting(component_name, parameter):
    cfg = all_cfg.get_config(component_name)
    value = reduce(getattr, [cfg] + parameter.split('.'))
    click.echo("{} cfg.{} = {}".format(component_name, parameter, value))


@main.command()
@click.option("--token", default=None)
def request_tokens(token):

    if token is None:
        token = os.environ.get("COMMON_RELEASE_TOKEN")

    if not token:
        click.echo("No token provided")
        exit(1)

    headers = {
        "Authorization": "OAuth {}".format(token)
    }

    success_count = skip_count = failure_count = 0

    for component_name in all_cfg.get_all_ci_names():

        cfg = all_cfg.get_config(component_name)

        if cfg.ci_cfg.secret != "sec-01desry8fbgvnkbeybem81ferv":
            click.echo("{} - skip".format(component_name))
            skip_count += 1
            continue

        try:

            url = "https://a.yandex-team.ru/gateway/ci/DelegateToken?configDir={}".format(
                cfg.ci_cfg.a_yaml_dir,
            )

            response = requests.get(url, headers=headers)

            if response.status_code == requests.codes.ok:
                click.echo("{} - success".format(component_name))
                success_count += 1
            else:
                click.echo("{} - failure".format(component_name))
                failure_count += 1

        except:
            click.echo("{} - exception".format(component_name))
            failure_count += 1
            continue

    click.echo("-" * 80)

    click.echo(
        "{} - success, {} - failed, {} - skipped".format(
            success_count,
            failure_count,
            skip_count,
        ),
    )


@main.command()
def ipython():
    import IPython.terminal.interactiveshell as ipsh
    shell = ipsh.TerminalInteractiveShell(banner2="Release Machine config shell")
    shell.show_banner()
    shell.mainloop()


if __name__ == "__main__":
    main()
