# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers


class BrandsCfg(configs.ReferenceBranchedConfig):
    name = "brands"
    responsible = "losev"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_db = "modadvert-brands-trunk"
        trunk_task_owner = "MODADVERT_RELEASE"
        branch_db_template = "modadvert-brands-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="MODADVERT_BUILD_TASK",
                        ctx={
                            "packages": ";".join([
                                "modadvert/programs/brands/classifier/package.json",
                                "modadvert/programs/brands/api/docker_package.json",
                                "modadvert/resources/brands/dev/package.json",
                                "modadvert/resources/brands/prod/package.json",
                                "modadvert/programs/brands/classifier_circuit/package.json",
                            ]),
                            "resource_type": ";".join([
                                "MODADVERT_BRANDS_CLASSIFIER",
                                "MODADVERT_BRANDS_API_BINARY",
                                "MODADVERT_BRANDS_CONFIG_TESTING",
                                "MODADVERT_BRANDS_CONFIG_PROD",
                                "MODADVERT_BRANDS_CLASSIFIER_CIRCUIT",
                            ]),
                            "package_type": "tarball",
                            "use_new_format": True,
                        },
                        out={
                            "MODADVERT_BRANDS_CLASSIFIER": 100,
                            "MODADVERT_BRANDS_API_BINARY": 100,
                            "MODADVERT_BRANDS_CONFIG_TESTING": 100,
                            "MODADVERT_BRANDS_CONFIG_PROD": 100,
                            "MODADVERT_BRANDS_CLASSIFIER_CIRCUIT": 100,
                        }
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="MODADVERT_PUSH_DOCKER_IMAGE",
                        job_params={
                            "job_name_parameter": "MODADVERT_BRANDS_API_BINARY",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="docker_package_checkout_arcadia_from_url",
                                        output_key="checkout_arcadia_from_url",
                                        transform=lambda x, params: x,
                                    )
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="packaged_resource_id",
                                        resource_name="MODADVERT_BRANDS_API_BINARY",
                                    )
                                )
                            )
                        ),
                        ctx={
                            "resource_type": "tarball",
                            "resource_type_name": "MODADVERT_BRANDS_API_BINARY",
                            "registry_login": "robot-modadvert-test",
                            "vault_item_owner": "robot-modadvert-test",
                            "vault_item_name": "registry-token",
                        },
                    )
                )
                for environment in ["TESTING", "PROD"]:
                    branch_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="MODADVERT_DEPLOY_BRANDS",
                            job_params={
                                "apiargs": {
                                    "hidden": False,
                                },
                                "job_name_parameter": "DEPLOY_{}".format(environment),
                            },
                            job_arrows=(
                                jg_arrows.ParamsData(
                                    input_key="release_number",
                                    transform=jg_utils.get_major_release_number,
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="brands_classifier_resource",
                                            resource_name="MODADVERT_BRANDS_CLASSIFIER",
                                        ),
                                        jg_job_data.ParentDataResource(
                                            input_key="brands_classifier_binaries_resource",
                                            resource_name="MODADVERT_BRANDS_CLASSIFIER_CIRCUIT",
                                        ),
                                        jg_job_data.ParentDataResource(
                                            input_key="brands_config_resource",
                                            resource_name="MODADVERT_BRANDS_CONFIG_{}".format(environment),
                                        )
                                    )
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="MODADVERT_BRANDS_API_BINARY",
                                    parent_job_data=(
                                        jg_job_data.ParentDataCtx(
                                            input_key="release_tag_number",
                                            output_key="release_tag_number",
                                            transform=lambda x, params: str(x),
                                        )
                                    )
                                ),
                                jg_job_triggers.JobTriggerActionRelease(
                                    job_name_parameter=rm_const.ReleaseStatus.stable,
                                ) if environment == "PROD" else None,
                            ),
                            ctx={
                                "vault_user": "robot-modadvert-test",
                                "st_token_vault_name": "st-token",
                                "release_environment": environment,
                            }
                        )
                    )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                release_part.append(
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "brands_api_binary",
                                        "MODADVERT_BRANDS_API_BINARY",
                                    )
                                )
                            )
                        )
                    )
                )
                release_part.append(
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                    )
                )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def change_frequency(self):
                uppercase_name = self.name.upper()
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "_{}_CHANGELOG_MINOR".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_{}_NEW_TAG".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_PUSH_MODADVERT_BRANDS_API_BINARY_{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

    class Releases(configs.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        resources_info = [
            configs.ReleasedResourceInfo(
                name="brands",
                resource_type="MODADVERT_BRANDS_API_BINARY",
                resource_name="brands_api_binary",
            ),
        ]
        allow_robots_to_release_stable = True
        release_followers_permanent = [
            "losev",
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""

        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            """Mail notifications configuration"""

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = []
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "losev"
            queue = "BRANDSRELEASE"
            dev_queue = "MODDEV"
            summary_template = u"Приемка справочника брендов (brands) {}"
            workflow = {}
            followers = [
                "losev",
                "losev",
                "vsuvorov",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "modadvert/releases/brands/"
        dirs = [
            "arcadia/modadvert/libs/brands",
            "arcadia/modadvert/programs/brands",
            "arcadia/modadvert/resources/brands",
        ]
        review_groups = [
            "modadvert",
        ]

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "modadvert/brands/"
