# -*- coding: utf-8 -*-

import sandbox.common.types.notification as task_notifications

import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri

FOLLOWERS = [
    'buryakov',
    'ajkon',
    'leftmain',
    'sharamed'
]


class BsCollectorCfg(configs.ReferenceBranchedConfig):
    name = "bs_collector"
    responsible = "buryakov"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "AB-TESTING"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            YT_POOL_ABT_METRICS_DIFF = "abt"

            # Build z2 package
            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part

                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="KOSHER_YA_PACKAGE",
                        ctx={
                            "packages": "packages/debian/mapreduce/yandex-search-ab-testing-bs/yandex-search-ab-testing-bs.json",
                            "package_type": "debian",
                            "build_type": "release",
                            "strip_binaries": True,
                            "compress_package_archive": True,
                            "key_user": "robot-eksperimentus",
                            "publish_package": True,
                            "publish_to": "search-precise",
                            "resource_type": "BS_COLLECTOR_YA_PACKAGE",
                            "run_tests": False,  # small tests
                            "run_medium_tests": False,
                            "release_to_nanny": False,
                            # "checkout_arcadia_from_url": "arcadia:/arc/trunk/arcadia@rev" - automatically
                        },
                        job_params={
                            "apiargs": {
                                "notifications": [{
                                    "statuses": ["SUCCESS", "FAILURE", "EXCEPTION", "NO_RES", "TIMEOUT", "STOPPED", "EXPIRED"],
                                    "recipients": FOLLOWERS,
                                    "transport": task_notifications.Transport.EMAIL
                                }],
                                "requirements": {
                                    "disk_space": 250 * (2 ** 30),  # 250 GB
                                    "client_tags": 'SSD',
                                },
                                "kill_timeout": 4 * 3600,  # 4 hours
                            }
                        },
                        out={
                            "BS_COLLECTOR_YA_PACKAGE": 300
                        },
                    )
                )

                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="RUN_BS_METRICS",
                        job_params={
                            "job_name_parameter": "RUN_CANDIDATE_BS_METRICS",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataId(
                                        input_key="package_task_id",
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            "yt_pool": self.YT_POOL_ABT_METRICS_DIFF,
                        },
                        out={
                            "RUN_ABT_METRICS_REPORT": 200,
                        }
                    )
                )

                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="RUN_BS_METRICS",
                        job_params={
                            "job_name_parameter": "RUN_RELEASED_BS_METRICS",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(),
                        ),
                        ctx={
                            "package_id": "stable",  # our task will find and use prod package_id
                            "yt_pool": self.YT_POOL_ABT_METRICS_DIFF,
                        },
                        out={
                            "RUN_ABT_METRICS_REPORT": 200,
                        }
                    )
                )

                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="REPORT_DIFF_METRICS_ABT",
                        job_params={
                            "job_name_parameter": "REPORT_DIFF_METRICS_ABT",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="RUN_CANDIDATE_BS_METRICS",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="metrics_next",
                                        resource_name="RUN_ABT_METRICS_REPORT",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="RUN_RELEASED_BS_METRICS",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="metrics_prev",
                                        resource_name="RUN_ABT_METRICS_REPORT",
                                    ),
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=jg_utils.get_major_release_number,
                            )
                        ),
                    )
                )

                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                release_stage = rm_const.ReleaseStatus.stable
                # Release z2 package
                release_part.append(jg_release.JobGraphElementReleaseBranched(
                    task_name="RELEASE_SEARCH_COMPONENT_Z_2",
                    release_to=release_stage,
                    job_params={
                        "ctx": {
                            "component_name": "bs_collector",
                            "z2_config_id": "MR_VELES",
                            "minutes_to_wait": 60,
                            "roll_to": "stable",
                            "need_send_st": True,
                            "marks_resources_as_released": True,
                        },
                        "observed_paths": [
                            "quality/ab_testing/bs_collector",
                            "quality/ab_testing/zc",
                        ]
                    },
                    job_arrows=[
                        jg_job_triggers.JobTriggerBuild([
                            # Wait for ya_package to build
                            jg_job_data.ParentDataDict(
                                input_key="component_resources",
                                dict_key="bs_collector_ya_package",
                                resource_name="BS_COLLECTOR_YA_PACKAGE"
                            ),
                            # "package_tasks": YA_PACKAGE.task_id
                            jg_job_data.ParentDataId("package_tasks"),
                        ]),
                        jg_job_triggers.JobTriggerNewTag([
                            # "release_number": release_number
                            jg_job_data.ParentDataCtx("release_number", "scope_number", override=True),
                        ]),
                    ]
                ))
                release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                    release_to=release_stage,
                ))
                return release_part

    class Notify(configs.ReferenceBranchedConfig.Notify):
        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            mailing_list = [
                "{}@yandex-team.ru".format(follower)
                for follower in FOLLOWERS
            ]

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "buryakov"
            queue = "BSCOLLECTORREL"
            dev_queue = "USEREXP"
            summary_template = u"Релиз bs_collector из bs_collector/stable-{}"

            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }

            followers = FOLLOWERS
            add_commiters_as_followers = True

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        observed_paths = [
            "arcadia/quality/ab_testing/bs_collector",
            "arcadia/quality/ab_testing/zc"
        ]
        wiki_page = "serp/experiments/bs-collector/changelog/"

    class Releases(configs.ReferenceBranchedConfig.Releases):
        allow_robots_to_release_stable = True

        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="bs_collector_ya_package",
                    data=ri.SandboxResourceData("BS_COLLECTOR_YA_PACKAGE"),
                    deploy_infos=[ri.SandboxInfo()]
                ),
            ]

        release_followers_permanent = FOLLOWERS
