# -*- coding: utf-8 -*-

from copy import deepcopy

import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.job_graph_element as jg_element
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.releasable_items as ri
from sandbox.projects.release_machine.components import configs as cfg


def merge_dicts(base, specific):
    result = deepcopy(base)
    result.update(specific)
    return result


class BulbasaurCfg(cfg.ReferenceBranchedConfig):
    name = 'bulbasaur'
    display_name = 'Bulbasaur'
    responsible = 'galecore'

    class Testenv(cfg.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = 'ALICE_IOT'

        class JobPatch(cfg.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def ignore_prefix(self):
                return super(self.__class__, self).ignore_prefix + [
                    "_IOTREGRESS_TESTS",
                ]

        class JobGraph(cfg.ReferenceBranchedConfig.Testenv.JobGraph):
            DOCKER_REGISTRY = 'registry.yandex.net'
            DOCKER_IMAGE_REPOSITORY = 'iot'
            DOCKER_USER = 'robot-bulbasaur'
            DOCKER_TOKEN_VAULT_NAME = 'ROBOT_BULBASAUR_DOCKER_REGISTRY_TOKEN'
            DOCKER_PUSH_IMAGE = True
            YT_TOKEN_YAV_SECRET = 'sec-01de9bv57aewxmqg9kn1br8111#yt.token'

            DOCKER_TASK_CTX = {
                # general parameters
                'use_new_format': True,
                'use_aapi_fuse': True,
                # package parameters
                'package_type': 'docker',
                # docker parameters
                'docker_registry': DOCKER_REGISTRY,
                'docker_image_repository': DOCKER_IMAGE_REPOSITORY,
                'docker_user': DOCKER_USER,
                'docker_token_vault_name': DOCKER_TOKEN_VAULT_NAME,
                'docker_push_image': DOCKER_PUSH_IMAGE,
                'ya_yt_token_yav_secret': YT_TOKEN_YAV_SECRET,
            }

            EVO_TESTS_PARAMETERS = {
                "test_filters": "iot",
                "uniproxy_url": "wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-hamster/uni.ws?srcrwr=IOT_USER_INFO:VOICE__IOT_USER_INFO_BETA",  # noqa
                "megamind_url": "http://vins.hamster.alice.yandex.net/speechkit/app/pa/?srcrwr=IoT:iot-beta.quasar.yandex.net:80&srcrwr=SCENARIO_IOT_APPLY:iot-beta.quasar.yandex.net:80&srcrwr=IoTScenarios:iot-beta.quasar.yandex.net:80&srcrwr=SCENARIO_IOTSCENARIOS_APPLY:iot-beta.quasar.yandex.net:80&srcrwr=IoTVoiceDiscovery:iot-beta.quasar.yandex.net:80&srcrwr=SCENARIO_IOTVOICEDISCOVERY_APPLY:iot-beta.quasar.yandex.net:80"  # noqa
            }

            BUILD_TASKS_PARAMETERS = [
                {
                    "ctx": merge_dicts(
                        DOCKER_TASK_CTX,
                        {
                            "description": "YaPackage build of bulbasaur docker image",
                            "packages": "alice/iot/bulbasaur/pkg.json",
                            "resource_type": "BULBASAUR_DOCKER_IMAGE",
                        }
                    ),
                    "resource_name": "bulbasaur_image",
                    "build_item": "server",
                    "resource_ttl": 90,
                },
                {
                    "ctx": merge_dicts(
                        DOCKER_TASK_CTX,
                        {
                            "description": "YaPackage build of xiaomi adapter docker image",
                            "packages": "alice/iot/adapters/xiaomi_adapter/pkg.json",
                            "resource_type": "XIAOMI_ADAPTER_DOCKER_IMAGE",
                        }
                    ),
                    "resource_name": "xiaomi_adapter_image",
                    "build_item": "xiaomi_adapter",
                    "resource_ttl": 90,
                },
                {
                    "ctx": merge_dicts(
                        DOCKER_TASK_CTX,
                        {
                            "description": "YaPackage build of tuya adapter docker image",
                            "packages": "alice/iot/adapters/tuya_adapter/pkg.json",
                            "resource_type": "TUYA_ADAPTER_DOCKER_IMAGE",
                        }
                    ),
                    "resource_name": "tuya_adapter_image",
                    "build_item": "tuya_adapter",
                    "resource_ttl": 90,
                },
                {
                    "ctx": merge_dicts(
                        DOCKER_TASK_CTX,
                        {
                            "description": "YaPackage build of time machine docker image",
                            "packages": "alice/iot/time_machine/pkg.json",
                            "resource_type": "TIME_MACHINE_DOCKER_IMAGE",
                        }
                    ),
                    "resource_name": "time_machine_image",
                    "build_item": "timemachine",
                    "resource_ttl": 90,
                },
                {
                    "ctx": merge_dicts(
                        DOCKER_TASK_CTX,
                        {
                            "description": "YaPackage build of steelix docker image",
                            "packages": "alice/iot/steelix/pkg.json",
                            "resource_type": "STEELIX_DOCKER_IMAGE",
                        }
                    ),
                    "resource_name": "steelix_docker_image",
                    "build_item": "steelix",
                    "resource_ttl": 90,
                }
            ]

            class JobTypes(object):
                # pre_release
                IOTREGRESS_TESTS = "IOTREGRESS_TESTS"

            @property
            def _prerelease(self):
                prerelease_part = [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrek(
                        job_arrows=(
                            jg_arrows.JobTrigger(
                                job_type=self.JobTypes.IOTREGRESS_TESTS
                            )
                        ),
                        # New changes in release process! We are so screwed.
                        # Friday 17:00
                        next_revision_custom_schedule=jg_utils.CustomSchedule(
                            time_interval_list=[(17, 17)],
                            weekday_list=[4],
                            once_a_day=True
                        ),
                        use_arc=self._use_arc,
                    ),
                    jg_element.JobGraphElement(
                        job_params={
                            "job_type": self.JobTypes.IOTREGRESS_TESTS,
                            "task_name": "IOTREGRESS_TESTS",
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "ctx": {
                                "description": "Create regress runs, assessors ticket and links to release ticket",
                                "assessors_queue": "ALICEASSESSORS",
                                "project_id": "iot-ui",
                                "tag": "smarthome",
                            }
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerStartrek([
                                jg_job_data.ParentDataOutput("release_ticket_id", "startrek_issue"),
                            ]),
                        ),
                    ),
                ]
                return prerelease_part

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part = self.add_build_tasks(branch_part)
                branch_part = self.add_acceptance_tasks(branch_part)
                return branch_part

            def add_build_tasks(self, branch_part):
                for build_task_parameters in self.BUILD_TASKS_PARAMETERS:
                    resource_type = build_task_parameters.get("ctx", {}).get("resource_type")
                    resource_ttl = build_task_parameters.get("resource_ttl", 0)
                    build_item = build_task_parameters.get("build_item", "")
                    ctx = build_task_parameters.get("ctx", {})

                    branch_part.append(
                        jg_build.JobGraphElementBuildBranched(
                            task_name="YA_PACKAGE",
                            build_item=build_item,
                            out={resource_type: resource_ttl},
                            ctx=ctx,
                        )
                    )
                return branch_part

            def add_acceptance_tasks(self, branch_part):
                branch_part.append(jg_test.JobGraphElementTestBranchCommon(
                    task_name="ALICE_EVO_CONTEXT_FAST",
                    job_params={
                        "job_name_parameter": "ALICE_EVO_CONTEXT_FAST",
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    },
                    ctx={"evo_in_component_name": "bulbasaur"},
                    job_arrows=(
                        jg_job_triggers.JobTriggerNewTag(
                            parent_job_data=(
                                jg_job_data.ParentDataOutput(
                                    input_key="evo_in_tag_num",
                                    output_key="new_tag_number",
                                ),
                            )
                        ),
                        jg_arrows.ParamsData("evo_in_branch_num", jg_utils.get_major_release_number),
                    )
                ))
                branch_part.append(jg_test.JobGraphElementTestBranchCommon(
                    task_name="ALICE_EVO_INTEGRATION_TESTS_WRAPPER",
                    job_params={
                        "job_name_parameter": "ALICE_EVO_INTEGRATION_TESTS_WRAPPER",
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    },
                    ctx=self.EVO_TESTS_PARAMETERS,
                    job_arrows=(
                        jg_job_triggers.JobTriggerTestBranchCommon(
                            job_name_parameter="ALICE_EVO_CONTEXT_FAST",
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key="launch_type",
                                    output_key="evo_ctx_component_name",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="branch_number",
                                    output_key="evo_ctx_branch_num",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="tag_number",
                                    output_key="evo_ctx_tag_num",
                                ),
                                jg_job_data.ParentDataOutput(
                                    input_key="release_ticket",
                                    output_key="evo_out_release_ticket",
                                ),
                            )
                        ),
                        jg_job_triggers.JobTriggerNewTag([
                            jg_job_data.ParentDataOutput("beta_name", "tag_name"),
                        ]),
                    ),
                ))
                return branch_part

            @property
            def _release(self):
                def get_task_frequency(stage):
                    if stage == rm_const.ReleaseStatus.testing:
                        return jg_utils.TestFrequency.CHECK_EACH_COMMIT, None
                    return jg_utils.TestFrequency.LAZY, None

                custom_release_part = []

                for release_stage in [rm_const.ReleaseStatus.testing, rm_const.ReleaseStatus.stable]:
                    custom_release_part.extend([
                        jg_release.JobGraphElementReleaseBranched(
                            task_name="RELEASE_RM_COMPONENT_2",
                            release_to=release_stage,
                            release_item="ALL",
                            job_params={
                                # there is no separate deploy system for releases docker to nanny, so use sandbox
                                "ctx": {"deploy_system": rm_const.DeploySystem.sandbox.name}
                            },
                            job_arrows=[
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=[
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            build_task_parameters["resource_name"],
                                            build_task_parameters["ctx"]["resource_type"],
                                        )
                                    ],
                                    job_name_parameter=build_task_parameters["build_item"],
                                )
                                for build_task_parameters in self.BUILD_TASKS_PARAMETERS
                            ] + [
                                jg_job_triggers.JobTriggerNewTag(
                                    parent_job_data=[
                                        jg_job_data.ParentDataOutput("major_release_num", "branch_number_for_tag"),
                                        jg_job_data.ParentDataOutput("minor_release_num", "new_tag_number"),
                                    ],
                                ),
                            ],
                        ),
                        jg_release.JobGraphElementActionReleaseBranched(
                            job_params={"frequency": get_task_frequency(release_stage)},
                            release_to=release_stage,
                            release_item="ALL",
                        ),
                    ])
                    for build_task_parameters in self.BUILD_TASKS_PARAMETERS:
                        resource_name = build_task_parameters.get("resource_name", "")
                        resource_type = build_task_parameters.get("ctx", {}).get("resource_type")
                        release_item = build_task_parameters.get("build_item", "")
                        custom_release_part.extend([
                            jg_release.JobGraphElementReleaseBranched(
                                task_name="RELEASE_RM_COMPONENT_2",
                                release_to=release_stage,
                                release_item=release_item,
                                job_params={
                                    # there is no separate deploy system for releases docker to nanny, so use sandbox
                                    "ctx": {"deploy_system": rm_const.DeploySystem.sandbox.name}
                                },
                                job_arrows=[
                                    jg_job_triggers.JobTriggerBuild(
                                        parent_job_data=jg_job_data.ParentDataDict(
                                            "component_resources", resource_name, resource_type
                                        ),
                                        job_name_parameter=release_item,
                                    ),
                                    jg_job_triggers.JobTriggerNewTag(
                                        parent_job_data=[
                                            jg_job_data.ParentDataOutput("major_release_num", "branch_number_for_tag"),
                                            jg_job_data.ParentDataOutput("minor_release_num", "new_tag_number"),
                                        ],
                                    ),
                                ],
                            ),
                            jg_release.JobGraphElementActionReleaseBranched(
                                job_params={"frequency": get_task_frequency(release_stage)},
                                release_to=release_stage,
                                release_item=release_item,
                            ),
                        ])
                return super(self.__class__, self)._release + custom_release_part

    class Notify(cfg.ReferenceBranchedConfig.Notify):

        notifications = []

        class Telegram(cfg.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["iot-dev"]
            config = cfg.RmTelegramNotifyConfig(chats=chats)
            invite_link = u"https://t.me/joinchat/B1OIIxw22vB4j42htm6I6A"

        class Startrek(cfg.ReferenceBranchedConfig.Notify.Startrek):
            assignee = 'lawyard'
            queue = 'ALICERELEASE'
            dev_queue = 'ALICERELEASE'
            summary_template = 'Релиз Умного Дома № {}'
            components = u'Умный дом'

            followers = [
                'mavlyutov',
                'lawyard',
                'galecore',
                'norchine',
                'levkovskayama',
                'oksana-sbr'
            ]
            hide_commits_under_cut = True
            notify_on_release_to_release_st_ticket = frozenset([rm_const.ReleaseStatus.stable])

    class ChangelogCfg(cfg.ReferenceBranchedConfig.ChangelogCfg):
        dirs = [
            'arcadia/alice/iot/adapters',
            'arcadia/alice/iot/bulbasaur',
            'arcadia/alice/iot/steelix',
            'arcadia/alice/iot/time_machine',
            'arcadia/alice/library/go',
        ]
        wiki_page = ''

    class SvnCfg(cfg.ReferenceBranchedConfig.SvnCfg):
        use_arc = True

    class Releases(cfg.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name=build_task_parameters["resource_name"],
                    data=ri.SandboxResourceData(build_task_parameters["ctx"]["resource_type"]),
                    deploy_infos=[ri.SandboxInfo()],
                )
                for build_task_parameters in BulbasaurCfg.Testenv.JobGraph.BUILD_TASKS_PARAMETERS
            ]

        deploy_system = rm_const.DeploySystem.sandbox  # todo: use nanny_docker when it will be implemented
        allow_robots_to_release_stable = True
        wait_for_deploy_time_sec = 60 * 60  # 1h
