# -*- coding: utf-8 -*-

import os
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow
from sandbox.projects.release_machine.components.config_core.jg.graph import base as graph_base
from sandbox.projects.release_machine.components.config_core.jg.preset import basic_build_presets
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import release as release_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib.upper_search.base import IceFlamePatched
from sandbox.projects.release_machine.components.config_core.jg.cube import base as jg_cube
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import build as build_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.configs.clickdaemon import ClickdaemonTestsBundle


EXECUTABLE_PATH = "yweb/webdaemons/clickdaemon"
BUILD_CLICKDAEMON_SANITIZERS = ["", "address", "memory", "undefined", "leak", "thread"]
CLICKDAEMON_OSX_RELEASABLE_NAME = "clickdaemon_osx"


class ClickdaemonCICfg(configs.ReferenceCIConfig):
    name = "clickdaemon_ci"
    display_name = "Clickdaemon CI"
    responsible = configs.Responsible(
        abc=configs.Abc(service_name="clickdaemon"),
        login="elshiko",
    )

    # metrics launch
    component_group = "reference_component_metricsed"

    class JG(basic_build_presets.SingleBuildYaMakeJGCfg):
        # configuring auto-deploy
        release_manually = False

        build_cube_name = "build_linux_bin_and_keys_json"

        build_task = "KOSHER_YA_MAKE"

        @jg_flow.release_flow(
            stages=basic_build_presets.JOINED_BUILD_RELEASE_FLOW_STAGES,
        )
        def release(self):
            graph = super(self.__class__, self).release(self)

            main_stage_entry_cube = graph.get("main_graph_entry")
            release_entry_cube = graph.get("release_stage_entry")

            release_entry_cube.input.update(manual=True)

            # build bin for osx is a separate task which only creates CLICKDAEMON_EXECUTABLE osx resource
            build_osx = build_cubes.YaMakeBuildCubeBase(
                name="build_osx_bin",
                task=self.build_task,
                targets=EXECUTABLE_PATH,
                artifacts=os.path.join(EXECUTABLE_PATH, "clickdaemon"),
                resource_types="CLICKDAEMON_EXECUTABLE",
                input=jg_cube.CubeInput(
                    result_attrs={
                        "major_release_num": rm_const.CIJMESPathCommon.MAJOR_RELEASE_NUM,
                        "minor_release_num": rm_const.CIJMESPathCommon.MINOR_RELEASE_NUM,
                    },
                    result_single_file=True,
                    result_rd="CLICKDAEMON for OSX",
                ),
                needs=[
                    graph.get(dummy_cubes.RMMainGraphEntry.NAME),
                ],
                attributes={
                    "requirements": {
                        "disk": "4GB",
                        "ram": "1GB",
                        "cores": 2,
                        "sandbox": {
                            "platform": "osx",
                            "client_tags": "OSX",
                        },
                    },
                },
            )

            # release cube for clickdaemon osx
            release_osx = release_cubes.ReleaseRmComponent2(
                name="release_stable_build_osx_sandbox",
                component_name=self.component_name,
                where_to_release="stable",
                input=jg_cube.CubeInput(
                    component_resources={
                        CLICKDAEMON_OSX_RELEASABLE_NAME: "${tasks.build_osx_bin.resources | [?type == 'CLICKDAEMON_EXECUTABLE'] | [0].id}"
                    },
                    deploy_system="sandbox",
                    release_item_name=CLICKDAEMON_OSX_RELEASABLE_NAME,
                ),
                manual=False,
            )

            #  tests init
            tests = ClickdaemonTestsBundle.get_bundle(
                sanitizers=BUILD_CLICKDAEMON_SANITIZERS,
                entry_cube=main_stage_entry_cube,
            )

            # add cubes to graph
            for test in tests:
                release_entry_cube.add_requirement(test)
                graph.add(test)

            release_entry_cube.add_requirement(build_osx)
            graph.add(build_osx)

            release_osx.add_requirement(release_entry_cube)
            graph.add(release_osx)

            return graph

        # activate Morty release and adding clickdaemon_osx to release
        def _get_release_cubes_for_deploy_system(self, deploy_system, where, releasable_items, graph, build):
            return [
                release_cubes.ScheduleRelease(
                    name="release_{}_{}".format(where, deploy_system),
                    component_name=self.component_name,
                    where_to_release=where,
                    flows=["morty_stable_release"],
                    warden_component_name="clickdaemon",
                    warden_parent_component_name="common",
                    deploy_system=deploy_system,
                    schedule_mode="create_and_schedule",
                    input=jg_cube.CubeInput(
                        component_resources={
                            ri.name: build.output.resources[ri.data.resource_type].first().id
                            for ri in releasable_items
                        },
                    ),
                    manual=self.release_manually,
                ),
            ]

        @jg_flow.register_flow(title="Run IceFlame")
        def run_iceflame(self):
            return graph_base.Graph([IceFlamePatched(ClickdaemonCICfg.name, "clickdaemon_linux", "hamster-clickdaemon-sas-yp", "clickdaemon.conf")])

    class CI(configs.ReferenceCIConfig.CI):
        a_yaml_dir = EXECUTABLE_PATH
        secret = "sec-01desry8fbgvnkbeybem81ferv"
        sb_owner_group = "SEARCH-RELEASERS"

        ya_make_abs_paths_glob = [
            os.path.join(EXECUTABLE_PATH, "**"),
        ]

    class Releases(configs.ReferenceCIConfig.Releases):
        """Releases configuration"""
        allow_robots_to_release_stable = True
        main_release_flow_branch_auto_start = False
        release_followers_permanent = [
            "elshiko",
            "dima-zakharov",
            "kkabulov"
        ]

        @property
        def releasable_items(self):
            deploy_infos = [
                ri.NannyDeployInfo(
                    [
                        ri.DeployService("production-clickdaemon-{}-yp".format(i), ["prod", "web"])
                        for i in rm_const.MAIN_LOCATIONS
                    ] + [
                        ri.DeployService("hamster-clickdaemon-{}-yp".format(i), ["hamster", "web"])
                        for i in rm_const.MAIN_LOCATIONS
                    ]
                ),
            ]

            return [
                ri.ReleasableItem(
                    name="clickdaemon_linux",
                    data=ri.SandboxResourceData("CLICKDAEMON_EXECUTABLE"),
                    build_data=ri.BuildData(
                        target=EXECUTABLE_PATH,
                        artifact=os.path.join(EXECUTABLE_PATH, "clickdaemon"),
                    ),
                    deploy_infos=deploy_infos
                ),
                ri.ReleasableItem(
                    name="clickdaemon_keys_json",
                    data=ri.SandboxResourceData("CLICKDAEMON_KEYS_JSON"),
                    build_data=ri.BuildData(
                        target="yweb/weblibs/signurl/data",
                        artifact="yweb/weblibs/signurl/data/test_keys.json",
                    ),
                    deploy_infos=[]
                ),
                ri.ReleasableItem(
                    name=CLICKDAEMON_OSX_RELEASABLE_NAME,
                    data=ri.SandboxResourceData("CLICKDAEMON_EXECUTABLE"),
                    deploy_infos=[ri.SandboxInfo()]
                ),
                ri.DynamicReleasableItem(
                    name="scheduled_bundle",
                    data=ri.SandboxResourceData("SCHEDULED_RM_RELEASE_DATA"),
                    deploy_infos=[ri.NannyDeployInfo()],
                )
            ]

    class Notify(configs.ReferenceCIConfig.Notify):
        class Telegram(configs.ReferenceCIConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = ["clickdaemon_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceCIConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            queue = "CLICKDAEMONREL"
            dev_queue = "CLICKDAEMON"
            assignee = configs.Responsible(login="elshiko", abc=configs.Abc(component_id=2088, role_id=None))
            summary_template = u"priemka Clickdaemon {}"
            workflow = {}
            followers = [
                "elshiko",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "logs/clickdaemon/releases/"

        observed_paths = [
            "arcadia/yweb/webdaemons/clickdaemon",
            "arcadia/yweb/weblibs/signurl",
            "arcadia/kernel/signurl",
        ]

        review_groups = [
            "clickdaemon",
        ]

    class SvnCfg(configs.ReferenceCIConfig.SvnCfg):
        start_version = 57
