# -*- coding: utf-8 -*-

import sandbox.common.types.notification as task_notifications

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri

import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease


FOLLOWERS = [
    "ajkon",
    "alexfilatov",
    "buryakov",
    "ilyusha",
    "primorial",
    "samhan",
    "timurguryev",
    "v-kulikov",
    "valgushev",
    "sharamed",
    "conterouz",
]


class CofeCfg(configs.ReferenceBranchedConfig):
    name = "cofe"
    responsible = "valgushev"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_db = "cofe_release_trunk"
        trunk_task_owner = "AB-TESTING"
        branch_db_template = "cofe-release-stable-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(
                        next_revision_custom_schedule=jg_utils.CustomSchedule(
                            time_interval_list=[(16, 16)],
                            weekday_list=[0, 1, 2, 3],
                            once_a_day=True,
                        )
                    ),
                ]

            # Build z2 package
            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="KOSHER_YA_PACKAGE",
                        ctx={
                            "packages": "packages/debian/mapreduce/yandex-search-ab-testing-cofe/yandex-search-ab-testing-cofe.json",
                            "package_type": "debian",
                            "build_type": "release",
                            "strip_binaries": True,
                            "compress_package_archive": True,
                            "key_user": "robot-eksperimentus",
                            "publish_package": True,
                            "publish_to": "search-precise",
                            "resource_type": "ABT_COFE_YA_PACKAGE",
                            "use_aapi_fuse": True,
                            "aapi_fallback": True,
                            "run_tests": True,  # small tests
                            "run_medium_tests": True,
                            "run_long_tests": True,
                            "release_to_nanny": True,
                            # "checkout_arcadia_from_url": "arcadia:/arc/trunk/arcadia@rev" - automatically
                        },
                        job_params={
                            "apiargs": {
                                "notifications": [{
                                    "statuses": ["SUCCESS", "FAILURE", "EXCEPTION", "NO_RES", "TIMEOUT", "STOPPED", "EXPIRED"],
                                    "recipients": FOLLOWERS,
                                    "transport": task_notifications.Transport.EMAIL
                                }],
                                "requirements": {
                                    "disk_space": 250 * (2 ** 30),  # 250 GB
                                    "client_tags": 'SSD',
                                },
                                "kill_timeout": 4 * 3600,   # 4 hours
                            }
                        },
                        out={
                            "ABT_COFE_YA_PACKAGE": 300
                        },
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                release_stage = rm_const.ReleaseStatus.stable
                # Release z2 package
                release_part.append(jg_release.JobGraphElementReleaseBranched(
                    task_name="RELEASE_SEARCH_COMPONENT_Z_2",
                    release_to=release_stage,
                    job_params={
                        "ctx": {
                            "component_name": "cofe",
                            "z2_config_id": "MR_VELES",
                            "minutes_to_wait": 60,
                            "roll_to": "stable",
                            "need_send_st": True,
                            "marks_resources_as_released": True,
                        },
                        "observed_paths": [
                            "quality/ab_testing/cofe"
                        ],
                    },
                    job_arrows=[
                        jg_job_triggers.JobTriggerBuild([
                            # Wait for ya_package to build
                            jg_job_data.ParentDataDict(
                                input_key="component_resources",
                                dict_key="abt_cofe_ya_package",
                                resource_name="ABT_COFE_YA_PACKAGE"
                            ),
                            # "package_tasks": YA_PACKAGE.task_id
                            jg_job_data.ParentDataId("package_tasks"),
                        ]),
                        jg_job_triggers.JobTriggerNewTag([
                            # "release_number": release_number
                            jg_job_data.ParentDataCtx("release_number", "scope_number", override=True),
                        ]),
                    ],
                    wait_for_deploy=True,
                ))
                release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                    release_to=release_stage,
                    job_params={
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    },
                ))
                return release_part

    class Notify(configs.ReferenceBranchedConfig.Notify):

        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            mailing_list = [
                "{}@yandex-team.ru".format(follower)
                for follower in FOLLOWERS
            ]

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "valgushev"
            queue = "COFEREL"
            dev_queue = "USEREXP"
            summary_template = u"Релиз cofe из cofe/stable-{}"

            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }

            followers = FOLLOWERS
            add_commiters_as_followers = True

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        observed_paths = [
            "arcadia/quality/ab_testing/cofe",
        ]
        wiki_page = "serp/experiments/cofe/changelog/"

    class Releases(configs.ReferenceBranchedConfig.Releases):
        allow_robots_to_release_stable = True

        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="abt_cofe_ya_package",
                    data=ri.SandboxResourceData("ABT_COFE_YA_PACKAGE"),
                    deploy_infos=[
                        ri.single_nanny_service("testing_ems", stage=rm_const.ReleaseStatus.testing),
                        ri.NannyDeployInfo(
                            [
                                ri.DeployService("production_ems{suffix}_{dc}".format(suffix=suffix, dc=datacenter))
                                for suffix in ("", "_dataset")
                                for datacenter in ("vla", "man", "sas")
                            ],
                            stage=rm_const.ReleaseStatus.stable
                        )
                    ]
                ),
            ]
        release_followers_permanent = FOLLOWERS
