# -*- coding: utf-8 -*-
#
# Links to release machine docs:
# * https://wiki.yandex-team.ru/releasemachine/
import os

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
from sandbox.projects.release_machine.components.config_core import yappy as yappy_cfg
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow
from sandbox.projects.release_machine.components.config_core.jg.cube import base as jg_cube
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import build as jg_build
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as jg_dummy
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import release as jg_release
from sandbox.projects.release_machine.components.config_core.jg.graph import base as jg_graph
from sandbox.projects.release_machine.components.config_core.jg.preset import basic_build_presets as jg_preset
from sandbox.projects.release_machine.core import releasable_items as ri

COLLECTIONS_BACKEND_ABC_SERVICE_ID = 2124
ROLE_DUTY = 413
CONFIGS = (
    (
        'backend_instancectl_config',
        'COLLECTIONS_BACKEND_INSTANCECTL_CONFIG'
    ),
    (
        'backend_upload_instancectl_config',
        'COLLECTIONS_BACKEND_UPLOAD_INSTANCECTL_CONFIG'
    ),
    (
        'backend_admin_instancectl_config',
        'COLLECTIONS_BACKEND_ADMIN_INSTANCECTL_CONFIG'
    ),
    (
        'worker_instancectl_config',
        'COLLECTIONS_WORKER_INSTANCECTL_CONFIG'
    ),
    (
        'backend_uwsgi_config',
        'COLLECTIONS_BACKEND_UWSGI_CONFIG'
    ),
    (
        'backend_uwsgi_sas_config',
        'COLLECTIONS_BACKEND_UWSGI_SAS_CONFIG'
    ),
    (
        'backend_uwsgi_man_config',
        'COLLECTIONS_BACKEND_UWSGI_MAN_CONFIG'
    ),
    (
        'backend_uwsgi_vla_config',
        'COLLECTIONS_BACKEND_UWSGI_VLA_CONFIG'
    ),
    (
        'backend_uwsgi_testing_config',
        'COLLECTIONS_BACKEND_UWSGI_TESTING_CONFIG'
    ),
    (
        'backend_upload_uwsgi_config',
        'COLLECTIONS_BACKEND_UPLOAD_UWSGI_CONFIG'
    ),
    (
        'backend_admin_uwsgi_config',
        'COLLECTIONS_BACKEND_ADMIN_UWSGI_CONFIG'
    ),
    (
        'worker_uwsgi_config',
        'COLLECTIONS_WORKER_UWSGI_CONFIG'
    ),
    (
        'uwsgi_push_client_production_config',
        'COLLECTIONS_UWSGI_PUSH_CLIENT_PRODUCTION_CONFIG'
    ),
    (
        'uwsgi_push_client_testing_config',
        'COLLECTIONS_UWSGI_PUSH_CLIENT_TESTING_CONFIG'
    ),
    (
        'celery_push_client_production_config',
        'COLLECTIONS_CELERY_PUSH_CLIENT_PRODUCTION_CONFIG'
    ),
    (
        'celery_push_client_testing_config',
        'COLLECTIONS_CELERY_PUSH_CLIENT_TESTING_CONFIG'
    ),
    (
        'push_client_hide_secrets_script',
        'COLLECTIONS_PUSH_CLIENT_HIDE_SECRETS_SCRIPT'
    ),
)


def build_on_commit_flow_decorator(paths):
    return jg_flow.register_flow(
        triggers=[
            jg_flow.ActionTrigger(
                on=jg_flow.ActionTriggerOn.COMMIT,
                into="trunk",
                filters=[
                    {
                        "discovery": "dir",
                        "abs-paths": paths,
                    },
                ],
            ),
        ]
    )


def build_simple_release_item(name, res_type):
    return ri.ReleasableItem(
        name=name,
        data=ri.SandboxResourceData(res_type),
        deploy_infos=[],
    )


class CollectionsBackendCfg(configs.ReferenceCIConfig):
    name = "collections_backend"
    responsible = configs.Responsible(
        abc=configs.Abc(service_name="collections-backend"),
        login="abakharev"
    )

    class MergesCfg(configs.ReferenceCIConfig.MergesCfg):
        number_last_branches_to_merge = 1

    class Releases(configs.ReferenceCIConfig.Releases):
        @property
        def releasable_items(self):
            return (
                [
                    ri.ReleasableItem(
                        name="api_server",
                        data=ri.SandboxResourceData("COLLECTIONS_BACKEND_API_SERVER"),
                        deploy_infos=(
                            [
                                ri.single_nanny_service("collections-backend-production-{}".format(loc))
                                for loc in rm_const.MAIN_LOCATIONS
                            ] + [
                                ri.single_nanny_service("collections-backend-production-upload-{}".format(loc))
                                for loc in rm_const.MAIN_LOCATIONS
                            ] + [
                                ri.single_nanny_service(
                                    'collections-backend-test', stage=rm_const.ReleaseStatus.testing
                                )
                            ]
                        )
                    ),
                    ri.ReleasableItem(
                        name="admin_api_server",
                        data=ri.SandboxResourceData("COLLECTIONS_BACKEND_ADMIN_API_SERVER"),
                        deploy_infos=[
                            ri.single_nanny_service("collections-admin-backend-production"),
                            ri.single_nanny_service('pdb-admin-backend-test', stage=rm_const.ReleaseStatus.testing),
                        ],
                    ),
                    ri.ReleasableItem(
                        name="celery_worker",
                        data=ri.SandboxResourceData("COLLECTIONS_BACKEND_CELERY_WORKER"),
                        deploy_infos=(
                            [
                                ri.single_nanny_service("pdb-worker-production-{}".format(loc))
                                for loc in rm_const.MAIN_LOCATIONS
                            ] + [
                                ri.single_nanny_service('pdb-worker-test', stage=rm_const.ReleaseStatus.testing),
                            ]
                        )
                    ),
                    ri.ReleasableItem(
                        name="collections_ipython",
                        data=ri.SandboxResourceData("COLLECTIONS_BACKEND_IPYTHON"),
                        deploy_infos=[],
                    ),
                    ri.ReleasableItem(
                        name="offer_parser_config.json",
                        data=ri.SandboxResourceData("COLLECTIONS_OFFER_PARSER_CONFIG"),
                        deploy_infos=[],
                    ),
                    build_simple_release_item(
                        'generate_push_notifications',
                        'COLLECTIONS_PUSHES_BINARY'
                    ),
                    build_simple_release_item(
                        'fast_dump',
                        'COLLECTIONS_FAST_DUMP_BINARY'
                    ),
                    build_simple_release_item(
                        'ban_of_surviving_entities',
                        'COLLECTIONS_BAN_OF_SURVIVING_ENTITIES_BINARY'
                    ),
                    build_simple_release_item(
                        'saas_snippets',
                        'COLLECTIONS_SAAS_SNIPPETS_BINARY'
                    ),
                    build_simple_release_item(
                        'celery_queue_monitoring',
                        'COLLECTIONS_BACKEND_CELERY_QUEUE_MONITORING_SERVER'
                    ),
                    build_simple_release_item(
                        'ban_urls',
                        'COLLECTIONS_EXTERNAL_BANS_URLS_FILTER'
                    ),
                    build_simple_release_item(
                        'export_banned_boards',
                        'COLLECTIONS_EXPORT_BANNED_BOARDS_BINARY'
                    ),
                    build_simple_release_item(
                        'monit_server',
                        'COLLECTIONS_MONIT_SERVER_BINARY'
                    ),
                    build_simple_release_item(
                        'merge_processed_tables',
                        'COLLECTIONS_MERGE_PROCESSED_TABLES_BINARY'
                    ),
                    build_simple_release_item(
                        'prepare_ban_tables',
                        'COLLECTIONS_PREPARE_ANTIFRAUD_BAN_TABLES_BINARY'
                    ),
                    build_simple_release_item(
                        'monitor',
                        'COLLECTIONS_COMMON_QUEUE_MONITOR_BINARY'
                    ),
                    build_simple_release_item(
                        'filter_out_saved_assignments',
                        'COLLECTIONS_FILTER_OUT_SAVED_ASSIGNMENTS_BINARY'
                    ),
                    build_simple_release_item(
                        'gathered_boards_delta',
                        'COLLECTIONS_GATHERED_BOARDS_DELTA_BINARY'
                    ),
                    build_simple_release_item(
                        'update_teasers',
                        'COLLECTIONS_UPDATE_OF_TEASERS_BINARY'
                    ),
                    build_simple_release_item(
                        'suggest_maker',
                        'COLLECTIONS_SUGGEST_MAKER_BINARY'
                    ),
                    build_simple_release_item(
                        'images_to_boards',
                        'COLLECTIONS_IMAGES_TO_BOARDS_BINARY'
                    ),
                    build_simple_release_item(
                        'offline',
                        'COLLECTIONS_OFFLINE_STATS_BINARY'
                    ),
                    build_simple_release_item(
                        'save_tops',
                        'COLLECTIONS_SAVE_TOPS_BINARY'
                    ),
                    build_simple_release_item(
                        'make_previews',
                        'COLLECTIONS_MAKE_PREVIEWS_BINARY'
                    ),
                    build_simple_release_item(
                        'rtmr_views_table_generator',
                        'COLLECTIONS_RTMR_VIEWS_BINARY'
                    ),
                    build_simple_release_item(
                        'support_ticket_processor',
                        'COLLECTIONS_SUPPORT_TICKET_PROCESSOR_BINARY'
                    ),
                    build_simple_release_item(
                        'ydb_backup_converter',
                        'COLLECTIONS_YDB_DUMP_CONVERTER'
                    ),
                    build_simple_release_item(
                        'prepare_exports',
                        'COLLECTIONS_PREPARE_ANTIALL_EXPORTS_BINARY'
                    ),
                    build_simple_release_item(
                        'mongo_copy',
                        'COLLECTIONS_MONGO_COPY_BINARY'
                    ),
                    build_simple_release_item(
                        'mongo_backup_converter',
                        'COLLECTIONS_MONGO_BACKUP_CONVERTER'
                    ),
                    build_simple_release_item(
                        'update_goods',
                        'COLLECTIONS_UPDATE_OF_GOODS_BINARY'
                    ),
                ] + [
                    build_simple_release_item(name, res_type)
                    for name, res_type in CONFIGS
                ]
            )

        wait_for_deploy_time_sec = 3600
        allow_robots_to_release_stable = True
        allow_old_releases = True

    class Notify(configs.ReferenceCIConfig.Notify):
        notifications = [
            rm_notifications.Notification(
                event_type="ReleaseFailed",
                chat_name="favorites_releases_chat::telegram",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="favorites_releases_chat::telegram",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_deployed_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="favorites_releases_chat::telegram",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
        ]

        class Mail(configs.ReferenceCIConfig.Notify.Mail):
            mailing_list = [
                "collections-releases@yandex-team.ru",
            ]

        class Startrek(configs.ReferenceCIConfig.Notify.Startrek):
            use_task_author_as_assignee = False
            assignee = configs.Responsible(
                login="abakharev",
                abc=configs.Abc(component_id=COLLECTIONS_BACKEND_ABC_SERVICE_ID, role_id=ROLE_DUTY)
            )
            queue = "PDBBACKREL"
            dev_queue = "PODB"
            summary_template = u"Релиз Бекенда Яндекс.Коллекций {}"
            components = "Backend"
            workflow = rm_const.Workflow.BETA_TEST
            add_commiters_as_followers = False
            deadline = 3
            hide_commits_under_cut = True

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = "collections/backend/Relizy-Bekenda/"
        observed_paths = [
            "arcadia/yweb/yasap/pdb/pdb-backend",
        ]

    class Yappy(yappy_cfg.YappyBaseCfg):
        betas = {
            "collections-backend": yappy_cfg.YappyTemplateCfg(
                template_name="collections-backend",
                working_betas_limit=3,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="collections-backend",
                        resources=[
                            yappy_cfg.YappyParametrizedResource(
                                local_path="api_server",
                                param_name="collections_backend_executable",
                            )
                        ],
                        parent_service="collections-backend-test",
                        ignore_instance_spec=True,
                    )
                ],
                new_yappy=True,
            )
        }
        working_betas_limit = 3

    class SvnCfg(configs.ReferenceCIConfig.SvnCfg):
        start_version = 157

    class CI(configs.ReferenceCIConfig.CI):
        a_yaml_dir = "yweb/yasap/pdb/pdb-backend"
        secret = "sec-01desry8fbgvnkbeybem81ferv"
        sb_owner_group = "YASAP"
        grant_config_update_permissions_to_release_machine_robot = False

    class JG(jg_preset.SingleBuildYaMakeTemplateJGCfg):
        @property
        def build_task(self):
            return 'projects/collections_backend/collections_build_backend'

        @jg_flow.release_flow(stages=jg_preset.DEFAULT_RELEASE_FLOW_STAGES)
        def release(self):
            graph = super(self.__class__, self).release(self)
            release_testing_nanny = graph.get('release_testing_nanny')
            release_testing_nanny.manual = False
            return graph

        @build_on_commit_flow_decorator(["yweb/yasap/pdb/pdb-backend/**"])
        def build_backend(self):
            main_dummy = jg_dummy.RMMainGraphEntry()
            graph = jg_graph.Graph([main_dummy])
            build = self._get_build_cube(graph)
            graph.add(build)
            release_dummy = self._get_release_stage_entry_dummy_cube(graph, build)
            graph.add(release_dummy)
            releases = self._get_release_cubes(graph, build)
            for release in releases:
                if release.name != 'release_testing_nanny':
                    continue
                release.input.update(
                    major_release_num='0',
                    minor_release_num='0',
                )
                release.add_requirement(release_dummy)
                graph.add(release)
            return graph

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/libs/**",
                "yweb/yasap/pdb/pdb-backend/libraries/**",
                "yweb/yasap/pdb/tools/push_notifications/**",
            ]
        )
        def build_tool_generate_push_notifications(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/push_notifications',
                'generate_push_notifications',
                'COLLECTIONS_PUSHES_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/libs/**",
                "yweb/yasap/pdb/pdb-backend/libraries/**",
                "yweb/yasap/pdb/tools/fast_dump/**",
            ]
        )
        def build_tool_fast_dump(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/fast_dump',
                'fast_dump',
                'COLLECTIONS_FAST_DUMP_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/libs/**",
                "yweb/yasap/pdb/pdb-backend/libraries/**",
                "yweb/yasap/pdb/tools/ban_of_surviving_entities/**",
            ]
        )
        def build_tool_ban_of_surviving_entities(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/ban_of_surviving_entities',
                'ban_of_surviving_entities',
                'COLLECTIONS_BAN_OF_SURVIVING_ENTITIES_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/libs/**",
                "yweb/yasap/pdb/pdb-backend/libraries/**",
                "yweb/yasap/pdb/tools/saas_snippets/**",
            ]
        )
        def build_tool_saas_snippets(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/saas_snippets',
                'saas_snippets',
                'COLLECTIONS_SAAS_SNIPPETS_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/celery-queue-monitoring/**",
            ]
        )
        def build_tool_celery_queue_monitoring(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/celery-queue-monitoring',
                'celery_queue_monitoring',
                'COLLECTIONS_BACKEND_CELERY_QUEUE_MONITORING_SERVER'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/nirvana/ban_urls/**",
            ]
        )
        def build_tool_ban_urls(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/nirvana/ban_urls',
                'ban_urls',
                'COLLECTIONS_EXTERNAL_BANS_URLS_FILTER'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/export_banned_boards/**",
                "yweb/yasap/pdb/tools/libs/**",
            ]
        )
        def build_tool_export_banned_boards(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/export_banned_boards',
                'export_banned_boards',
                'COLLECTIONS_EXPORT_BANNED_BOARDS_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/monit/monit_server/**",
            ]
        )
        def build_tool_monit_server(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/monit/monit_server',
                'monit_server',
                'COLLECTIONS_MONIT_SERVER_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/anti_all/merge_processed_tables/**",
                "yweb/yasap/pdb/tools/libs/**",
            ]
        )
        def build_tool_merge_processed_tables(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/anti_all/merge_processed_tables',
                'merge_processed_tables',
                'COLLECTIONS_MERGE_PROCESSED_TABLES_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/pdb-backend/libraries/bans/**",
                "yweb/yasap/pdb/tools/anti_all/prepare_ban_tables/**",
                "yweb/yasap/pdb/tools/libs/**",
            ]
        )
        def build_tool_prepare_ban_tables(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/anti_all/prepare_ban_tables',
                'prepare_ban_tables',
                'COLLECTIONS_PREPARE_ANTIFRAUD_BAN_TABLES_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/common_queue/monitor/**",
                "yweb/yasap/pdb/tools/libs/**",
                "extsearch/images/functionality/monit_robot_process/monit_client/**",
            ]
        )
        def build_tool_monitor(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/common_queue/monitor',
                'monitor',
                'COLLECTIONS_COMMON_QUEUE_MONITOR_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/nirvana/filter_out_saved_assignments/**",
            ]
        )
        def build_tool_filter_out_saved_assignments(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/nirvana/filter_out_saved_assignments',
                'filter_out_saved_assignments',
                'COLLECTIONS_FILTER_OUT_SAVED_ASSIGNMENTS_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/nirvana/gathered_boards_delta/**",
            ]
        )
        def build_tool_gathered_boards_delta(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/nirvana/gathered_boards_delta',
                'gathered_boards_delta',
                'COLLECTIONS_GATHERED_BOARDS_DELTA_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/update_teasers/**",
                "yweb/yasap/pdb/tools/libs/**",
            ]
        )
        def build_tool_update_teasers(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/update_teasers',
                'update_teasers',
                'COLLECTIONS_UPDATE_OF_TEASERS_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/suggest/**",
            ]
        )
        def build_tool_suggest_maker(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/suggest',
                'suggest_maker',
                'COLLECTIONS_SUGGEST_MAKER_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/quality/process/images/to_boards/**",
            ]
        )
        def build_tool_images_to_boards(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/quality/process/images/to_boards/app',
                'images_to_boards',
                'COLLECTIONS_IMAGES_TO_BOARDS_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/nirvana/stats/offline/**",
                "yweb/yasap/pdb/nirvana/stats/resources/**",
            ]
        )
        def build_tool_offline(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/nirvana/stats/offline',
                'offline',
                'COLLECTIONS_OFFLINE_STATS_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/nirvana/stats/save_tops/**",
                "yweb/yasap/pdb/nirvana/stats/resources/**",
            ]
        )
        def build_tool_save_tops(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/nirvana/stats/save_tops',
                'save_tops',
                'COLLECTIONS_SAVE_TOPS_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/nirvana/make_previews/make_previews/**",
            ]
        )
        def build_tool_make_previews(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/nirvana/make_previews/make_previews',
                'make_previews',
                'COLLECTIONS_MAKE_PREVIEWS_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/libs/**",
                "yweb/yasap/pdb/tools/rtmr_views_table_generator/**",
            ]
        )
        def build_tool_rtmr_views_table_generator(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/rtmr_views_table_generator',
                'rtmr_views_table_generator',
                'COLLECTIONS_RTMR_VIEWS_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/pdb-backend/libraries/**",
                "yweb/yasap/pdb/tools/anti_all/support_ticket_processor/**",
            ]
        )
        def build_tool_support_ticket_processor(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/anti_all/support_ticket_processor',
                'support_ticket_processor',
                'COLLECTIONS_SUPPORT_TICKET_PROCESSOR_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/libs/**",
                "yweb/yasap/pdb/tools/ydb_backup_converter/**",
            ]
        )
        def build_tool_ydb_backup_converter(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/ydb_backup_converter',
                'ydb_backup_converter',
                'COLLECTIONS_YDB_DUMP_CONVERTER'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/pdb-backend/libraries/**",
                "yweb/yasap/pdb/tools/libs/**",
                "yweb/yasap/pdb/tools/anti_all/prepare_exports/**",
            ]
        )
        def build_tool_prepare_exports(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/anti_all/prepare_exports',
                'prepare_exports',
                'COLLECTIONS_PREPARE_ANTIALL_EXPORTS_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/mongo_copy/**",
                "yweb/yasap/pdb/pdb-backend/libraries/**",
                "yweb/yasap/pdb/tools/libs/**",
            ]
        )
        def build_tool_mongo_copy(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/mongo_copy',
                'mongo_copy',
                'COLLECTIONS_MONGO_COPY_BINARY'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/pdb-backend/libraries/**",
                "yweb/yasap/pdb/tools/libs/**",
                "yweb/yasap/pdb/tools/mongo_backup_converter/**",
            ]
        )
        def build_tool_mongo_backup_converter(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/mongo_backup_converter',
                'mongo_backup_converter',
                'COLLECTIONS_MONGO_BACKUP_CONVERTER'
            )

        @build_on_commit_flow_decorator(
            [
                "yweb/yasap/pdb/tools/update_goods/**",
                "yweb/yasap/pdb/tools/libs/**",
            ]
        )
        def build_tool_update_goods(self):
            return self._build_tool_flow(
                'yweb/yasap/pdb/tools/update_goods',
                'update_goods',
                'COLLECTIONS_UPDATE_OF_GOODS_BINARY'
            )

        @build_on_commit_flow_decorator(["yweb/yasap/pdb/pdb-backend/configs/**"])
        def build_backend_configs(self):
            main_dummy = jg_dummy.RMMainGraphEntry()
            graph = jg_graph.Graph([main_dummy])
            resources = [res_type for name, res_type in CONFIGS]
            build = jg_build.YaMakeTemplate(
                target_resources=resources,
                name="build",
                task='projects/collections_backend/export_template_resources_and_notify_telegram',
                input=jg_cube.CubeInput(
                    template_resources=resources,
                ),
                needs=[
                    main_dummy,
                ],
            )
            graph.add(build)
            release_dummy = self._get_release_stage_entry_dummy_cube(graph, build)
            graph.add(release_dummy)
            for where in [rm_const.ReleaseStatus.testing, rm_const.ReleaseStatus.stable]:
                release = jg_release.ReleaseRmComponent2Trunk(
                    name="release_{}_{}".format(where, rm_const.DeploySystem.sandbox.name),
                    component_name=self.component_name,
                    task='RELEASE_RM_COMPONENT_2',
                    where_to_release=where,
                    input=jg_cube.CubeInput(
                        component_resources={
                            name: build.output.resources[res_type].first().id
                            for name, res_type in CONFIGS
                        },
                        deploy_system=rm_const.DeploySystem.sandbox.name,
                    ),
                    manual=True,
                )
                release.add_requirement(release_dummy)
                graph.add(release)
            return graph

        def _check_releasable_item_is_suitable(self, ri):
            return hasattr(ri.data, "resource_type") and ri.name in {
                "api_server",
                "admin_api_server",
                "celery_worker",
                "collections_ipython",
                "offer_parser_config.json",
            }

        def _build_tool_flow(self, target, name, resource_type):
            main_dummy = jg_dummy.RMMainGraphEntry()
            graph = jg_graph.Graph([main_dummy])
            build = jg_build.KosherYaMake(
                targets=[target],
                artifacts=[os.path.join(target, name)],
                resource_types=[resource_type],
                needs=[main_dummy],
            )
            graph.add(build)
            release_dummy = self._get_release_stage_entry_dummy_cube(graph, build)
            graph.add(release_dummy)
            for where in [rm_const.ReleaseStatus.testing, rm_const.ReleaseStatus.stable]:
                release = jg_release.ReleaseRmComponent2Trunk(
                    name="release_{}_{}".format(where, rm_const.DeploySystem.sandbox.name),
                    component_name=self.component_name,
                    task='RELEASE_RM_COMPONENT_2',
                    where_to_release=where,
                    input=jg_cube.CubeInput(
                        component_resources={
                            name: build.output.resources[resource_type].first().id
                        },
                        deploy_system=rm_const.DeploySystem.sandbox.name,
                    ),
                    manual=True,
                )
                release.add_requirement(release_dummy)
                graph.add(release)
            return graph
