# -*- coding: utf-8 -*-
import os
import itertools as it
from sandbox.projects.common import constants as sandbox_constants
from sandbox.projects.modadvert.rm.constants import CUPID_CONFIGS_PATH, CUPID_OBJECT_TYPES
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components.job_graph import job_data as jg_job_data
from sandbox.projects.release_machine.components.job_graph import job_triggers as jg_job_triggers
from sandbox.projects.release_machine.components.job_graph import job_arrows as jg_arrows
from sandbox.projects.release_machine.components.job_graph import utils as jg_utils
from sandbox.projects.release_machine.components.job_graph.stages import build_stage as jg_build
from sandbox.projects.release_machine.components.job_graph.stages import test_stage as jg_test
from sandbox.projects.release_machine.components.job_graph.stages import release_stage as jg_release


ya_deploy_release_components = {
    rm_const.ReleaseStatus.testing: "TESTING",
    rm_const.ReleaseStatus.stable: "PRODUCTION"
}


class CupidCfg(configs.ReferenceBranchedConfig):

    name = "cupid"
    responsible = "granilace"

    class MergesCfg(configs.ReferenceBranchedConfig.MergesCfg):
        number_last_branches_to_merge = 1

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_db = "modadvert-cupid-trunk"
        trunk_task_owner = "MODADVERT_RELEASE"
        branch_db_template = "modadvert-cupid-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part

                for object_type in CUPID_OBJECT_TYPES:
                    branch_part.append(
                        jg_build.JobGraphElementYaMakeBuildBranched(
                            task_name="MODADVERT_BUILD_LAAS",
                            build_item=object_type,
                            ctx={
                                "binary_executor_release_type": "stable",
                                "workers_path": os.path.join(CUPID_CONFIGS_PATH, object_type, "workers.yaml"),
                                "peerdirs_path": os.path.join(CUPID_CONFIGS_PATH, object_type, "peerdirs.json"),
                                "build_docker_image": False,  # Docker image is built in the separate job `PUSH_MODADVERT_CUPID_HTTP_ENVIRONMENT_IMAGE`
                                "resource_attributes": {"component": "cupid", "object_type": object_type}
                            },
                            out={
                                "MODADVERT_LAAS_IN_MEMORY_RUNNER_BINARY": 100,
                                "MODADVERT_LAAS_HTTP_RUNNER_BINARY": 100,
                                "MODADVERT_LAAS_MR_RUNNER_BINARY": 100,
                            },
                        )
                    )
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="MODADVERT_PUSH_DOCKER_IMAGE",
                        job_params={
                            "job_name_parameter": "PUSH_MODADVERT_CUPID_HTTP_ENVIRONMENT_IMAGE",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="docker_package_checkout_arcadia_from_url",
                                        output_key=sandbox_constants.ARCADIA_URL_KEY,
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            "docker_package_json": "modadvert/docker/http_environment/package.json",
                            "resource_type_name": "MODADVERT_CUPID_HTTP_ENVIRONMENT_IMAGE",
                            "registry_login": "robot-modadvert-test",
                            "vault_item_owner": "robot-modadvert-test",
                            "vault_item_name": "registry-token",
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="MODADVERT_TEST_CUPID_B2B",
                        job_params={
                            "job_name_parameter": "RUN_CUPID_B2B",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_arrows.ParentsData(
                                input_key="feature_mr_runners",
                                triggers=[
                                    jg_job_triggers.JobTriggerBuild(
                                        job_name_parameter=object_type,
                                        parent_job_data=(
                                            jg_job_data.ParentDataResource(
                                                input_key=object_type,
                                                resource_name="MODADVERT_LAAS_MR_RUNNER_BINARY"
                                            )
                                        )
                                    )
                                    for object_type in CUPID_OBJECT_TYPES
                                ],
                                transform=lambda **resources: resources
                            ),
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=jg_utils.get_major_release_number,
                            )
                        ),
                        ctx={
                            "yt_proxy_url": "hahn",
                            "vault_user": "robot-modadvert-test",
                            "tokens": {"YT_TOKEN": "yt-token", "SANDBOX_TOKEN": "sandbox-token", "YAV_TOKEN": "yav-token"},
                            "max_tables_age": 336,  # 14 days
                            "sampling_rate": 0.03,
                            "runner_memory_limit": 38654705664,  # 36 GB
                            "runner_job_count": 24,
                            "runner_user_slots": 8,
                            "use_last_released_resources": True,
                            "comparison_yt_memory_limit": 19327352832,  # 18 GB
                            "mount_sandbox_in_tmpfs": True,
                            "lock_attempts": 3,
                        },
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release

                build_triggers = []
                for object_type in CUPID_OBJECT_TYPES:
                    build_triggers.append(
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter=object_type,
                            parent_job_data=(
                                jg_job_data.ParentDataDict(
                                    "component_resources", "{}_in_memory_runner".format(object_type),
                                    "MODADVERT_LAAS_IN_MEMORY_RUNNER_BINARY",
                                ),
                                jg_job_data.ParentDataDict(
                                    "component_resources", "{}_http_runner".format(object_type),
                                    "MODADVERT_LAAS_HTTP_RUNNER_BINARY",
                                ),
                                jg_job_data.ParentDataDict(
                                    "component_resources", "{}_mr_runner".format(object_type),
                                    "MODADVERT_LAAS_MR_RUNNER_BINARY",
                                )
                            )
                        )
                    )

                for release_stage, ya_deploy_component in ya_deploy_release_components.items():
                    release_part.extend([
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=release_stage,
                            job_arrows=build_triggers
                        ),
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="MODADVERT_DEPLOY_CUPID",
                            job_params={
                                "job_type": rm_const.JobTypes.DEPLOY,
                                "job_name_parameter": release_stage,
                                "ctx": {
                                    "vault_user": "robot-modadvert-test" if release_stage == rm_const.ReleaseStatus.testing else "robot-modadvert-yt",
                                    "st_token_vault_name": "st-token",
                                    "release_environment": ya_deploy_component,
                                    "deploy_policy": "DEPLOY",
                                }
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerRelease(
                                    job_name_parameter=release_stage
                                ),
                                jg_arrows.ParamsData(
                                    input_key="release_number",
                                    transform=jg_utils.get_major_release_number,
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="PUSH_MODADVERT_CUPID_HTTP_ENVIRONMENT_IMAGE",
                                    parent_job_data=(
                                        jg_job_data.ParentDataCtx(
                                            input_key="release_tag_number",
                                            output_key="release_tag_number",
                                            transform=lambda x, params: str(x),
                                        ),
                                    ),
                                ),
                                jg_arrows.ParentsData(
                                    input_key="http_runners_binaries",
                                    triggers=[
                                        jg_job_triggers.JobTriggerBuild(
                                            job_name_parameter=object_type,
                                            parent_job_data=(
                                                jg_job_data.ParentDataResource(
                                                    input_key=object_type,
                                                    resource_name="MODADVERT_LAAS_HTTP_RUNNER_BINARY"
                                                )
                                            )
                                        )
                                        for object_type in CUPID_OBJECT_TYPES
                                    ],
                                    transform=lambda **resources: resources
                                ),
                            )
                        ),
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=rm_const.JobTypes.DEPLOY,
                                    job_name_parameter=release_stage
                                )
                            )
                        )
                    ])
                return release_part

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            deploy_infos = [ri.YaDeployInfo(
                services=ri.DeployService("modadvert-cupid-prod"),
                stage=rm_const.ReleaseStatus.stable,
            )]
            return list(it.chain.from_iterable(
                [
                    ri.ReleasableItem(
                        name="{}_in_memory_runner".format(i),
                        data=ri.SandboxResourceData(
                            "MODADVERT_LAAS_IN_MEMORY_RUNNER_BINARY",
                            attributes={"object_type": i}
                        ),
                        deploy_infos=deploy_infos
                    ),
                    ri.ReleasableItem(
                        name="{}_http_runner".format(i),
                        data=ri.SandboxResourceData(
                            "MODADVERT_LAAS_HTTP_RUNNER_BINARY",
                            attributes={"object_type": i}
                        ),
                        deploy_infos=deploy_infos
                    ),
                    ri.ReleasableItem(
                        name="{}_mr_runner".format(i),
                        data=ri.SandboxResourceData(
                            "MODADVERT_LAAS_MR_RUNNER_BINARY",
                            attributes={"object_type": i}
                        ),
                        deploy_infos=deploy_infos
                    ),
                ] for i in CUPID_OBJECT_TYPES
            ))
        allow_robots_to_release_stable = True
        release_followers_permanent = [
            "granilace",
        ]
        deploy_system = rm_const.DeploySystem.ya_deploy

        token_vault_owner = 'robot-modadvert-test'
        token_vault_name = 'registry-token'

    class Notify(configs.ReferenceBranchedConfig.Notify):
        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "granilace"
            queue = "CUPIDRELEASE"
            dev_queue = "MODDEV"
            summary_template = u"Приемка автомодератора Купидон (cupid) {}"
            workflow = {}
            followers = [
                "granilace",
                "losev",
                "vsuvorov",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "modadvert/releases/cupid/"
        observed_paths = [
            "arcadia/modadvert/libs/laas",
            "arcadia/modadvert/nirvana/laas",
            "arcadia/modadvert/programs/laas",
            "arcadia/modadvert/resources/automoderators",
        ]
        review_groups = [
            "modadvert",
        ]

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "modadvert/cupid/"
