# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import yappy as yappy_cfg
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
from sandbox.projects.common import constants as sandbox_constants
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers

YAPPY_TEMPLATE = "entitysearch-common"


class EntitySearchBackendCfg(configs.ReferenceBranchedConfig):
    component_group = "reference_component_metricsed"
    name = "entitysearch"
    responsible = "konovodov"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_db = "entitysearch-backend-trunk"
        trunk_task_owner = "ENTITYSEARCH"
        branch_db_template = "entitysearch-backend-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            autoupdating_msg = r"autoupdating ya.make \(no merge\)"

            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(
                        next_revision_custom_schedule=jg_utils.CustomSchedule(
                            time_interval_list=[(4, 23)],
                            weekday_list=[6],
                            last_run_finished=False,
                            once_a_day=True
                        ),
                    ),
                ]

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="ENTITYSEARCH_BINARIES_BUILD",
                        ctx={
                            "search_projects": "build_all",
                            sandbox_constants.CLEAR_BUILD_KEY: True,
                            sandbox_constants.BUILD_SYSTEM_KEY: sandbox_constants.YMAKE_BUILD_SYSTEM,
                        },
                        out={
                            "ENTITY_SEARCH_EXECUTABLE": 90,
                            "ENTITY_SEARCH_CONVERTER": 90,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ENTITY_SEARCH_GET_PROD_DATA_VERSION",
                        job_params={
                            "job_name_parameter": "GET_PROD_DATA_VERSION",
                        },
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="ENTITYSEARCH_DATA_BUILD",
                        job_params={
                            "job_name_parameter": "DATA",
                            "simple_filter": jg_utils.SimpleFilterRegex(
                                commit_message_regex=self.autoupdating_msg,
                            ),
                            "filter_branches": True,
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="checkout_arcadia_from_url",
                                transform=lambda x, rm_config: "arcadia:/arc/branches/entitysearch/stable-{}/arcadia".format(
                                    jg_utils.get_major_release_number(x, rm_config),
                                ),
                                override=True,
                            ),
                        ),
                        ctx={
                            "checkout": True,
                        },
                        out={"ENTITY_SEARCH_DATA": 90},
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="ENTITYSEARCH_CONFIG_BUILD",
                        job_params={
                            "job_name_parameter": "CONFIG",
                            "filter_branches": True,
                        },
                        ctx={
                            "search_projects": "build_all",
                        },
                        out={"ENTITY_SEARCH_CONFIG": 90},
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ENTITY_SEARCH_END_TO_END_TESTING",
                        job_params={
                            "job_name_parameter": "LAUNCH_END_TO_END_TESTING",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="beta",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="beta_url",
                                        output_key="beta_name",
                                        transform=lambda x, params: "https://{}.hamster.yandex.ru".format(x),
                                    ),
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=lambda x, rm_config: jg_utils.get_major_release_number(x, rm_config),
                            ),
                        ),
                        ctx={
                            "beta_source_type": "yappy_beta",
                            "wait_consistency": True,
                            "run_mode_type": "release_machine",
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ACCEPTANCE_ENTITYSEARCH_EXECUTABLE",
                        job_params={
                            "job_name_parameter": "AUTO_ACCEPTANCE",
                            "simple_filter": jg_utils.SimpleFilterRegex(
                                commit_message_regex="^((?!" + self.autoupdating_msg + ").)*$"),
                            "filter_branches": True,
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataId(
                                        input_key="binary_new",
                                    ),
                                )
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="CONFIG",
                                parent_job_data=(
                                    jg_job_data.ParentDataId(
                                        input_key="config_new",
                                    ),
                                )
                            )
                        )
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementYappyBetaGeneratorBranched(
                        beta_conf_type="beta",
                        job_arrows=(
                            jg_arrows.ParentsData(
                                input_key="component_resources",
                                triggers=(
                                    jg_job_triggers.JobTriggerTestBranchCommon(
                                        job_name_parameter="GET_PROD_DATA_VERSION",
                                        parent_job_data=(
                                            jg_job_data.ParentDataOutput(
                                                "main_db_res_id",
                                                "resource_id",
                                            ),
                                        ),
                                    ),
                                    jg_job_triggers.JobTriggerBuild(
                                        parent_job_data=(
                                            jg_job_data.ParentDataResource(
                                                "entitysearch_binary",
                                                "ENTITY_SEARCH_EXECUTABLE",
                                            )
                                        ),
                                    ),
                                    jg_job_triggers.JobTriggerBuild(
                                        job_name_parameter="CONFIG",
                                        parent_job_data=(
                                            jg_job_data.ParentDataResource(
                                                "entity_search_config",
                                                "ENTITY_SEARCH_CONFIG",
                                            ),
                                        ),
                                    )
                                ),
                                transform=lambda main_db_res_id, entitysearch_binary, entity_search_config: {
                                    "entity_search_main_db_res_id": main_db_res_id,
                                    "entitysearch_binary": entitysearch_binary,
                                    "entity_search_config": entity_search_config
                                }
                            ),
                        )
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementYappyBetaGeneratorBranched(
                        beta_conf_type="beta_as_prod",
                        job_params={
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        job_arrows=(
                            jg_arrows.ParentsData(
                                input_key="component_resources",
                                triggers=(
                                    jg_job_triggers.JobTriggerTestBranchCommon(
                                        job_name_parameter="GET_PROD_DATA_VERSION",
                                        parent_job_data=(
                                            jg_job_data.ParentDataOutput(
                                                "main_db_res_id",
                                                "resource_id",
                                            ),
                                        ),
                                    ),
                                ),
                                transform=lambda main_db_res_id: dict(entity_search_main_db_res_id=main_db_res_id)
                            ),
                            jg_arrows.ParamsData(
                                input_key="patch_name",
                                transform=lambda x, rm_config: "{}-as-prod".format(jg_utils.get_major_release_number(x, rm_config)),
                            )
                        ),
                        ctx={"beta_name_source": "STRING"},
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="RUN_CHIMERA_TASK",
                        job_params={
                            "job_name_parameter": "RUN_CHIMERA",
                            # hack for running first tag only (? see "UPDATE_DB_RESOURCES")
                            "simple_filter": jg_utils.SimpleFilterRegex(
                                commit_message_regex="^((?!" + self.autoupdating_msg + ").)*$",
                            ),
                            "filter_branches": True,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="beta",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="beta",
                                        output_key="beta_name",
                                        transform=lambda x, params: "{}.hamster".format(x),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="beta_as_prod",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="beta_as_prod",
                                        output_key="beta_name",
                                        transform=lambda x, params: "{}.hamster".format(x),
                                    ),
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=lambda x, rm_config: jg_utils.get_major_release_number(x, rm_config),
                            ),
                        ),
                        ctx={
                            "beta_source_type": "yappy_beta",
                            "wait_consistency": True,
                            "run_mode_type": "release_machine",
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementLaunchMetrics(
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="beta",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="checked_beta",
                                        output_key="beta_name",
                                        transform=lambda x, params: "{}.hamster".format(x),
                                    ),
                                ),
                            )
                        ),
                        ctx={
                            "custom_template_name": "entitysearch.json",
                            "launch_template_quota": "objectanswer",
                            "run_findurl": False,
                            "sample_beta": "priemka-vm.hamster",
                            "scraper_over_yt_pool": "objectanswer",
                            "enable_autoclicker": True,
                            "autoclicker_metric_name": "diff-2-esearch",
                            "autoclicker_filter": "true",
                        }
                    )
                )
                umbrella_parents = (
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_name_parameter="AUTO_ACCEPTANCE",
                    ),
                    jg_job_triggers.JobTriggerLaunchMetrics(),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_name_parameter="LAUNCH_END_TO_END_TESTING",
                    ),
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedScheduled(
                        job_params={
                            "simple_filter": jg_utils.SimpleFilterRegex(
                                commit_message_regex="^((?!" + self.autoupdating_msg + ").)*$",
                            ),
                            "filter_branches": True,
                            "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 1440),
                        },
                        job_arrows=umbrella_parents,
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedByMarker(
                        job_arrows=umbrella_parents,
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                for release_stage in [rm_const.ReleaseStatus.testing, rm_const.ReleaseStatus.stable]:
                    release_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "entitysearch_binary",
                                            "ENTITY_SEARCH_EXECUTABLE",
                                        )
                                    )
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="CONFIG",
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "entity_search_config",
                                            "ENTITY_SEARCH_CONFIG",
                                        )
                                    )
                                ),
                            )
                        )
                    )
                    release_part.append(
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                        )
                    )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "_BUILD_ENTITYSEARCH_DATA": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_ENTITYSEARCH_CONFIG": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_UPDATE_ENTITYSEARCH_DB_RESOURCES": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_GENERATE_YAPPY_BETA_ENTITYSEARCH_WITH_DATA": rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

            @property
            def ignore_match(self):
                common_ignored = super(self.__class__, self).ignore_match
                return common_ignored + [
                    "ENTITYSEARCH_DATA_BUILD",

                    # builds come from _BUILD_*-tasks
                    "ENTITYSEARCH_BINARIES_BUILD_DEBUG",
                    "ENTITYSEARCH_BINARIES_BUILD_RELEASE",
                    "ENTITYSEARCH_CONFIG_BUILD",
                ]

            @property
            def ignore_prefix(self):
                common_ignored = super(self.__class__, self).ignore_prefix
                return common_ignored + [
                    "ENTITYSEARCH_SHARP_SHOOTER_RELEASE_",  # tests replaced with _ENTITYSEARCH_AUTO_ACCEPTANCE perf checks for binaries from tags
                ]

    class Releases(configs.ReferenceBranchedConfig.Releases):
        allow_old_releases = True

        @property
        def releasable_items(self):
            deploy_infos = [
                ri.NannyDeployInfo(
                    ri.DeployService("{}-production-entitysearch-yp".format(loc)) for loc in rm_const.MAIN_LOCATIONS
                ),
                ri.NannyDeployInfo(
                    [
                        ri.DeployService("{}-hamster-entitysearch-yp".format(loc), tags=["hamster"])
                        for loc in rm_const.MAIN_LOCATIONS
                    ],
                    stage=rm_const.ReleaseStatus.testing,
                ),
            ]
            return [
                ri.ReleasableItem(
                    name="entitysearch_binary",
                    data=ri.SandboxResourceData("ENTITY_SEARCH_EXECUTABLE"),
                    deploy_infos=deploy_infos,
                ),
                ri.ReleasableItem(
                    name="entity_search_config",
                    data=ri.SandboxResourceData("ENTITY_SEARCH_CONFIG"),
                    deploy_infos=deploy_infos,
                ),
            ]

    class Notify(configs.ReferenceBranchedConfig.Notify):

        notifications = [
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="entitysearch_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewTag",
                chat_name="entitysearch_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TestenvDbClone",
                chat_name="entitysearch_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewWikiPage",
                chat_name="entitysearch_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="entitysearch_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="KpiAlert",
                chat_name="entitysearch_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="ReleaseFailed",
                chat_name="entitysearch_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="MergeCommit",
                chat_name="entitysearch_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="entitysearch_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.STARTREK_TICKET_CREATED_CONDITIONS,
                    join_strategy="OR",
                ),
                message_template_file="notifications/ticket_created.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="entitysearch_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_deployed_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="entitysearch_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
        ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = ["entitysearch_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)
            invite_link = u"https://t.me/joinchat/AMbWjBKAB3e_NZhlyBuNTQ"

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "robot-ontodb"
            queue = "ENTITYREL"
            dev_queue = "SEARCH"
            summary_template = u"Приемка бэкенда объектного ответа {}"
            components = "backend"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = []
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "JandeksPoisk/EntitySearch/backend/releases/"
        dirs = [
            "arcadia/dict/yanswer",
            "arcadia/search/wizard/entitysearch",
        ]
        review_groups = [
            "entitysearch",
        ]

    class Yappy(yappy_cfg.YappyBaseCfg):
        PARENT_SERVICE = "sas-production-entitysearch-yp"
        TEMPLATE_NAME = YAPPY_TEMPLATE
        PATCH_DIR = YAPPY_TEMPLATE

        _fixed_yappy_resources = [
            yappy_cfg.YappyLastReleasedResource(
                local_path="main_delta",
                res_type="ENTITY_SEARCH_MAIN_DELTA"
            ),
            yappy_cfg.YappyLastReleasedResource(
                local_path="fresh",
                res_type="ENTITY_SEARCH_FRESH"
            ),
            yappy_cfg.YappyLastReleasedResource(
                local_path="resource.tar.gz",
                res_type="ENTITY_SEARCH_REALTIME"
            )
        ]

        _tunable_bin_cfg_resources = [
            yappy_cfg.YappyParametrizedResource(
                local_path="entitysearch",
                param_name="entitysearch_binary",
                checkconfig_name="entitysearch_bin_md5",
            ),
            yappy_cfg.YappyParametrizedResource(
                local_path="production.cfg",
                param_name="entity_search_config",
                checkconfig_name="entitysearch_cfg_md5",
            )
        ]

        _tunable_data_resource = [
            yappy_cfg.YappyParametrizedResource(
                local_path="entitysearch_data",
                param_name="entity_search_data_res_id",
            ),
            yappy_cfg.YappyParametrizedResource(
                local_path="main_db",
                param_name="entity_search_main_db_res_id",
            ),
        ]

        _tunable_ner_resource = [
            yappy_cfg.YappyParametrizedResource(
                local_path="ner",
                param_name="entity_search_ner_res_id"
            )
        ]

        betas = {
            "beta": yappy_cfg.YappyTemplateCfg(
                template_name=TEMPLATE_NAME,
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir=PATCH_DIR,
                        resources=_tunable_bin_cfg_resources + _tunable_data_resource + _fixed_yappy_resources,
                        parent_service=PARENT_SERVICE,
                    ),
                ],
            ),
            "beta_with_ner": yappy_cfg.YappyTemplateCfg(
                template_name=TEMPLATE_NAME,
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir=PATCH_DIR,
                        resources=_tunable_ner_resource + _tunable_data_resource + _fixed_yappy_resources,
                        parent_service=PARENT_SERVICE,
                    ),
                ],
            ),
            "beta_as_prod": yappy_cfg.YappyTemplateCfg(
                template_name=TEMPLATE_NAME,
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir=PATCH_DIR,
                        resources=_tunable_data_resource + _fixed_yappy_resources,
                        parent_service=PARENT_SERVICE,
                    ),
                ],
            )
        }
