# -*- coding: utf-8 -*-
import os
import datetime as dt

from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


class ExpFormulasBaseCfg(configs.ReferenceTaggedConfig):
    name = "exp_formulas_base"
    responsible = "ilnurkh"

    class Testenv(configs.ReferenceTaggedConfig.Testenv):
        trunk_db = "formulas"
        trunk_task_owner = "WEB_RELEVANCE"

        class JobGraph(configs.ReferenceTaggedConfig.Testenv.JobGraph):
            @property
            def _tag_part(self):
                return super(self.__class__, self)._tag_part + [
                    jg_build.JobGraphElementBuildTagged(
                        task_name="BUILD_WEB_DYNAMIC_MODELS",
                        out={"DYNAMIC_MODELS_ARCHIVE_BASE": 30},
                        ctx={
                            "models_for": "base",
                            "save_unpacked_archive": True,
                        },
                        job_params={
                            "filter_branches": True,
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="exp_models_url",
                                        output_key="result_path",
                                    )
                                )
                            ),
                        ),
                    ),
                    jg_test.JobGraphElementTestTagCommon(
                        task_name="COMPARE_WEB_MIDDLESEARCH_RESPONSES_SINGLE_HOST",
                        ctx={
                            "compare_complete_only": True,
                            "compare_fully_answered_only": True,
                            "sending_timeout": "5000ms",
                            "max_snippets_per_request": 500,
                            "n_groups_for_source_mult": 5,
                            "ignore_diff_in_compressed_all_factors": True,
                            "ignore_diff_in_doc_ranking_factors": True,
                            "relev_factors_for_soft_check": "450,451,1861,1862,1920",
                        },
                        job_params={
                            "filter_branches": True,
                            "job_name_parameter": "ACCEPTANCE_NO_DIFF",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="models_archive2",
                                        resource_name="DYNAMIC_MODELS_ARCHIVE_BASE",
                                    )
                                )
                            ),
                        ),
                    ),
                    jg_test.JobGraphElementTestTagCommon(
                        task_name="FAST_BASESEARCH_ACCEPTANCE",
                        ctx={
                            "get_all_factors": True,
                            "snippet_experiments_changed": True,
                            "fail_on_low_amount_of_queries": False,
                            "ignore_got_error": False,
                            "polite_mode": True,
                            "dynamic_models_archive2_source": "existing",
                            "web_formulas_acceptance": dt.datetime.now().strftime("%Y-%m-%d_%H:%M"),
                            "fail_on_diff": True,
                        },
                        job_params={
                            "filter_branches": True,
                            "job_name_parameter": "ACCEPTANCE",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="dynamic_models2_res_id",
                                        resource_name="DYNAMIC_MODELS_ARCHIVE_BASE",
                                    )
                                )
                            ),
                        ),
                    ),
                ]

            @property
            def _release(self):
                min_release_frequency_hours = 12
                return super(self.__class__, self)._release + [
                    jg_release.JobGraphElementNewTagTagged(),
                    jg_release.JobGraphElementReleaseTagged(
                        release_to=rm_const.ReleaseStatus.stable,
                        job_params={
                            "observed_paths": ["branches/base/dynamic_ranking_models"],
                            "filter_branches": True,
                            "frequency": (jg_utils.TestFrequency.DEFINED_BY_CODE, None),
                            "next_revision_custom_schedule": jg_utils.CustomSchedule(
                                time_interval_list=[(3, 6)],  # tick at evening, wishing have release at early morning
                                seconds_from_last_run_commit=min_release_frequency_hours * 3600,
                                last_run_finished=True,
                                seconds_from_last_run_commit_if_run_not_finished=1.5 * min_release_frequency_hours * 3600
                            )
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "base_exp_models_res_id",
                                        "DYNAMIC_MODELS_ARCHIVE_BASE"
                                    )
                                )
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(job_name_parameter="ACCEPTANCE_NO_DIFF"),
                            jg_job_triggers.JobTriggerTestBranchCommon(job_name_parameter="ACCEPTANCE"),
                        )
                    ),
                ]

    class Releases(configs.ReferenceTaggedConfig.Releases):
        resources_info = [
            configs.ReleasedResourceInfo(
                name="basesearch_models",
                resource_type="DYNAMIC_MODELS_ARCHIVE_BASE",
                resource_name="base_exp_models_res_id",
                build_ctx_key="used_exp_models_url",
                deploy=[(rm_const.ReleaseStatus.stable, "vla_web_tier1_base")],
            ),
        ]

        allow_robots_to_release_stable = True

    class SvnCfg(configs.ReferenceTaggedConfig.SvnCfg):
        REPO_NAME = "robots"
        tag_prefix = "stable"
        tag_folder = "tags/dynamic_ranking_models"
        tag_name = "experiments/base"
        tag_folder_template = r"{tag_prefix}-{tag_num}"

        @property
        def main_url(self):
            return os.path.join(self.repo_base_url, "branches/base/dynamic_ranking_models/experiment")

    class Notify(configs.ReferenceTaggedConfig.Notify):

        notifications = [
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="formula_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="base_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewTag",
                chat_name="formula_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewTag",
                chat_name="base_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="formula_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.STARTREK_TICKET_CREATED_CONDITIONS,
                    join_strategy="OR",
                ),
                message_template_file="notifications/ticket_created.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="formula_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
        ]

        use_startrek = False

        class Telegram(configs.ReferenceTaggedConfig.Notify.Telegram):
            chats = ["formula_comp", "base_comp"]
