# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
import sandbox.projects.release_machine.components.job_graph.presets as jg_presets
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers


class FeedbackCfg(configs.ReferenceBranchedConfig):
    name = "feedback_platform"
    # https://abc-back.yandex-team.ru/api/v4/services/members/?service__slug=MED&role__in=16
    responsible = configs.Responsible(abc=configs.Abc(role_id=16, component_id=3790), login="denvr")

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_task_owner = "CHATS_FEEDBACK"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            _preset = jg_presets.SinglePackagePreset(
                build_ctx={
                    "package_type": "docker",
                    "docker_push_image": True,
                    "docker_registry": "registry.yandex.net",
                    "docker_image_repository": "webmaster",
                    "docker_user": "robot-chats-releases",
                    "docker_token_vault_name": "robot-chats-releases-doker-oauth-token",
                    "docker_build_network": "host",
                    "checkout": False,
                    "ya_yt_store": True,
                    "ignore_recurses": False,
                },
                package_names="quality/functionality/chats/feedback/package.json",
                build_apiargs={"kill_timeout": 10800},
                deploy_system=rm_const.DeploySystem.nanny.name,
                stages=[
                    jg_utils.StageReleaseFrequency(rm_const.ReleaseStatus.stable),
                    jg_utils.StageReleaseFrequency(rm_const.ReleaseStatus.testing),
                ],
            )

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="CHATS_RUN_LOADTEST",
                        job_params={
                            "job_name_parameter": "LOADTEST",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="docker_image_to_deploy",
                                        resource_name="FEEDBACK_PACKAGE",
                                    )
                                )
                            ),
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=lambda x, rm_config: jg_utils.get_major_release_number(x, rm_config),
                            ),
                        )
                    )
                )
                return branch_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):

            @property
            def change_frequency(self):
                uppercase_name = self.name.upper()
                return {
                    "_BUILD_{}_PACKAGE".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_LOG_MERGE__{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_{}_UMBRELLA_ACCEPT_BY_MARKER".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_{}_UMBRELLA_ACCEPT_SCHEDULED".format(uppercase_name): rm_const.TestFrequencies.ONCE_A_DAY_TEST,
                    "_BUILD_RELEASE_{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_RELEASE_{}SEARCH".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_NEW_TAG__{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                }

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="feedback_package_res_id",
                    data=ri.SandboxResourceData("FEEDBACK_PACKAGE", ttl=30),
                    deploy_infos=[ri.NannyDeployInfo(ri.DeployService("production_chats_api"))],
                )
            ]

        release_followers_permanent = [
            "denvr",
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):

        notifications = [
            rm_notifications.Notification(
                event_type="MergeCommit",
                chat_name="medchats_comp",
                conditions=rm_notifications.NotificationCondition(conditions=[], join_strategy="OR"),
            ),
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="medchats_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewTag",
                chat_name="medchats_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TestenvDbClone",
                chat_name="medchats_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewWikiPage",
                chat_name="medchats_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="medchats_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="BuildTest",
                chat_name="medchats_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="KpiAlert",
                chat_name="medchats_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="ReleaseFailed",
                chat_name="medchats_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="medchats_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.STARTREK_TICKET_CREATED_CONDITIONS,
                    join_strategy="OR",
                ),
                message_template_file="notifications/ticket_created.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="medchats_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
        ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = ["medchats_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "denvr"
            queue = "FEEDBACKREL"
            dev_queue = "FEEDBACK"
            summary_template = u"FEEDBACK: Релиз {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "denvr",
                "woolly",
                "a-sadovnikov",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "chats/feedback/releases/"
        observed_paths = [
            "arcadia/quality/functionality/chats/feedback",
        ]
