# -*- coding: utf-8 -*-
"""
Release machine for Fury Megaera graph
"""
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_triggers
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.xurma.rm as xurma_rm

RESOURCES = {
    "FURY_MEGAERA_CONFIGS": "configs/megaera.json",
    "FURY_MEGAERA_CONTROL_NODE": "control-nodes/megaera.json",
    "FURY_MEGAERA_DATA_NODE": "data-nodes/megaera.json",
}


class FuryMegaeraCfg(configs.ReferenceBranchedConfig):
    name = "fury_megaera"
    responsible = xurma_rm.RESPONSIBLE

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        dirs = [
            "arcadia/quality/antifraud/xurma",
        ]
        wiki_page = "JandeksPoisk/Antirobots/Xurma/releases/releases-megaera/"
        wiki_page_owner = xurma_rm.WIKI_PAGE_OWNER

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name=resource.lower(),
                    data=ri.SandboxResourceData(resource),
                    deploy_infos=[],
                ) for resource in RESOURCES
            ]

        allow_robots_to_release_stable = True

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = xurma_rm.TRUNK_TASK_OWNER
        testenv_db_owners = xurma_rm.TESTENV_DB_OWNERS

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _prerelease(self):
                """
                    Prerelease part of job graph for Testenv with arrows and dependencies
                    :return: list with jobs
                """
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    xurma_rm.JobGraphElementChangelogProcessor(ctx={
                        "graph_name": "megaera",
                    }),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrekBase(job_arrows=(
                        jg_arrows.JobTrigger(
                            "CHANGELOG_PROCESSOR",
                            jg_data.ParentDataResource("changelog", "RELEASE_MACHINE_CHANGELOG"),
                        ),
                    )),
                    jg_prerelease.JobGraphElementWikiBase(job_arrows=(
                        jg_arrows.JobTrigger(
                            "CHANGELOG_PROCESSOR",
                            jg_data.ParentDataResource("changelog", "RELEASE_MACHINE_CHANGELOG"),
                        ),
                        jg_triggers.JobTriggerStartrek(),
                    )),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(),
                ]

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="FURY_BUILD_MEGAERA",
                        out={resource: 30 for resource in RESOURCES},
                        ctx={
                            "resource_type": ";".join(RESOURCES.keys()),
                            "packages": ";".join(
                                "quality/antifraud/xurma/deploy/fury/" + path
                                for path in RESOURCES.values()
                            ),
                        },
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                for release_to in (
                    rm_const.ReleaseStatus.testing,
                    rm_const.ReleaseStatus.prestable,
                    rm_const.ReleaseStatus.stable,
                ):
                    release_part.append(jg_release.JobGraphElementReleaseBranched(
                        release_to=release_to,
                        job_arrows=[
                            jg_triggers.JobTriggerBuild(
                                parent_job_data=[
                                    jg_data.ParentDataDict(
                                        input_key="component_resources",
                                        dict_key=resource.lower(),
                                        resource_name=resource,
                                    ) for resource in RESOURCES
                                ],
                            ),
                        ],
                    ))
                # stable and testing
                for release_to in (rm_const.ReleaseStatus.testing, rm_const.ReleaseStatus.stable):
                    release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                        release_to=release_to,
                    ))
                # prestable
                release_part.append(xurma_rm.JobGraphElementAgreementMaker(
                    job_name_parameter="prestable",
                    ctx={
                        "text": u"Согласование на приемку релиза",
                        "approvers": ["olga-buyanova"]
                    }
                ))
                release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                    release_to=rm_const.ReleaseStatus.prestable,
                    job_arrows=jg_arrows.JobTrigger("AGREEMENT_MAKER"),
                ))
                return release_part

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "xurma/graphs/megaera"
        tag_name = "xurma/graphs/megaera"

    class Notify(configs.ReferenceBranchedConfig.Notify):

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            queue = xurma_rm.STARTREK_QUEUE
            assignee = xurma_rm.STARTREK_ASSIGNEE
            ticket_type = "Release"
            summary_template = u"Fury Megaera graph {}"
            add_commiters_as_followers = False
            followers = [assignee, "olga-buyanova"]
            workflow = xurma_rm.STARTREK_WORKFLOW
