# -*- coding: utf-8 -*-

from sandbox.projects.common import constants as sandbox_constants
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components.configs.geo import _common as geo_utils
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers


class GeoBasesearchCfg(configs.ReferenceBranchedConfig):
    component_group = "geo_common"
    name = "addrs_base"
    responsible = "evelkin"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_db = "geobasesearch-trunk"
        trunk_task_owner = "GEOMETA-SEARCH"
        branch_db_template = "geobasesearch-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_SEARCH",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 90 * (1024 ** 3),
                                },
                                "kill_timeout": 18000,
                            }
                        },
                        ctx={
                            "build_geobasesearch": True,
                            "build_geobusinessindexer": True,
                            "build_geourlsextractor": True,
                            "build_geocollectionsindexer": True,
                            "build_geosearch_standalone_indexer": True,
                            "build_geosearch_index_downloader": True,
                            "build_geosearch_yt_url_parser": True,
                            "build_geosearch_yt_indexer_configs": True,
                            "build_geosearch_yt_data_preparer": True,
                            sandbox_constants.BUILD_SYSTEM_KEY: sandbox_constants.YMAKE_BUILD_SYSTEM,
                            sandbox_constants.BUILD_TYPE_KEY: sandbox_constants.RELEASE_BUILD_TYPE,
                        },
                        out={"GEOBASESEARCH_EXECUTABLE": 40},
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ADDRS_COLLECT_RMDATA",
                        job_params={
                            "job_name_parameter": "COLLECT_AUX_RESOURCES",
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="branch",
                                transform=jg_utils.get_major_release_number,
                            ),
                        )
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="GET_ADDRS_REQUESTS",
                        job_params={
                            "job_name_parameter": "GET_REQUESTS",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(),
                        ),
                        ctx={
                            "location": "SAS",
                            "number_of_hosts": 1,
                            "service": "Business",
                            "time_interval": 300,
                            "number_of_requests": 100000,
                            "sample_origins_uniformly": False,
                        },
                        out={"PLAIN_TEXT_QUERIES": 40},
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="GENERATE_PLAN_FROM_QUERIES",
                        job_params={
                            "job_name_parameter": "SHOOTING_PLAN",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="GET_REQUESTS",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="queries_resource_id",
                                        resource_name="PLAIN_TEXT_QUERIES",
                                    ),
                                )
                            )
                        ),
                        ctx={
                            "minimal_plan_size": 1000000,
                            "resource_attrs": "type=addrs_business",
                        },
                        out={"BASESEARCH_PLAN": 40},
                    )
                )
                for beta_type, transform_params, beta_transform_res, test_transform_res in [
                    (
                        "ref",
                        lambda x, rm_config: "{rev}-ref".format(rev=x.revision),
                        lambda build_res, aux_res: [aux_res],
                        lambda build_res: [],
                    ),
                    (
                        "test",
                        lambda x, rm_config: "{rev}-test".format(rev=x.revision),
                        lambda build_res, aux_res: [aux_res, build_res],
                        lambda build_res: [build_res],
                    ),
                ]:
                    branch_part.append(
                        jg_test.JobGraphElementYappyBetaGeneratorBranched(
                            beta_conf_type=beta_type,
                            job_params={
                                "task_name": "CREATE_ADDRS_BETA",
                                "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            },
                            job_arrows=(
                                jg_arrows.ParamsData(
                                    input_key="branch",
                                    transform=jg_utils.get_major_release_number,
                                ),
                                jg_arrows.ParamsData(
                                    input_key="tag",
                                    transform=transform_params,
                                ),
                                jg_arrows.ParentsData(
                                    input_key="resources",
                                    triggers=(
                                        jg_job_triggers.JobTriggerBuild(
                                            parent_job_data=(
                                                jg_job_data.ParentDataResource(
                                                    input_key="build_res",
                                                    resource_name="GEOBASESEARCH_EXECUTABLE",
                                                )
                                            )
                                        ),
                                        jg_job_triggers.JobTriggerTestBranchCommon(
                                            job_name_parameter="COLLECT_AUX_RESOURCES",
                                            parent_job_data=(
                                                jg_job_data.ParentDataCtx(
                                                    input_key="aux_res",
                                                    output_key="shardmap",
                                                )
                                            )
                                        ),
                                    ),
                                    transform=beta_transform_res,
                                ),
                            ),
                            ctx={
                                "beta_type_choices": "base",
                                "wait_time": 90,
                                "launch_type_choices": "RM",
                            },
                        )
                    )
                    branch_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="CREATE_ADDRS_BETA",
                            job_params={
                                "job_name_parameter": "CREATE_COLLECTIONS_BETA_{}".format(beta_type.upper()),
                            },
                            job_arrows=(
                                jg_arrows.ParamsData(
                                    input_key="branch",
                                    transform=jg_utils.get_major_release_number,
                                ),
                                jg_arrows.ParamsData(
                                    input_key="tag",
                                    transform=transform_params,
                                ),
                                jg_arrows.ParentsData(
                                    input_key="resources",
                                    triggers=(
                                        jg_job_triggers.JobTriggerBuild(
                                            parent_job_data=(
                                                jg_job_data.ParentDataResource(
                                                    input_key="build_res",
                                                    resource_name="GEOBASESEARCH_EXECUTABLE",
                                                )
                                            )
                                        ),
                                    ),
                                    transform=test_transform_res,
                                ),
                            ),
                            ctx={
                                "beta_type_choices": "collections",
                                "wait_time": 90,
                                "launch_type_choices": "RM",
                            },
                        )
                    )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ADDRS_BASE_PERFORMANCE_PARALLEL_ACCEPTANCE",
                        job_params={
                            "job_name_parameter": "LAUNCH_PERFORMANCE_TEST",
                        },
                        job_arrows=(
                            jg_arrows.ParamsData("release_number", jg_utils.get_major_release_number),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="binary2",
                                        resource_name="GEOBASESEARCH_EXECUTABLE",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="SHOOTING_PLAN",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="dolbilo_plan_resource_id",
                                        resource_name="BASESEARCH_PLAN",
                                    )
                                )
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="COLLECT_AUX_RESOURCES",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="fast_features_sharded_resource_id",
                                        output_key="fast_features_sharded_resource_id",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="geobasesearch_config_resource_id",
                                        output_key="geobasesearch_config_resource_id",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="maps_search_advert_db_resource_id",
                                        output_key="maps_search_advert_db_resource_id",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="binary1",
                                        output_key="binary1",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="shardmap1",
                                        output_key="shardmap",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="shardmap2",
                                        output_key="shardmap",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="formulas_dir_resource_id",
                                        output_key="formulas_dir_resource_id",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="dynamic_formula_resource_id",
                                        output_key="dynamic_formula_resource_id",
                                    ),
                                )
                            )
                        ),
                        ctx={
                            "dolbilka_executor_max_simultaneous_requests": 50,
                            "dolbilka_executor_mode": "finger",
                            "dolbilka_executor_requests_limit": 500000,
                            "dolbilka_executor_sessions": 5,
                            "dolbilka_request_timeout": 600,
                            "offline_warmup_time": 1,
                            "offline_shutdown_time": 1,
                            "launch_type": "RM",
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementLaunchMetrics(
                        job_params={
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="test",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="checked_extra_params",
                                        output_key="experimental_sources",
                                        transform=lambda x, params: geo_utils.make_beta_cgi(x, "stable"),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="ref",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="sample_extra_params",
                                        output_key="experimental_sources",
                                        transform=lambda x, params: geo_utils.make_beta_cgi(x, "stable"),
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            "metrics_mode_type": "release_machine",
                            "custom_template_name": "common.json",
                            "template_source": "arcadia",  # METRICSSUPPORT-4702
                            "launch_template_quota": "geosearch",
                            "search_subtype": "geo",
                            "sample_beta": "reference",
                            "checked_beta": "test",
                            "enable_autoclicker": True,
                            "autoclicker_metric_name": "diff-2-serps-query-10",
                            "autoclicker_retry_count": 3,
                            "scraper_over_yt_pool": "geo",
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ADDRS_MAPKIT_COMPARE_TEST",
                        job_params={
                            "job_name_parameter": "LAUNCH_MAPKIT_ACCEPTANCE_YAPPY",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        ctx={
                            "behave_framevork_url": "arcadia:/arc/trunk/arcadia/search/geo/tools/geosearch_acceptance",
                            "launch_type_choices": "RM",
                        },
                        job_arrows=(
                            jg_arrows.ParamsData("release_number", jg_utils.get_major_release_number),
                            jg_arrows.ParentsData(
                                input_key="reference_stand_url",
                                triggers=(
                                    jg_job_triggers.JobTriggerGenerateBeta(
                                        job_name_parameter="ref",
                                        parent_job_data=(
                                            jg_job_data.ParentDataCtx(
                                                input_key="ref_base_sources",
                                                output_key="experimental_sources",
                                            ),
                                        ),
                                    ),
                                    jg_job_triggers.JobTriggerTestBranchCommon(
                                        job_name_parameter="CREATE_COLLECTIONS_BETA_REF",
                                        parent_job_data=(
                                            jg_job_data.ParentDataCtx(
                                                input_key="ref_collections_sources",
                                                output_key="experimental_sources",
                                            )
                                        ),
                                    ),
                                ),
                                transform=lambda ref_base_sources, ref_collections_sources: geo_utils.make_beta_url(
                                    [ref_base_sources, ref_collections_sources],
                                ),
                            ),
                            jg_arrows.ParentsData(
                                input_key="test_stand_url",
                                triggers=(
                                    jg_job_triggers.JobTriggerGenerateBeta(
                                        job_name_parameter="test",
                                        parent_job_data=(
                                            jg_job_data.ParentDataCtx(
                                                input_key="test_base_sources",
                                                output_key="experimental_sources",
                                            ),
                                        ),
                                    ),
                                    jg_job_triggers.JobTriggerTestBranchCommon(
                                        job_name_parameter="CREATE_COLLECTIONS_BETA_TEST",
                                        parent_job_data=(
                                            jg_job_data.ParentDataCtx(
                                                input_key="test_collections_sources",
                                                output_key="experimental_sources",
                                            )
                                        ),
                                    ),
                                ),
                                transform=lambda test_base_sources, test_collections_sources: geo_utils.make_beta_url(
                                    [test_base_sources, test_collections_sources],
                                ),
                            ),

                        ),
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ALICE_EVO_INTEGRATION_TESTS_WRAPPER",
                        job_params={
                            "job_name_parameter": "LAUNCH_ALICE_INTEGRATION_TESTS",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        ctx={
                            "uniproxy_url": "wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-hamster/uni.ws",
                            "enable_stats": True,
                            "test_filters": "poi",
                            "launch_type": "geo",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="COLLECT_AUX_RESOURCES",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="startek_tiket",
                                        output_key="release_ticket",
                                    ),
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="branch_number",
                                transform=jg_utils.get_major_release_number,
                            ),
                            jg_arrows.ParentsData(
                                input_key='tag_number',
                                triggers=(
                                    jg_job_triggers.JobTriggerNewTag(
                                        parent_job_data=(
                                            jg_job_data.ParentDataOutput(
                                                input_key='tag_name',
                                                output_key='tag_name',
                                            ),
                                        ),
                                    ),
                                ),
                                transform=lambda tag_name: "{}".format(tag_name),
                            ),
                            jg_arrows.ParentsData(
                                input_key="megamind_url",
                                triggers=(
                                    jg_job_triggers.JobTriggerGenerateBeta(
                                        job_name_parameter="test",
                                        parent_job_data=(
                                            jg_job_data.ParentDataCtx(
                                                input_key="test_base_sources",
                                                output_key="experimental_sources",
                                            ),
                                        ),
                                    ),
                                    jg_job_triggers.JobTriggerTestBranchCommon(
                                        job_name_parameter="CREATE_COLLECTIONS_BETA_TEST",
                                        parent_job_data=(
                                            jg_job_data.ParentDataCtx(
                                                input_key="test_collections_sources",
                                                output_key="experimental_sources",
                                            )
                                        ),
                                    ),
                                ),
                                transform=lambda test_base_sources, test_collections_sources: geo_utils.make_megamind_url(
                                    [test_base_sources, test_collections_sources],
                                ),
                            ),
                        ),
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementE2ETest(
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="test",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="hermionee2e_base_url",
                                        output_key="hamster_url",
                                    ),
                                ),
                            )
                        ),
                        ctx={
                            "task_retries_delay": ["2", "4"],
                            "task_retries": 2,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedByMarker(
                        job_arrows=(
                            jg_job_triggers.JobTriggerLaunchMetrics(),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="LAUNCH_PERFORMANCE_TEST",
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="LAUNCH_MAPKIT_ACCEPTANCE_YAPPY",
                            ),
                            jg_job_triggers.JobTriggerE2ETest(),
                        )
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                release_part.append(
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "geobasesearch_binary",
                                        "GEOBASESEARCH_EXECUTABLE",
                                    ),
                                ),
                            )
                        )
                    )
                )
                release_part.append(
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                    ),
                )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def change_frequency(self):
                uppercase_name = self.name.upper()
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "_BUILD_{}_SHOOTING_PLAN".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_PROFILE_GEOBASESEARCH": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_RELEASE_GEOBASESEARCH": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_CREATE_GEOBASESEARCH_BETA_ref": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_CREATE_GEOBASESEARCH_BETA_test": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_GET_{}_REQUESTS".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_LAUNCH_MAPKIT_ACCEPTANCE_{}_YAPPY".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_LAUNCH_METRICS_{}_YAPPY".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_LAUNCH_PERFORMANCE_TEST_{}_YAPPY".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

    class Releases(configs.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="geobasesearch_binary",
                    data=ri.SandboxResourceData("GEOBASESEARCH_EXECUTABLE"),
                    deploy_infos=[ri.single_nanny_service("addrs_base")],
                )
            ]

        release_followers_permanent = [
            "evelkin",
            "karas-pv",
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""

        notifications = [
            rm_notifications.Notification(
                event_type="MergeCommit",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewTag",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TestenvDbClone",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewWikiPage",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="AcceptanceTest",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="BuildTest",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewBetaGeneration",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="ReleaseFailed",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.STARTREK_TICKET_CREATED_CONDITIONS,
                    join_strategy="OR",
                ),
                message_template_file="notifications/ticket_created.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_deployed_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="geobasesearch_comp_q",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
        ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = ["geobasesearch_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "robot-geosearch"
            queue = "GEOACCEPT"
            dev_queue = "SEARCH"
            summary_template = u"Приёмка бинарникa базового геопоиска {}"
            workflow = {}
            followers = [
                "evelkin",
                "karas-pv",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7
            notify_on_deploy_to_feature_st_tickets = True

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "JandeksPoisk/GeoPoisk/log/GeobasesearchChangelog/"
        wiki_page_owner = "robot-geosearch"
        observed_paths = [
            "arcadia/extsearch/geo/base",
            "arcadia/extsearch/geo/saas",
            "arcadia/extsearch/geo/indexer",
            "arcadia/extsearch/geo/kernel",
        ]

    class MetricsCfg(configs.ReferenceBranchedConfig.MetricsCfg):
        """Metrics configuration"""
        limit_s = None

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "map-dev/basesearch"
        branch_prefix = "pre-stable"
        tag_name = "geosearch/base"
        tag_folder_template = "{tag_prefix}-{branch_num}/r{tag_num}"
