# -*- coding: utf-8 -*-

from sandbox.projects.release_machine.components import configs as cfg
from sandbox.projects.release_machine.components.config_core import yappy as yappy_cfg
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri

import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils

from sandbox.projects.common.constants import constants as common_const


TASK_GET_HOLLYWOOD_YAPPY_BETA_INFO = "GET_HOLLYWOOD_YAPPY_BETA_INFO"


class HollywoodCfg(cfg.ReferenceBranchedConfig):
    name = "hollywood"
    display_name = "Hollywood (common)"
    responsible = cfg.Responsible(abc=cfg.Abc(component_id=7596, schedule_slug="hotline-and-release"), login="akhruslan")

    class MergesCfg(cfg.ReferenceBranchedConfig.MergesCfg):
        permissions = cfg.MergePermissions(
            permission_type=rm_const.PermissionType.ALLOWED,
            people_groups=cfg.PeopleGroups(
                abc_services=[cfg.Abc(component_id=7596, role_id=None)],
                logins={"akhruslan", "alexanderplat", "avitella", "d-dima", "ispetrukhin", "klim-roma", "vitvlkv", "yagafarov", "zhigan"},
                staff_groups=[],
            ),
        )

    class Testenv(cfg.ReferenceBranchedConfig.Testenv):
        """
        TestEnv configuration settings
        """
        trunk_task_owner = "VINS"
        branch_task_owner = "BASS"

        class JobGraph(cfg.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(
                        ctx={"use_previous_branch_as_baseline": True}
                    ),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_test.JobGraphElementTestTagCommon(
                        task_name="BASS_REGRESS_TESTS_TASK",
                        job_params={
                            "job_name_parameter": "TESTPALM",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerStartrek(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="target_ticket",
                                        output_key="startrek_issue",
                                    ),
                                )
                            )
                        ),
                        ctx={
                            "target_service": "hollywood",
                            "testsuite": "alice",
                        }
                    ),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="TESTPALM",
                            )
                        ),
                        next_revision_custom_schedule=jg_utils.CustomSchedule(
                            time_interval_list=[(4, 4)],
                            weekday_list=[2, 6],
                            once_a_day=True
                        )
                    ),
                ]

            @property
            def _trunk_part(self):
                trunk_part = super(self.__class__, self)._trunk_part
                trunk_part.append(jg_build.JobGraphElementBuildTrunk(
                    task_name="BUILD_HOLLYWOOD_PACKAGE_AND_RELEASE",
                    ctx={
                        "copy_trunk": True,
                        "notify_if_failed": "akhruslan",
                        "package_path": "alice/hollywood/shards/all/prod/hollywood_package.json",
                        "raw_package": False,
                        "release_type": "unstable",
                        "resource_type": "HOLLYWOOD_PACKAGE",
                    },
                    job_arrows=(
                        jg_arrows.ParamsData(
                            input_key=common_const.ARCADIA_URL_KEY,
                            transform=lambda x, rm_config: "{svn_ssh_url}@{revision}".format(
                                svn_ssh_url=x.svn_ssh_url[:x.svn_ssh_url.rfind("arcadia") + len("arcadia")],
                                revision=x.revision,
                            ),
                            override=True,
                        ),
                    ),
                    job_params={
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        "observed_paths": ["alice/hollywood", "alice/megamind/protos/scenarios"],
                    },
                    build_item="TRUNK",
                ))
                for task_name, diff_resource, ammo_resource in [
                    ("HOLLYWOOD_DIFF_TEST", "HOLLYWOOD_DIFF", "HOLLYWOOD_AMMO"),
                    ("HOLLYWOOD_BASS_DIFF_TEST", "HOLLYWOOD_BASS_DIFF", "HOLLYWOOD_BASS_AMMO"),
                ]:
                    trunk_part.append(jg_test.JobGraphElementTestTrunkCommon(
                        task_name="BUILD_HOLLYWOOD_PACKAGE_AND_RELEASE",
                        job_params={
                            "job_name_parameter": task_name,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            "observed_paths": ["alice/hollywood", "alice/megamind/protos/scenarios"],
                            "test_type": jg_utils.TestType.DIFF_TEST,
                            "has_diff": lambda ctx: ctx["has_diff"],
                            "compare_task_name": task_name,
                            "diff_resource_type": {diff_resource: 30},
                            "compare_job_triggers": {
                                jg_arrows.CompareTaskDataCtx(
                                    input_key="old_hollywood_package",
                                    output_key="hollywood_package_resource_id",
                                    task_number=1,
                                ),
                                jg_arrows.CompareTaskDataCtx(
                                    input_key="new_hollywood_package",
                                    output_key="hollywood_package_resource_id",
                                    task_number=2,
                                ),
                                jg_arrows.GlobalResourceData(
                                    input_key="shooter_binary",
                                    resource_name="MEGAMIND_SHOOTER_BINARY",
                                ),
                                jg_arrows.GlobalResourceData(
                                    input_key="hollywood_ammo",
                                    resource_name=ammo_resource,
                                ),
                            },
                        },
                        ctx={
                            "requests_limit": 1000000,
                            "threads": 15,
                            "package_path": "alice/hollywood/shards/all/prod/hollywood_package.json",
                        },
                        platform="linux",
                    ))
                trunk_part.append(jg_test.JobGraphElementMetaTestTrunk(
                    ctx={
                        "targets": "alice/hollywood",
                    },
                    job_arrows=(
                        jg_arrows.ParamsData(
                            input_key=common_const.ARCADIA_URL_KEY,
                            transform=lambda x, rm_config: "{svn_ssh_url}@{revision}".format(
                                svn_ssh_url=x.svn_ssh_url[:x.svn_ssh_url.rfind("arcadia") + len("arcadia")],
                                revision=x.revision,
                            ),
                            override=True,
                        ),
                    ),
                    job_params={
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        "observed_paths": ["alice/hollywood", "alice/megamind/protos/scenarios"],
                    },
                ))
                return trunk_part

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(jg_build.JobGraphElementBuildBranched(
                    task_name="BUILD_HOLLYWOOD_PACKAGE",
                    out={
                        "HOLLYWOOD_PACKAGE": 90,
                    },
                    ctx={
                        "notify_if_failed": "akhruslan",
                        "packages": "alice/hollywood/shards/common/prod/hollywood_package.json",
                        "raw_package": False,
                        "resource_type": "HOLLYWOOD_PACKAGE",
                    },
                ))
                branch_part.append(jg_test.JobGraphElementMetaTestBranched(
                    ctx={
                        "targets": "alice/hollywood",
                    },
                    job_params={
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        "job_name_parameter": "BRANCH",
                    },
                ))
                branch_part.append(jg_test.JobGraphElementYappyBetaGeneratorBranched(
                    beta_conf_type="beta",
                    job_params={
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerBuild(
                            parent_job_data=(
                                jg_job_data.ParentDataDict(
                                    "component_resources",
                                    "package.tar.gz",
                                    "HOLLYWOOD_PACKAGE",
                                ),
                            ),
                        ),
                    ),
                ))

                branch_part.append(jg_test.JobGraphElementTestBranchCommon(
                    task_name=TASK_GET_HOLLYWOOD_YAPPY_BETA_INFO,
                    job_params={
                        "job_name_parameter": TASK_GET_HOLLYWOOD_YAPPY_BETA_INFO,
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    },
                    ctx={
                        "shard": "Common",
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerGenerateBeta(
                            job_name_parameter="beta"
                        ),
                        jg_job_triggers.JobTriggerNewTag(
                            parent_job_data=(
                                jg_job_data.ParentDataOutput(
                                    input_key="in_tag_num",
                                    output_key="new_tag_number",
                                ),
                            )
                        ),
                        jg_arrows.ParamsData("in_branch_num", jg_utils.get_major_release_number),
                    )
                ))
                branch_part.append(jg_test.JobGraphElementTestBranchCommon(
                    task_name="UPDATE_HOLLYWOOD_RELEASE_TICKETS",
                    job_params={
                        "job_name_parameter": "UPDATE_HOLLYWOOD_RELEASE_TICKETS",
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerTestBranchCommon(
                            job_name_parameter=TASK_GET_HOLLYWOOD_YAPPY_BETA_INFO,
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key="in_release_id",
                                    output_key="ctx_release_id",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="in_release_ticket",
                                    output_key="ctx_release_ticket",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="in_assessors_ticket",
                                    output_key="ctx_assessors_ticket",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="in_vins_url",
                                    output_key="ctx_vins_hamster_url",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="in_yandex_station_firmware_version",
                                    output_key="ctx_yandex_station_firmware_version",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="in_yandex_mini_firmware_version",
                                    output_key="ctx_yandex_mini_firmware_version",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="in_yandex_station_config",
                                    output_key="ctx_yandex_station_config",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="in_yandex_mini_config",
                                    output_key="ctx_yandex_mini_config",
                                ),
                            )
                        )
                    )
                ))
                branch_part.append(jg_test.JobGraphElementTestBranchCommon(
                    task_name="BUILD_AND_RUN_MARKER_TESTS",
                    job_params={
                        "job_name_parameter": "BUILD_AND_RUN_MARKER_TESTS",
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerTestBranchCommon(
                            job_name_parameter=TASK_GET_HOLLYWOOD_YAPPY_BETA_INFO,
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key="vins_url",
                                    output_key="ctx_vins_hamster_url",
                                )
                            )
                        )
                    )
                ))
                branch_part.append(jg_test.JobGraphElementTestBranchCommon(
                    task_name="ALICE_EVO_UE_2_E",
                    job_params={
                        "job_name_parameter": "ALICE_EVO_UE_2_E",
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    },
                    ctx={
                        "abc_id": 7596,
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerTestBranchCommon(
                            job_name_parameter=TASK_GET_HOLLYWOOD_YAPPY_BETA_INFO,
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key="component_name",
                                    output_key="ctx_component_name",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="branch_num",
                                    output_key="ctx_branch_num",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="tag_num",
                                    output_key="ctx_tag_num",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="megamind_url",
                                    output_key="ctx_vins_hamster_url",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="release_ticket",
                                    output_key="ctx_release_ticket",
                                )
                            )
                        ),
                    ),
                ))
                branch_part.append(jg_test.JobGraphElementTestBranchCommon(
                    task_name="ALICE_EVO_INTEGRATION_TESTS_WRAPPER",
                    job_params={
                        "job_name_parameter": "ALICE_EVO_INTEGRATION_TESTS_WRAPPER",
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    },
                    ctx={
                        "repeat_failed_test": True,
                        "run_call_owner_subtask": True,
                        "release_fail_threshold": 50,
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerTestBranchCommon(
                            job_name_parameter=TASK_GET_HOLLYWOOD_YAPPY_BETA_INFO,
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key="launch_type",
                                    output_key="ctx_component_name",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="branch_number",
                                    output_key="ctx_branch_num",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="tag_number",
                                    output_key="ctx_tag_num",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="megamind_url",
                                    output_key="ctx_vins_hamster_url",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="release_ticket",
                                    output_key="ctx_release_ticket",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="beta_name",
                                    output_key="ctx_beta_name",
                                )
                            )
                        ),
                    ),
                ))
                branch_part.append(jg_test.JobGraphElementTestBranchCommon(
                    task_name="ALICE_EVO_INTEGRATION_TESTS_WRAPPER",
                    job_params={
                        "job_name_parameter": "ALICE_EVO_MARKER_TESTS_WRAPPER",
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    },
                    ctx={
                        "run_call_owner_subtask": True,
                        "subtask_type": "ALICE_EVO_MARKER_TESTS",
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerTestBranchCommon(
                            job_name_parameter=TASK_GET_HOLLYWOOD_YAPPY_BETA_INFO,
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key="launch_type",
                                    output_key="ctx_component_name",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="branch_number",
                                    output_key="ctx_branch_num",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="tag_number",
                                    output_key="ctx_tag_num",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="megamind_url",
                                    output_key="ctx_vins_hamster_url",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="release_ticket",
                                    output_key="ctx_release_ticket",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="beta_name",
                                    output_key="ctx_beta_name",
                                )
                            )
                        ),
                    ),
                ))
                branch_part.append(jg_test.JobGraphElementTestBranchCommon(
                    task_name="SANDBOX_CI_MEGAMIND_GOODWIN_E2E",
                    job_params={
                        "job_name_parameter": "SANDBOX_CI_MEGAMIND_GOODWIN_E2E",
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerTestBranchCommon(
                            job_name_parameter=TASK_GET_HOLLYWOOD_YAPPY_BETA_INFO,
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key="megamind_url",
                                    output_key="ctx_vins_hamster_url",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="release_ticket",
                                    output_key="ctx_release_ticket",
                                )
                            )
                        )
                    )
                ))
                branch_part.append(jg_test.JobGraphElementTestBranchCommon(
                    task_name="START_HOLLYWOOD_PERF_TESTING",
                    job_params={
                        "job_name_parameter": "START_HOLLYWOOD_PERF_TESTING",
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerTestBranchCommon(
                            job_name_parameter=TASK_GET_HOLLYWOOD_YAPPY_BETA_INFO,
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key="startrek_ticket_id",
                                    output_key="ctx_release_ticket",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="testing_target_url",
                                    output_key="ctx_beta_perf_slot",
                                ),
                                jg_job_data.ParentDataCtx(
                                    input_key="stable_target_url",
                                    output_key="ctx_hamster_no_fast_data_perf_slot",
                                ),
                            )
                        )
                    ),
                    ctx={
                        "ammo_resource": 2185513198,
                        "binary_executor_release_type": "stable",
                        "rps_load_profile": "[{type: const, duration: 60m, ops: 30}]",
                    },
                ))
                for task_name, diff_resource, ammo_resource in [
                    ("HOLLYWOOD_DIFF_TEST", "HOLLYWOOD_DIFF", "HOLLYWOOD_AMMO"),
                    ("HOLLYWOOD_BASS_DIFF_TEST", "HOLLYWOOD_BASS_DIFF", "HOLLYWOOD_BASS_AMMO"),
                ]:
                    branch_part.append(jg_test.JobGraphElementTestBranchCommon(
                        task_name="BUILD_HOLLYWOOD_PACKAGE_AND_RELEASE",
                        job_params={
                            "job_name_parameter": task_name + "_BRANCH",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            "test_type": jg_utils.TestType.DIFF_TEST,
                            "has_diff": lambda ctx: ctx["has_diff"],
                            "compare_task_name": task_name,
                            "diff_resource_type": {diff_resource: 30},
                            "compare_job_triggers": {
                                jg_arrows.CompareTaskDataCtx(
                                    input_key="old_hollywood_package",
                                    output_key="hollywood_package_resource_id",
                                    task_number=1,
                                ),
                                jg_arrows.CompareTaskDataCtx(
                                    input_key="new_hollywood_package",
                                    output_key="hollywood_package_resource_id",
                                    task_number=2,
                                ),
                                jg_arrows.GlobalResourceData(
                                    input_key="shooter_binary",
                                    resource_name="MEGAMIND_SHOOTER_BINARY",
                                ),
                                jg_arrows.GlobalResourceData(
                                    input_key="hollywood_ammo",
                                    resource_name=ammo_resource,
                                ),
                            },
                        },
                        ctx={
                            "requests_limit": 1000000,
                            "threads": 15,
                            "package_path": "alice/hollywood/shards/common/prod/hollywood_package.json",
                        },
                        platform="linux",
                    ))

                return branch_part

            @property
            def _release(self):
                release_part = [
                    jg_release.JobGraphElementNewTag(),
                    jg_release.JobGraphElementChangelogFinal(ctx={"use_previous_branch_as_baseline": True}),
                ]
                release_stage = rm_const.ReleaseStatus.stable
                release_part.append(jg_release.JobGraphElementReleaseBranched(
                    release_to=release_stage,
                    job_arrows=[
                        jg_job_triggers.JobTriggerBuild([
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                "package.tar.gz",
                                "HOLLYWOOD_PACKAGE"),
                        ]),
                        jg_job_triggers.JobTriggerNewTag([
                            jg_job_data.ParentDataCtx("major_release_num", "branch_number_for_tag"),
                            jg_job_data.ParentDataCtx("minor_release_num", "new_tag_number"),
                            jg_job_data.ParentDataCtx("release_number", "scope_number", override=True),
                        ]),
                    ],
                ))
                release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                    release_to=release_stage,
                ))
                return release_part

        class JobPatch(cfg.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def ignore_match(self):
                additional_ignored = [
                    "BUILD_HOLLYWOOD_PACKAGE_AND_RELEASE",
                ]
                return super(self.__class__, self).ignore_match + additional_ignored

    class Notify(cfg.ReferenceBranchedConfig.Notify):
        class Mail(cfg.ReferenceBranchedConfig.Notify.Mail):
            mailing_list = [
                "hollywood-releases@yandex-team.ru",
            ]

        class Startrek(cfg.ReferenceBranchedConfig.Notify.Startrek):
            assignee = cfg.Responsible(abc=cfg.Abc(component_id=7596, schedule_slug="hotline-and-release"), login="akhruslan")

            queue = "ALICERELEASE"

            summary_template = u"Release Hollywood (common) ver.{}"

            add_commiters_as_followers = False
            deadline = 14
            hide_commits_under_cut = True
            important_changes_limit = 1000

            workflow = {
                "open": "testing",
                "testing": "deploying",
                "deploying": "production",
                "production": "close",
            }
            followers = [
                "akhruslan",
                "a-square",
                "vitvlkv",
            ]

            use_task_author_as_assignee = False
            write_merges_in_table = True

            @property
            def ticket_description_prefix(self):
                custom_description = [
                    'Инструкция: https://wiki.yandex-team.ru/alice/hollywood/dezhurstvo-hollywood-priemka-relizov',
                    '',
                    'Дежурный заполняет таблицу ниже. В колонку **"Ссылка"** нужно дать'
                    ' ссылки на подробности соответствующих проверок, а в колонке'
                    ' **"ОК/НЕОК"** нужно оставить что-то одно из двух. Где !!(зел)ОК!! – можно катить в'
                    ' продакшен, а !!НЕОК!! – нельзя катить в продакшен',
                    '',
                    'Чаты:',
                    '* ((https://t.me/joinchat/BBPaeU754t2iM8ROyTY1LQ Чат Дежурства по Алисе))',
                    '* ((https://t.me/joinchat/EHnbnBCKkUbWHGpJHw5GVQ Чат тестирования))',
                    '',
                    '===CHECKS',
                    '#|',
                    '|| **Проверка** | **ОК/НЕОК** | **Cсылка** ||',
                ]
                columns = [
                    'Маркерные тесты',
                    'UE2E',
                    'Перф-тест',
                    'Alice EVO Integration Tests',
                    'Ручная приёмка',
                ]
                custom_description.extend(
                    '|| {} ||'.format(' | '.join([i, '!!(зел)ОК!!/!!НЕОК!!?', '-']))
                    for i in columns
                )
                custom_description.append('|#')
                return '\n'.join(custom_description)

    class ChangelogCfg(cfg.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = ""
        dirs = [
            "arcadia/alice/hollywood",
            "arcadia/alice/megamind/protos/scenarios"
        ]

    class SvnCfg(cfg.ReferenceBranchedConfig.SvnCfg):
        max_active_branches = 4

    class ReleaseViewer(cfg.ReferenceBranchedConfig.ReleaseViewer):

        RELEASE_SAW_GRAPH = (
            "https://solomon.yandex-team.ru/?"
            "project=release_machine&cluster=default&service=release_statistics&description=Release+Saw&"
            "l.sensor=release_saw*&"
            "l.component_name=hollywood&"
            "graph=auto&stack=false&"
            "min=0&b=180d&e=&aggr=max&downsampling=byInterval&grid=1d&downsamplingAggr=max&graphOnly=y"
        )

        statistics_page_charts = [
            RELEASE_SAW_GRAPH,
        ] + cfg.ReferenceBranchedConfig.ReleaseViewer.statistics_page_charts

    class Releases(cfg.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="package.tar.gz",
                    data=ri.SandboxResourceData("HOLLYWOOD_PACKAGE"),
                    deploy_infos=[ri.NannyDeployInfo(
                        [ri.DeployService("hollywood_{}".format(i)) for i in rm_const.MAIN_LOCATIONS]
                    )]
                ),
            ]
        release_followers_permanent = [
            "akhruslan",
        ]
        wait_for_deploy_time_sec = 60 * 60  # 1 hour
        allow_robots_to_release_stable = True

    class Yappy(yappy_cfg.YappyBaseCfg):
        """ Config to generate yappy betas """
        betas = {
            "beta": yappy_cfg.YappyTemplateCfg(
                template_name="alice-hollywood-common-rc",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="alice-hollywood-common-rc-custom",
                        resources=[
                            yappy_cfg.YappyParametrizedResource(
                                param_name="package.tar.gz",
                                local_path="package.tar.gz",
                            ),
                        ],
                        parent_service="hollywood_hamster_no_fast_data",
                    ),
                ],
                working_betas_limit=3,
            )
        }
