# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import yappy as yappy_cfg
from sandbox.projects.common import constants as sandbox_constants
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers


class ImagesBaseCfg(configs.ReferenceBranchedConfig):
    name = "images_base"
    display_name = u"Images Basesearch"
    responsible = "ulgen4a"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_db = "images-base-trunk"
        trunk_task_owner = "IMAGES-TESTENV"
        branch_db_template = "images-base-{testenv_db_num}"
        last_good_revision_ignore_release_jobs = True

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(
                        next_revision_custom_schedule=jg_utils.CustomSchedule(
                            time_interval_list=[(0, 1), (4, 23)],
                            last_run_finished=True,
                            once_a_day=True
                        ),
                    )
                ]

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="IMAGES_BUILD_SEARCH_BINARY",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "ram": 120 * 1024 ** 3,  # 120 GB should be enough to build with LTO
                                },
                                "kill_timeout": 16 * 3600,  # 16 hours
                            },
                        },
                        ctx={
                            "components": "build_all",
                            "register_shard": True,
                            sandbox_constants.LTO: True,
                            "target_platform_flags": "--target-platform=clang12",
                            sandbox_constants.YA_TIMEOUT: 16 * 3600,  # 16 hours
                            "kill_timeout": 16 * 3600,  # 16 hours
                        },
                        out={
                            "IMGSEARCH_EXECUTABLE": 90,
                            "IMAGES_SEARCH_CONFIG": 90,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementYappyBetaGeneratorBranched(
                        beta_conf_type="beta",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "images_basesearch_binary_res_id",
                                        "IMGSEARCH_EXECUTABLE",
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            "beta_name_source": "STRING",
                            "patch_name": "priemka",
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementLaunchMetrics(
                        search_subtype="images",
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="beta",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="checked_beta",
                                        output_key="beta_name",
                                        transform=lambda x, params: "{}.hamster".format(x),
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            "custom_template_name": "common_wizard.json",
                            "enable_autoclicker": False,
                            "sample_beta": "imgs-base-quick.hamster",
                            "scraper_over_yt_pool": "base_images_priemka",
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="IMAGES_PRIEMKA_BASESEARCH_BINARY",
                        job_params={
                            "job_name_parameter": "DAILY_PRIEMKA",
                            "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 1440),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataId("build_task_id"),
                                )
                            ),
                            jg_arrows.ParamsData(
                                input_key="revision_tested",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=jg_utils.get_major_release_number,
                            ),
                        )
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTrunkWaiter()
                )
                umbrella_parents = (
                    jg_job_triggers.JobTriggerLaunchMetrics(
                        job_name_parameter="images",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_name_parameter="DAILY_PRIEMKA",
                    ),
                    jg_job_triggers.JobTriggerTrunkWaiter(),
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedScheduled(
                        job_arrows=umbrella_parents,
                        job_params={
                            "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 1440),
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedByMarker(
                        job_arrows=umbrella_parents,
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                release_part.append(
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "images_basesearch_binary_res_id",
                                        "IMGSEARCH_EXECUTABLE",
                                    ),
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "images_basesearch_config_res_id",
                                        "IMAGES_SEARCH_CONFIG",
                                    ),
                                ),
                            ),
                        )
                    )
                )
                release_part.append(
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                    )
                )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def change_frequency(self):
                uppercase_name = self.name.upper()
                return {
                    "_{}_DAILY_PRIEMKA".format(uppercase_name): rm_const.TestFrequencies.ONCE_A_DAY_TEST,
                    "_LOG_MERGE__{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_{}_UMBRELLA_ACCEPT_BY_MARKER".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_{}_UMBRELLA_ACCEPT_SCHEDULED".format(uppercase_name): rm_const.TestFrequencies.ONCE_A_DAY_TEST,
                    "_BUILD_RELEASE_{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_RELEASE_{}SEARCH".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_NEW_TAG__{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                }

    class Releases(configs.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        allow_robots_to_release_stable = True
        responsible = configs.Responsible(login="ulgen4a", abc=configs.Abc(component_id=90, schedule_slug="DevOpsDuty"))

        resources_info = [
            configs.ReleasedResourceInfo(
                name="images_base",
                resource_type="IMGSEARCH_EXECUTABLE",
                resource_name="images_basesearch_binary_res_id",
                deploy=[(rm_const.ReleaseStatus.stable, "vla-images-base-resources")],
            ),
            configs.ReleasedResourceInfo(
                name="images_shard_checker_binary",
                resource_type="IMGSEARCH_SHARD_CHECKER_EXECUTABLE",
                resource_name="images_shard_checker_binary_res_id",
            ),
            configs.ReleasedResourceInfo(
                name="images_basesearch_config_res_id",
                resource_type="IMAGES_SEARCH_CONFIG",
            ),
        ]
        release_followers_permanent = ["ulgen4a"]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = []
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "ulgen4a"
            queue = "BETATEST"
            dev_queue = "IMAGES"
            summary_template = u"Приёмка бинарников картинок {}"
            workflow = {}
            followers = [
                "ulgen4a",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7
            nanny_reports = False
            commit_importance_threshold = 4
            close_prev_tickets_stage = rm_const.PipelineStage.deploy
            banned_queues = {
                "IGNIETFERRO",
                "MLTOOLS",  # see TOOLSUP-59829, RMDEV-1186
            }

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "yandeximages/releases/"
        dirs = []
        svn_paths_filter = configs.ChangelogPathsFilter(
            rm_const.PermissionType.CHANGED, ["arcadia/extsearch/images"], 5
        )

        @staticmethod
        def calculate_importance(changelog_entry):
            for action, path in changelog_entry.revision_paths:
                if path.startswith("/trunk/arcadia/extsearch/images"):
                    return 5
            return min(3, changelog_entry.mark)

    class MetricsCfg(configs.ReferenceBranchedConfig.MetricsCfg):
        """Metrics configuration"""
        limit_s = None

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "images/base"
        branch_prefix = "pre-stable"
        tag_name = "images/base"
        tag_prefix = "stable"
        tag_folder_template = "{tag_prefix}-{branch_num}/r{tag_num}"

        max_active_branches = 6

    class Yappy(yappy_cfg.YappyBaseCfg):

        betas = {
            "beta": yappy_cfg.YappyTemplateCfg(
                template_name="base-imgs",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="base-imgs",
                        resources=[
                            yappy_cfg.YappyParametrizedResource(
                                local_path="basesearch.executable",
                                param_name="images_basesearch_binary_res_id",
                                checkconfig_name="base-linux-bin-md5",
                            ),
                            yappy_cfg.YappyStaticResource(local_path="basesearch.models", manage_type="BC_DEFAULT"),
                            yappy_cfg.YappyStaticResource(local_path="mmeta.executable", manage_type="BC_DEFAULT"),
                            yappy_cfg.YappyStaticResource(local_path="mmeta.models", manage_type="BC_DEFAULT"),
                            yappy_cfg.YappyStaticResource(local_path="mmeta.rearrange_data", manage_type="BC_DEFAULT"),
                            yappy_cfg.YappyStaticResource(local_path="mmeta.rearrange_index", manage_type="BC_DEFAULT"),
                        ],
                    ),
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="base-quick-imgs",
                        resources=[
                            yappy_cfg.YappyStaticResource(local_path="quick_base.executable", manage_type="BC_DEFAULT"),
                            yappy_cfg.YappyStaticResource(local_path="quick_base.config", manage_type="BC_DEFAULT"),
                            yappy_cfg.YappyStaticResource(local_path="quick_base.models", manage_type="BC_DEFAULT"),
                            yappy_cfg.YappyStaticResource(local_path="quick_base.shardwriter_config", manage_type="BC_DEFAULT"),
                            yappy_cfg.YappyStaticResource(local_path="quick_base.static_models", manage_type="BC_DEFAULT"),
                        ],
                    ),
                ],
            ),
        }
