# -*- coding: utf-8 -*-
#
# Links to release machine docs:
# * https://wiki.yandex-team.ru/releasemachine/
# * https://wiki.yandex-team.ru/releasemachine/create-new-release-machine/

import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.releasable_items as ri
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
from sandbox.projects.release_machine.components.config_core import yappy as yappy_cfg
from sandbox.projects.common import constants as sandbox_constants
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


class ImagesRobotCfg(configs.ReferenceBranchedConfig):
    name = "images_robot"
    responsible = "ulgen4a"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_db = "images-robot-trunk"
        branch_db_template = "images-robot-{testenv_db_num}"
        trunk_task_owner = "IMAGES-TESTENV"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(
                        next_revision_custom_schedule=jg_utils.CustomSchedule(
                            time_interval_list=[(0, 3)],
                            last_run_finished=True,
                            seconds_from_last_run_start_time=2 * 24 * 3600,  # every 2 days
                            once_a_day=True
                        ),
                    )
                ]

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="IMAGES_BUILD_MAIN_ROBOT",
                        job_params={
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="arcadia_url",
                                        output_key=sandbox_constants.ARCADIA_URL_KEY,
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            "clear_build": True,
                            "build_type": "last_tag",
                        },
                        out={
                            "IMAGES_CM_PACKAGE": 40,
                            "IMAGES_ROBOT_RESOURCES_LIST": 40,
                            "IMAGES_MR_INDEX_CONFIG": 40,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="IMAGES_PRIEMKA_RT_ROBOT",
                        job_params={
                            "job_name_parameter": "QUICK_ROBOT_ACCEPTANCE",
                            "should_add_to_db": jg_utils.should_add_to_db_branch,
                            "cancel_fallbehind_runs_on_fix": False,
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="quick_bundle_param",
                                        output_key="build_quick_task_id",
                                    ),
                                    jg_job_data.ParentDataResource(
                                        input_key="mr_config_param",
                                        resource_name="IMAGES_MR_INDEX_CONFIG",
                                    ),
                                ),
                            ),
                        ),
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="IMAGES_ROBOT_CM_RUN_WAIT",
                        job_params={
                            "job_name_parameter": "NEW_ROBOT_BRANCH_ACCEPTANCE",
                            "should_add_to_db": jg_utils.should_add_to_db_branch,
                            "cancel_fallbehind_runs_on_fix": False,
                            "apiargs": {
                                "max_restarts": 40,  # ~20h
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerRelease(
                                job_name_parameter=rm_const.ReleaseStatus.testing
                            ),
                        ),
                        ctx={
                            "cm_url": "https://images-cm-new-acceptance.n.yandex-team.ru",
                            "cm_goal": "workflow-finish",
                            "wait_goal": True,
                            "use_last_binary": True,
                        }
                    )
                )
                umbrella_parents = (
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_name_parameter="QUICK_ROBOT_ACCEPTANCE"
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_name_parameter="NEW_ROBOT_BRANCH_ACCEPTANCE"
                    ),
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedScheduled(
                        job_arrows=umbrella_parents,
                        job_params={
                            "job_name_parameter": "UMBRELLA_ACCEPT_SCHEDULED",
                            "frequency": (jg_utils.TestFrequency.EVERY_N_COMMIT, 1000000),
                        },
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                for release_stage in [rm_const.ReleaseStatus.stable, rm_const.ReleaseStatus.testing]:
                    release_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "images_robot_cm_package",
                                            "IMAGES_CM_PACKAGE",
                                        ),
                                    ),
                                ),
                            )
                        )
                    )
                    release_part.append(
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                        )
                    )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "DELIVER_IMAGEST_ROBOT_BACKEND_BINARIES_TAG": rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

    class Releases(configs.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        @property
        def releasable_items(self):
            deploy_infos = [
                ri.NannyDeployInfo(ri.DeployService("sas_imgs_cm_main"), stage=rm_const.ReleaseStatus.stable),
                ri.NannyDeployInfo(ri.DeployService("sas_imgs_cm_mrdb"), stage=rm_const.ReleaseStatus.stable),
                ri.NannyDeployInfo(ri.DeployService("sas_imgs_cm_taas_big"), stage=rm_const.ReleaseStatus.stable),
                ri.NannyDeployInfo(ri.DeployService("sas_imgs_cm_acceptance"), stage=rm_const.ReleaseStatus.testing),
                ri.NannyDeployInfo(ri.DeployService("imgs_cm_new_acceptance"), stage=rm_const.ReleaseStatus.testing),
            ]

            return [
                ri.ReleasableItem(
                    name="images_robot_cm_package",
                    data=ri.SandboxResourceData("IMAGES_CM_PACKAGE"),
                    deploy_infos=deploy_infos,
                ),
            ]

        allow_robots_to_release_stable = True
        allow_old_releases = True

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""

        notifications = [
            rm_notifications.Notification(
                event_type="BuildTest",
                chat_name="images_robot_mail",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="ReleaseFailed",
                chat_name="images_robot",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="MergeCommit",
                chat_name="images_robot",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="images_robot",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_deployed_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="images_robot",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
        ]

        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            """Mail notifications configuration"""
            mailing_list = [
                "images-robot@yandex-team.ru",
            ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = ["images_robot"]
            config = configs.RmTelegramNotifyConfig(chats=chats)
            invite_link = u"https://t.me/joinchat/BzpcrkOG9XCnKJcE47Pwkg"

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "ulgen4a"
            queue = "IMGROBOT"
            summary_template = u"Релиз бинарей Яндекс.Робот {}"
            workflow = {}
            add_commiters_as_followers = False
            deadline = 7
            hide_commits_under_cut = True

            @property
            def followers(self):
                team = [
                    "igorluk",
                    "mrboogie",
                    "sakhnov",
                    "ulgen4a",
                ]

                return team

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "YandexImages/Robot/Releases/"
        dirs = [
            "arcadia/extsearch/images/robot",
        ]
        review_groups = [
            "images-robot",
        ]

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "images/mrindexer"
        branch_prefix = "pre-stable"
        tag_name = "images/mrindexer"
        tag_prefix = "stable"
        tag_folder_template = "{tag_prefix}-{branch_num}/r{tag_num}"
        allow_autobranches = True

        max_active_branches = 6

    class Yappy(yappy_cfg.YappyBaseCfg):
        pass
