# -*- coding: utf-8 -*-
import datetime

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.common import constants as sandbox_constants
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


class ImagesSemidupsCfg(configs.ReferenceBranchedConfig):
    name = "images_semidups"
    responsible = "volynkin"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_db = "images-semidups-trunk"
        trunk_task_owner = "SEMIDUPS-RELEASERS"
        testenv_db_owners = ["volynkin"]
        branch_db_template = "images-semidups-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            IMAGES_SEMIDUPS_COMPLEX_TEST_CACHE_PATH = "//home/cvdup/semidups/acceptance_states/cache_v8v9"

            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(
                        next_revision_custom_schedule=jg_utils.CustomSchedule(
                            time_interval_list=[(1, 1)],
                            weekday_list=[5],
                            last_run_finished=True,
                            once_a_day=True
                        ),
                    ),
                ]

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                build_params = [
                    (
                        "",
                        "BUILD_CVDUP_BUNDLE",
                        {
                            "CVDUP_IMAGEREDUCE": 90,
                            "CVDUP_TESTTOOL": 90,
                            "IMAGES_DOWNLOAD_RESOURCE_EXECUTABLE": 90,
                            "CV_IMAGEPARSER": 90,
                            "CVDUP_IMAGEPARSER_CONFIG": 90
                        },
                    ),
                    (
                        "TEST_TOOLS",
                        "BUILD_CVDUP_TOOLS_BUNDLE",
                        {
                            "CVDUP_SAMPLING_TOOL": 90,
                            "CVDUP_IMAGE_EXTRACTOR_TOOL": 90,
                        },
                    ),
                    (
                        "CVDUP_ACCEPTANCE_UTIL_EXECUTABLE",
                        "IMAGES_BUILD_CVDUP_ACCEPTANCE_UTIL_RESOURCES",
                        {
                            "IMAGES_CVDUP_ACCEPTANCE_UTIL_EXECUTABLE": 90,
                        },
                    )
                ]
                for build_item, task_name, out in build_params:
                    branch_part.append(
                        jg_build.JobGraphElementYaMakeBuildBranched(
                            task_name=task_name,
                            job_params={
                                "apiargs": {
                                    "requirements": {
                                        "ram": 12 * 1024 ** 3,  # 12 Gb
                                        "disk_space": 80 * (1024 ** 3),  # 80 Gb
                                    },
                                    "kill_timeout": 5 * 3600,
                                }
                            },
                            build_item=build_item,
                            ctx={
                                "components": "build_all",
                                sandbox_constants.LTO: True,
                                sandbox_constants.YA_TIMEOUT: 5 * 3600,  # 5 hours
                                "timestamp": datetime.datetime.now().strftime("%Y%m%d-%H%M%S") if build_item == "" else "",
                            },
                            out=out,
                        )
                    )
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="TEST_CVDUP_ON_SYNTHETIC",
                        build_item="SYNTHETIC_TEST",
                        job_params={
                            "apiargs": {
                                "kill_timeout": 24 * 3600,
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="imagereduce_binary",
                                        resource_name="CVDUP_IMAGEREDUCE",
                                    ),
                                    jg_job_data.ParentDataResource(
                                        input_key="testtool_binary",
                                        resource_name="CVDUP_TESTTOOL",
                                    ),
                                    jg_job_data.ParentDataResource(
                                        input_key="imparser_binary",
                                        resource_name="CV_IMAGEPARSER",
                                    ),
                                    jg_job_data.ParentDataResource(
                                        input_key="imparser_config",
                                        resource_name="CVDUP_IMAGEPARSER_CONFIG",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="timestamp",
                                        output_key="timestamp",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="mr_path",
                                        output_key="timestamp",
                                        transform=lambda x, params: "//home/cvdup/synthetic_test/testenv_images_semidups/launch_{}_{}".format(
                                            params.ctx["testenv_database"],
                                            x,
                                        ),
                                    ),
                                )
                            )
                        ),
                        ctx={
                            sandbox_constants.YA_TIMEOUT: 24 * 3600,  # 24 hours
                        }
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="TEST_CVDUP_CLEAN_OLD_BRANCHES_DATA",
                        build_item="SYNTHETIC_TEST_CLEAN_DATA",
                        job_params={
                            "apiargs": {
                                "kill_timeout": 24 * 3600,
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="branch_number_for_tag",
                                        output_key="branch_number_for_tag",
                                        transform=lambda x, params: str(x),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="SYNTHETIC_TEST",
                            ),
                        ),
                        ctx={
                            "mr_path": "//home/cvdup/synthetic_test/testenv_images_semidups",
                            "num_branches": 5,
                            sandbox_constants.YA_TIMEOUT: 24 * 3600,  # 24 hours
                        }
                    )
                )
                for job_name_param, number, stable_transform, branch_transform in [
                    (
                        "FIRST",
                        1,
                        lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/dataset1_stable_binary".format(
                            params.ctx["testenv_database"],
                            x,
                        ),
                        lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/dataset1_branch_binary".format(
                            params.ctx["testenv_database"],
                            x,
                        ),
                    ),
                    (
                        "SECOND",
                        2,
                        lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/dataset2_stable_binary".format(
                            params.ctx["testenv_database"],
                            x,
                        ),
                        lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/dataset2_branch_binary".format(
                            params.ctx["testenv_database"],
                            x,
                        ),
                    )
                ]:
                    branch_part.append(
                        jg_build.JobGraphElementYaMakeBuildBranched(
                            task_name="TEST_CVDUP_CREATE_STATE",
                            build_item="COMPLEX_TEST_CREATE_{}_STATE_BY_STABLE_BINARY".format(job_name_param),
                            job_params={
                                "apiargs": {
                                    "kill_timeout": 80 * 3600,
                                },
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="timestamp",
                                            output_key="timestamp",
                                        ),
                                        jg_job_data.ParentDataOutput(
                                            input_key="mr_path",
                                            output_key="timestamp",
                                            transform=stable_transform,
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerNewTag(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="branch_number_for_tag",
                                            output_key="branch_number_for_tag",
                                            transform=lambda x, params: str(x),
                                        ),
                                    ),
                                ),
                            ),
                            ctx={
                                "dataset_path": "//home/cvdup/semidups/acceptance_data/source{}.v8v9ocr2class9".format(str(number)),
                                "mr_states_cache_path": self.IMAGES_SEMIDUPS_COMPLEX_TEST_CACHE_PATH,
                                "state_id": number,
                                sandbox_constants.YA_TIMEOUT: 80 * 3600,  # 80 hours
                            }
                        )
                    )
                    branch_part.append(
                        jg_build.JobGraphElementYaMakeBuildBranched(
                            task_name="TEST_CVDUP_CREATE_STATE",
                            build_item="COMPLEX_TEST_CREATE_{}_STATE_BY_BRANCH_BINARY".format(job_name_param),
                            job_params={
                                "apiargs": {
                                    "kill_timeout": 80 * 3600,
                                },
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="timestamp",
                                            output_key="timestamp",
                                        ),
                                        jg_job_data.ParentDataOutput(
                                            input_key="mr_path",
                                            output_key="timestamp",
                                            transform=branch_transform,
                                        ),
                                        jg_job_data.ParentDataResource(
                                            input_key="image_reducer_binary",
                                            resource_name="CVDUP_IMAGEREDUCE",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerNewTag(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="branch_number_for_tag",
                                            output_key="branch_number_for_tag",
                                            transform=lambda x, params: str(x),
                                        ),
                                    ),
                                ),
                            ),
                            ctx={
                                "dataset_path": "//home/cvdup/semidups/acceptance_data/source{}.v8v9ocr2class9".format(
                                    str(number)),
                                "mr_states_cache_path": self.IMAGES_SEMIDUPS_COMPLEX_TEST_CACHE_PATH,
                                "state_id": number,
                                sandbox_constants.YA_TIMEOUT: 80 * 3600,  # 80 hours
                            },
                        )
                    )
                for test_type, mr_path_1_transform, mr_path_2_transform in [
                    (
                        "stable",
                        lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/dataset1_stable_binary".format(
                            params.ctx["testenv_database"],
                            x,
                        ),
                        lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/dataset2_stable_binary".format(
                            params.ctx["testenv_database"],
                            x,
                        ),
                    ),
                    (
                        "branch",
                        lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/dataset1_branch_binary".format(
                            params.ctx["testenv_database"],
                            x,
                        ),
                        lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/dataset2_branch_binary".format(
                            params.ctx["testenv_database"],
                            x,
                        ),
                    )
                ]:
                    branch_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="TEST_CVDUP_MERGE_STATES",
                            job_params={
                                "job_name_parameter": "COMPLEX_TEST_MERGE_STATES_BY_{}_BINARY".format(test_type.upper()),
                                "apiargs": {
                                    "kill_timeout": 80 * 3600,
                                },
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="timestamp",
                                            output_key="timestamp",
                                        ),
                                        jg_job_data.ParentDataOutput(
                                            input_key="mr_path_1",
                                            output_key="timestamp",
                                            transform=mr_path_1_transform,
                                        ),
                                        jg_job_data.ParentDataOutput(
                                            input_key="mr_path_2",
                                            output_key="timestamp",
                                            transform=mr_path_2_transform,
                                        ),
                                        jg_job_data.ParentDataResource(
                                            input_key="image_reducer_binary",
                                            resource_name="CVDUP_IMAGEREDUCE",
                                        ) if test_type == "branch" else None,
                                    ),
                                ),
                                jg_job_triggers.JobTriggerNewTag(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="branch_number_for_tag",
                                            output_key="branch_number_for_tag",
                                            transform=lambda x, params: str(x),
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="COMPLEX_TEST_CREATE_FIRST_STATE_BY_{}_BINARY".format(test_type.upper()),
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="COMPLEX_TEST_CREATE_SECOND_STATE_BY_{}_BINARY".format(test_type.upper()),
                                ),
                            ),
                            ctx={
                                "mr_states_cache_path": self.IMAGES_SEMIDUPS_COMPLEX_TEST_CACHE_PATH,
                                "state_id": 2,
                                sandbox_constants.YA_TIMEOUT: 80 * 3600,  # 80 hours
                            }
                        )
                    )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="TEST_CVDUP_SAMPLE_DIFF",
                        job_params={
                            "job_name_parameter": "COMPLEX_TEST_SAMPLE_DIFF",
                            "apiargs": {
                                "kill_timeout": 80 * 3600,
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="timestamp",
                                        output_key="timestamp",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="mr_path",
                                        output_key="timestamp",
                                        transform=lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}".format(
                                            params.ctx["testenv_database"],
                                            x,
                                        ),
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="mr_path_1",
                                        output_key="timestamp",
                                        transform=lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/dataset2_stable_binary".format(
                                            params.ctx["testenv_database"],
                                            x,
                                        ),
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="mr_path_2",
                                        output_key="timestamp",
                                        transform=lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/dataset2_branch_binary".format(
                                            params.ctx["testenv_database"],
                                            x,
                                        ),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="branch_number_for_tag",
                                        output_key="branch_number_for_tag",
                                        transform=lambda x, params: str(x),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="TEST_TOOLS",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="sampling_tool_binary",
                                        resource_name="CVDUP_SAMPLING_TOOL",
                                    ),
                                    jg_job_data.ParentDataResource(
                                        input_key="image_extractor_tool_binary",
                                        resource_name="CVDUP_IMAGE_EXTRACTOR_TOOL",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="COMPLEX_TEST_MERGE_STATES_BY_STABLE_BINARY",
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="COMPLEX_TEST_MERGE_STATES_BY_BRANCH_BINARY",
                            ),
                        ),
                        ctx={
                            "dataset_path1": "//home/cvdup/semidups/acceptance_data/source1.v8v9ocr2class9",
                            "dataset_path2": "//home/cvdup/semidups/acceptance_data/source2.v8v9ocr2class9",
                            sandbox_constants.YA_TIMEOUT: 80 * 3600,  # 80 hours
                        },
                        out={
                            "CVDUP_ACCEPTANCE_IMAGE_URL_PAIRS_JSON": 90,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="TEST_CVDUP_TOLOKA_ESTIMATING",
                        job_params={
                            "job_name_parameter": "COMPLEX_TEST_ESTIMATE_DIFF",
                            "apiargs": {
                                "kill_timeout": 48 * 3600,
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="branch_number_for_tag",
                                        output_key="branch_number_for_tag",
                                        transform=lambda x, params: str(x),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="timestamp",
                                        output_key="timestamp",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="COMPLEX_TEST_SAMPLE_DIFF",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="url_pairs_json",
                                        resource_name="CVDUP_ACCEPTANCE_IMAGE_URL_PAIRS_JSON",
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            sandbox_constants.YA_TIMEOUT: 48 * 3600,  # 48 hours
                        },
                        out={
                            "CVDUP_ACCEPTANCE_TOLOKA_ESTIMATES_JSON": 90,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="TEST_CVDUP_LOAD_TOLOKA_ESTIMATES",
                        job_params={
                            "job_name_parameter": "COMPLEX_TEST_LOAD_ESTIMATES",
                            "apiargs": {
                                "kill_timeout": 8 * 3600,
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="branch_number_for_tag",
                                        output_key="branch_number_for_tag",
                                        transform=lambda x, params: str(x),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="timestamp",
                                        output_key="timestamp",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="output_table_path",
                                        output_key="timestamp",
                                        transform=lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/toloka_estimates".format(
                                            params.ctx["testenv_database"],
                                            x,
                                        ),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="COMPLEX_TEST_ESTIMATE_DIFF",
                            ),
                        ),
                        ctx={
                            sandbox_constants.YA_TIMEOUT: 8 * 3600,  # 8 hours
                        },
                    )
                )
                for test_type, branch_transform in [
                    (
                        "stable",
                        lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/dataset2_stable_binary".format(
                            params.ctx["testenv_database"],
                            x
                        )
                    ),
                    (
                        "branch",
                        lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/dataset2_branch_binary".format(
                            params.ctx["testenv_database"],
                            x
                        )
                    )
                ]:
                    branch_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="TEST_CVDUP_CALCULATE_METRICS",
                            job_params={
                                "job_name_parameter": "COMPLEX_TEST_CALCULATE_METRICS_{}".format(test_type.upper()),
                                "apiargs": {
                                    "kill_timeout": 6 * 3600,
                                },
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerNewTag(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="branch_number_for_tag",
                                            output_key="branch_number_for_tag",
                                            transform=lambda x, params: str(x),
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="timestamp",
                                            output_key="timestamp",
                                        ),
                                        jg_job_data.ParentDataOutput(
                                            input_key="state_path",
                                            output_key="timestamp",
                                            transform=branch_transform
                                        ),
                                        jg_job_data.ParentDataOutput(
                                            input_key="toloka_estimates_table",
                                            output_key="timestamp",
                                            transform=lambda x, params: "//home/cvdup/complex_test/testenv_images_semidups/launch_{}_{}/toloka_estimates".format(
                                                params.ctx["testenv_database"],
                                                x,
                                            ),
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="CVDUP_ACCEPTANCE_UTIL_EXECUTABLE",
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="images_cvdup_acceptance_util_binary",
                                            resource_name="IMAGES_CVDUP_ACCEPTANCE_UTIL_EXECUTABLE",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter="COMPLEX_TEST_ESTIMATE_DIFF",
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="toloka_estimates_portion_json",
                                            resource_name="CVDUP_ACCEPTANCE_TOLOKA_ESTIMATES_JSON",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter="COMPLEX_TEST_LOAD_ESTIMATES" if test_type == "stable" else "COMPLEX_TEST_CALCULATE_METRICS_STABLE",
                                ),
                            ),
                            ctx={
                                sandbox_constants.YA_TIMEOUT: 6 * 3600,  # 6 hours
                            },
                            out={
                                "CVDUP_ACCEPTANCE_STATE_METRICS_JSON": 90,
                            },
                        )
                    )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="TEST_CVDUP_CHECK_METRICS",
                        job_params={
                            "job_name_parameter": "COMPLEX_TEST_CHECK_METRICS",
                            "apiargs": {
                                "kill_timeout": 2 * 3600,
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="branch_number_for_tag",
                                        output_key="branch_number_for_tag",
                                        transform=lambda x, params: str(x),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="timestamp",
                                        output_key="timestamp",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="COMPLEX_TEST_CALCULATE_METRICS_STABLE",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="stable_state_metrics_json",
                                        resource_name="CVDUP_ACCEPTANCE_STATE_METRICS_JSON",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="COMPLEX_TEST_CALCULATE_METRICS_BRANCH",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="branch_state_metrics_json",
                                        resource_name="CVDUP_ACCEPTANCE_STATE_METRICS_JSON",
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            sandbox_constants.YA_TIMEOUT: 2 * 3600,  # 2 hours
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="TEST_CVDUP_CLEAN_CACHE",
                        job_params={
                            "job_name_parameter": "COMPLEX_TEST_CLEAN_CACHE",
                            "apiargs": {
                                "kill_timeout": 24 * 3600,
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="branch_number_for_tag",
                                        output_key="branch_number_for_tag",
                                        transform=lambda x, params: str(x),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="COMPLEX_TEST_CHECK_METRICS",
                            ),
                        ),
                        ctx={
                            "mr_path": self.IMAGES_SEMIDUPS_COMPLEX_TEST_CACHE_PATH,
                            "num_binaries": 5,
                            sandbox_constants.YA_TIMEOUT: 24 * 3600,  # 24 hours
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="TEST_CVDUP_CLEAN_OLD_BRANCHES_DATA",
                        job_params={
                            "job_name_parameter": "COMPLEX_TEST_CLEAN_DATA",
                            "apiargs": {
                                "kill_timeout": 24 * 3600,
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="branch_number_for_tag",
                                        output_key="branch_number_for_tag",
                                        transform=lambda x, params: str(x),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="COMPLEX_TEST_CLEAN_CACHE",
                            ),
                        ),
                        ctx={
                            "mr_path": "//home/cvdup/complex_test/testenv_images_semidups",
                            "num_binaries": 5,
                            sandbox_constants.YA_TIMEOUT: 24 * 3600,  # 24 hours
                        },
                    )
                )
                umbrella_parents = (
                    jg_job_triggers.JobTriggerBuild(
                        job_name_parameter="SYNTHETIC_TEST_CLEAN_DATA",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_name_parameter="COMPLEX_TEST_CLEAN_DATA",
                    ),
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedScheduled(
                        job_arrows=umbrella_parents,
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedByMarker(
                        job_arrows=umbrella_parents,
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                release_part.append(
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "cvdup_imagereduce_binary_res_id",
                                        "CVDUP_IMAGEREDUCE",
                                    ),
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "cvdup_testtool_binary_res_id",
                                        "CVDUP_TESTTOOL",
                                    ),
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "cv_imageparser_binary_res_id",
                                        "CV_IMAGEPARSER",
                                    ),
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "cvdup_imageparser_config_res_id",
                                        "CVDUP_IMAGEPARSER_CONFIG",
                                    ),
                                )
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="TEST_TOOLS",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "cvdup_samplingtool_binary_res_id",
                                        "CVDUP_SAMPLING_TOOL",
                                    ),
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "cvdup_imageextractortool_binary_res_id",
                                        "CVDUP_IMAGE_EXTRACTOR_TOOL",
                                    ),
                                )
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="CVDUP_ACCEPTANCE_UTIL_EXECUTABLE",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "images_cvdupacceptanceutil_binary_res_id",
                                        "IMAGES_CVDUP_ACCEPTANCE_UTIL_EXECUTABLE",
                                    ),
                                )
                            )
                        )
                    )
                )
                release_part.append(
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                    )
                )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""
            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "_ACTION_RUN_ACCEPTANCE_BRANCHED__IMAGES_SEMIDUPS__SCHEDULED": rm_const.TestFrequencies.EACH_REV_TEST
                })
                return tests

    class Releases(configs.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        resources_info = [
            configs.ReleasedResourceInfo(
                name="cvdup_imagereduce_binary",
                resource_type="CVDUP_IMAGEREDUCE",
                resource_name="cvdup_imagereduce_binary_res_id",
            ),
            configs.ReleasedResourceInfo(
                name="cvdup_testtool_binary",
                resource_type="CVDUP_TESTTOOL",
                resource_name="cvdup_testtool_binary_res_id",
            ),
            configs.ReleasedResourceInfo(
                name="cv_imageparser_binary",
                resource_type="CV_IMAGEPARSER",
                resource_name="cv_imageparser_binary_res_id",
            ),
            configs.ReleasedResourceInfo(
                name="cvdup_imageparser_config",
                resource_type="CVDUP_IMAGEPARSER_CONFIG",
                resource_name="cvdup_imageparser_config_res_id",
            ),
            configs.ReleasedResourceInfo(
                name="cvdup_samplingtool_binary",
                resource_type="CVDUP_SAMPLING_TOOL",
                resource_name="cvdup_samplingtool_binary_res_id",
            ),
            configs.ReleasedResourceInfo(
                name="cvdup_imageextractortool_binary",
                resource_type="CVDUP_IMAGE_EXTRACTOR_TOOL",
                resource_name="cvdup_imageextractortool_binary_res_id",
            ),
            configs.ReleasedResourceInfo(
                name="images_cvdupacceptanceutil_binary",
                resource_type="IMAGES_CVDUP_ACCEPTANCE_UTIL_EXECUTABLE",
                resource_name="images_cvdupacceptanceutil_binary_res_id",
            )
        ]
        release_followers_permanent = [
            "volynkin",
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""

        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            """Mail notifications configuration"""
            mailing_list = [
                "images-semidups@yandex-team.ru",
            ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = ["images-semidups"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "volynkin"
            queue = "IMGSEMIDUPS"
            dev_queue = "SEARCH"
            summary_template = u"Приемка IMAGES Semidups {}"
            followers = [
                "volynkin",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "JandeksPoisk/IMAGES_SEMIDUPS/releases/"
        dirs = [
            "arcadia/cv/semidup2",
            "arcadia/extsearch/images/robot/tools/cvdup_acceptance_util",
            "arcadia/extsearch/images/robot/tools/cvdup_acceptance_util_protos",
        ]
        review_groups = [
            "images",
        ]

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "images/semidups"
        tag_name = "images/semidups"
