import os.path

from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core.jg import base as jg_base
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow
from sandbox.projects.release_machine.components.config_core.jg.cube import base as jg_cube
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import build as build_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import internal as internal_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import release as release_cubes
from sandbox.projects.release_machine.components.config_core.jg.preset import basic_build_presets as jg_preset


BANNERLAND_WEB_BACKEND_BIN = "BANNER_LAND_WEB_BACKEND_BINARY"
BANNERLAND_WEB_STATIC = "BANNER_LAND_WEB_STATIC"
BANNER_LAND_WEB_DATABASE_MIGRATE = "BANNER_LAND_WEB_DATABASE_MIGRATE"
BANNER_LAND_WEB_API_TEST = "BANNER_LAND_WEB_API_TEST"

SANDBOX_TASKS_BINARY = "SANDBOX_TASKS_BINARY"

PROJECT_PATH = "irt/bannerland/backend"


def get_path(path):
    return "{}/{}".format(PROJECT_PATH, path)


class ReleaseInfo:
    def __init__(self, resource_type, target, deploy_system):
        self.resource_type = resource_type
        self.target = get_path(target)
        self.deploy_system = deploy_system


YA_MAKE_TASKS = (
    ReleaseInfo(
        resource_type=BANNERLAND_WEB_BACKEND_BIN,
        target="bin/app",
        deploy_system=rm_const.DeploySystem.ya_deploy,
    ),
    ReleaseInfo(
        resource_type=BANNERLAND_WEB_STATIC,
        target="static",
        deploy_system=rm_const.DeploySystem.kosher_sandbox_release,
    ),
)

SANDBOX_TASKS = {
    BANNER_LAND_WEB_DATABASE_MIGRATE: get_path("sandbox_tasks/database_migrate"),
    BANNER_LAND_WEB_API_TEST: get_path("sandbox_tasks/integration_api_test"),
}


DEPLOY_SERVICE = "bannerland-web"


RELEASED_TYPES = {
    rm_const.ReleaseStatus.testing: rm_const.ReleaseStatus.prestable,
    rm_const.ReleaseStatus.prestable: rm_const.ReleaseStatus.stable,
    rm_const.ReleaseStatus.stable: None,
}

BUILD_STAGE = "build"


def get_deploy_stage_name(release_type):
    return "{}-{}".format(DEPLOY_SERVICE, release_type)


def _arcadia_url(new_tag_cube):
    return jg_cube.CubeOutputTransformed(
        [
            new_tag_cube.output.vcs_data.arc.tag_path,
        ],
        lambda l: "arcadia-arc:/#{}".format(*l),
    )


def _get_sandbox_task_build_cube(new_tag_cube, task_type, name, targets):
    return build_cubes.DeployBinaryTask(
        target=targets,
        target_task_type=task_type,
        name=name,
        input=jg_cube.CubeInput(
            arcadia_url=_arcadia_url(new_tag_cube),
        ),
    )


def _cube_name(*args):
    return "__".join(map(lambda x: x.lower(), args))


def _get_build_cube(graph, artifact_type):
    return graph.get(_cube_name(BUILD_STAGE, artifact_type))


def _get_build_artifact(graph, artifact_type, resource_type=None):
    return _get_build_cube(graph, artifact_type).output.resources[resource_type or artifact_type].first().id


def _get_deploy_info(deploy_system, release_types):
    if deploy_system.name == rm_const.DeploySystem.ya_deploy.name:
        return [
            ri.YaDeployInfo(
                ri.DeployService(get_deploy_stage_name(release_type)),
                stage=release_type,
            ) for release_type in release_types
        ]
    elif deploy_system.name == rm_const.DeploySystem.kosher_sandbox_release.name:
        return [
            ri.SandboxInfo(stage=release_type) for release_type in release_types
        ] + [
            ri.SandboxKosherReleaseInfo(stage=release_type) for release_type in release_types
        ]
    else:
        raise ValueError("Please add support of DeploySystem `{}`".format(deploy_system))


class BannerLandWebCfg(configs.ReferenceCIConfig):
    name = "bannerland_web"
    display_name = "BannerLand Web UI"

    responsible = configs.Responsible(
        abc=configs.Abc(service_name="bannerland"),
        login="danila-eremin",
    )

    class JG(jg_base.BaseReleaseMachineJobGraph):
        @jg_flow.release_flow(
            True,
            stages=[
                jg_preset.DEFAULT_STAGE_NEW_TAG,
                jg_preset.DEFAULT_STAGE_BUILD.add_cube_name(
                    _cube_name(BUILD_STAGE, "*")
                ),
                jg_preset.DEFAULT_STAGE_TEST.add_cube_name(
                    _cube_name(rm_const.ReleaseStatus.testing, "*")
                ),
                jg_preset.DEFAULT_STAGE_PRESTABLE_RELEASE.add_cube_name(
                    _cube_name(rm_const.ReleaseStatus.prestable, "*")
                ),
                jg_preset.DEFAULT_STAGE_STABLE_RELEASE.add_cube_name(
                    _cube_name(rm_const.ReleaseStatus.stable, "*")
                ),
            ],
        )
        def release(self):
            graph = super(BannerLandWebCfg.JG, self).release(self)
            new_tag = graph.get("new_tag")
            main_entry = graph.get(dummy_cubes.RMMainGraphEntry.NAME)

            testing_entry = dummy_cubes.TestStageEntry()
            testing_entry.add_requirement(graph.get(internal_cubes.LinkFeatureTickets.NAME))
            testing_entry.add_requirement(graph.get(internal_cubes.PostChangelogToStartrek.NAME))
            graph.add(testing_entry)

            prestable_release_entry = dummy_cubes.ReleasePrestableStageEntry()
            prestable_release_entry.add_requirement(testing_entry)
            graph.add(prestable_release_entry)

            stable_release_entry = dummy_cubes.ReleaseStableStageEntry(manual=True)
            stable_release_entry.add_requirement(prestable_release_entry)
            graph.add(stable_release_entry)

            release_stage_entry = {
                rm_const.ReleaseStatus.stable: stable_release_entry,
                rm_const.ReleaseStatus.prestable: prestable_release_entry,
                rm_const.ReleaseStatus.testing: testing_entry,
            }

            # Build

            # Build SB tasks

            for sandbox_task_type in SANDBOX_TASKS:
                build_cube = _get_sandbox_task_build_cube(
                    new_tag,
                    sandbox_task_type,
                    name=_cube_name(BUILD_STAGE, sandbox_task_type),
                    targets=SANDBOX_TASKS[sandbox_task_type],
                )
                build_cube.add_requirement(main_entry)
                testing_entry.add_requirement(build_cube)
                graph.add(build_cube)

            # Build arcadia targets

            for release_info in YA_MAKE_TASKS:
                build_cube = build_cubes.KosherYaMake(
                    name=_cube_name(BUILD_STAGE, release_info.resource_type),
                    targets=os.path.dirname(release_info.target),
                    artifacts=release_info.target,
                    resource_types=release_info.resource_type,
                    input=jg_cube.CubeInput(
                        result_single_file=True,
                        checkout_arcadia_from_url=_arcadia_url(new_tag),
                    ),
                )
                build_cube.add_requirement(main_entry)
                testing_entry.add_requirement(build_cube)
                graph.add(build_cube)

            # Upload static_to_s3

            upload_static_to_s3 = jg_cube.Cube(
                name=_cube_name(BUILD_STAGE, "upload_static_to_s3"),
                task="projects/irt/upload_sb_resource_to_s3",
                input=jg_cube.CubeInput(
                    tar_resource=_get_build_artifact(graph, BANNERLAND_WEB_STATIC),
                    prefix="${context.version_info.major}-${not_null(context.version_info.minor, `0`)}",
                    s3_bucket="bannerland-static",
                    access_key_id="sec-01g2wnpwtcvjjhxv4ykjyhaya2#access_key_id",
                    access_secret_key="sec-01g2wnpwtcvjjhxv4ykjyhaya2#access_secret_key",
                ),
                attributes={
                    "requirements": {
                        "sandbox": {
                            "tasks_resource": "3105306458",
                        },
                    },
                },
            )

            upload_static_to_s3.add_requirement(_get_build_cube(graph, BANNERLAND_WEB_STATIC))
            testing_entry.add_requirement(upload_static_to_s3)
            graph.add(upload_static_to_s3)

            # Test

            run_integration_test = jg_cube.Cube(
                name=_cube_name(rm_const.ReleaseStatus.testing, "integration_api_test"),
                task="projects/irt/bl_web_integration_api_test",
                input=jg_cube.CubeInput(
                    binary_executor_release_type=rm_const.ReleaseStatus.testing,
                    target_major_version="${context.version_info.major}",
                    target_minor_version="${not_null(context.version_info.minor, `0`)}",
                    yenv_type=rm_const.ReleaseStatus.testing,
                ),
                attributes={
                    "requirements": {
                        "sandbox": {
                            "tasks_resource": jg_cube.CubeInput.format_cube_output_value(
                                _get_build_artifact(graph, BANNER_LAND_WEB_API_TEST, SANDBOX_TASKS_BINARY),
                            ),
                        },
                    },
                },
            )

            run_integration_test.add_requirement(_get_build_cube(graph, BANNER_LAND_WEB_API_TEST))
            prestable_release_entry.add_requirement(run_integration_test)
            graph.add(run_integration_test)

            # Release

            for release_type in RELEASED_TYPES:
                for sandbox_task_type in SANDBOX_TASKS:
                    sb_release_cube = release_cubes.ReleaseRmComponent2(
                        name=_cube_name(release_type, sandbox_task_type),
                        title="Release SB task {} to {}".format(sandbox_task_type, release_type),
                        manual=False,
                        where_to_release=release_type,
                        component_name=self.component_name,
                        needs=[release_stage_entry[release_type]],
                        input=jg_cube.CubeInput(
                            component_resources={
                                sandbox_task_type.lower(): _get_build_artifact(graph, sandbox_task_type, SANDBOX_TASKS_BINARY),
                            },
                            deploy_system=rm_const.DeploySystem.kosher_sandbox_release.name,
                            wait_for_deploy=True,
                            wait_for_deploy_time_sec=300,
                        ),
                    )
                    sb_release_cube.add_requirement(_get_build_cube(graph, sandbox_task_type))
                    graph.add(sb_release_cube)
                    if RELEASED_TYPES[release_type] is not None:
                        release_stage_entry[RELEASED_TYPES[release_type]].add_requirement(sb_release_cube)

                run_migrate_cube = jg_cube.Cube(
                    name=_cube_name(release_type, "migrate"),
                    task="projects/irt/bannerland_web_database_migrate",
                    input=jg_cube.CubeInput(
                        yenv_type=release_type,
                        binary_executor_release_type=release_type,
                    ),
                    needs=[release_stage_entry[release_type]],
                    attributes={
                        "requirements": {
                            "sandbox": {
                                "tasks_resource": jg_cube.CubeInput.format_cube_output_value(
                                    _get_build_artifact(graph, BANNER_LAND_WEB_DATABASE_MIGRATE, SANDBOX_TASKS_BINARY)
                                ),
                            },
                        },
                    },
                )
                if RELEASED_TYPES[release_type] is not None:
                    release_stage_entry[RELEASED_TYPES[release_type]].add_requirement(run_migrate_cube)
                graph.add(run_migrate_cube)

                for release_info in YA_MAKE_TASKS:
                    release_cube = release_cubes.ReleaseRmComponent2(
                        name=_cube_name(release_type, release_info.resource_type),
                        title="Release {} to {}".format(release_info.resource_type, release_type),
                        manual=False,
                        where_to_release=release_type,
                        component_name=self.component_name,
                        needs=[release_stage_entry[release_type]],
                        input=jg_cube.CubeInput(
                            component_resources={
                                release_info.resource_type.lower(): _get_build_artifact(graph, release_info.resource_type),
                            },
                            deploy_system=release_info.deploy_system.name,
                            wait_for_deploy=True,
                            wait_for_deploy_time_sec=300,
                        ),
                    )
                    release_cube.add_requirement(_get_build_cube(graph, release_info.resource_type))
                    graph.add(release_cube)

                    if RELEASED_TYPES[release_type] is not None:
                        release_stage_entry[RELEASED_TYPES[release_type]].add_requirement(release_cube)

                # Migrate require release of static and migrate sb task
                run_migrate_cube.add_requirement(graph.get(_cube_name(release_type, BANNER_LAND_WEB_DATABASE_MIGRATE)))
                run_migrate_cube.add_requirement(_get_build_cube(graph, BANNER_LAND_WEB_DATABASE_MIGRATE))
                run_migrate_cube.add_requirement(graph.get(_cube_name(release_type, BANNERLAND_WEB_STATIC)))

                # Release binary only after migration
                graph.get(_cube_name(release_type, BANNERLAND_WEB_BACKEND_BIN)).add_requirement(run_migrate_cube)

            for release_info in YA_MAKE_TASKS:
                run_integration_test.add_requirement(graph.get(_cube_name(rm_const.ReleaseStatus.testing, release_info.resource_type)))

            for sandbox_task_type in SANDBOX_TASKS:
                run_integration_test.add_requirement(graph.get(_cube_name(rm_const.ReleaseStatus.testing, sandbox_task_type)))

            return graph

    class CI(configs.ReferenceCIConfig.CI):
        a_yaml_dir = PROJECT_PATH
        secret = "sec-01desry8fbgvnkbeybem81ferv"
        sb_owner_group = "BANNERLAND"

        ya_make_abs_paths_glob = [
            get_path("**"),
        ]

    class Releases(configs.ReferenceCIConfig.Releases):
        main_release_flow_independent_stages = True
        main_release_flow_branch_auto_start = True

        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name=release_info.resource_type.lower(),
                    data=ri.SandboxResourceData(release_info.resource_type, ttl=30),
                    deploy_infos=_get_deploy_info(release_info.deploy_system, RELEASED_TYPES),
                ) for release_info in YA_MAKE_TASKS
            ] + [
                ri.ReleasableItem(
                    name=sandbox_task_type.lower(),
                    data=ri.SandboxResourceData(SANDBOX_TASKS_BINARY, ttl=30),
                    deploy_infos=_get_deploy_info(rm_const.DeploySystem.kosher_sandbox_release, RELEASED_TYPES),
                ) for sandbox_task_type in SANDBOX_TASKS
            ]

        allow_old_releases = True
        allow_robots_to_release_stable = True

    class Notify(configs.ReferenceCIConfig.Notify):
        class Startrek(configs.ReferenceCIConfig.Notify.Startrek):
            assignee = "robot-bannerland"
            queue = "IRTRELEASE"
            ticket_type = "Release"
            summary_template = "BannerLand Web Release {}"
            workflow = rm_const.Workflow.BETA_TEST
            add_commiters_as_followers = False
            followers = []

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = None
        dirs = [
            PROJECT_PATH,
        ]
