# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.common import constants as sandbox_constants
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils

JUPITER_BUNDLES = [
    "yandex_bundle",
    "dessert_bundle",
    "shards_prepare_bundle",
    "gemini_saas_bundle",
    "rt_yandex_bundle",
    "canonized_factors_bundle",
    "externaldat_host_bundle",
    "hostdat_bundle",
    "yandex_deploy_bundle"
]


class JupiterCfg(configs.ReferenceBranchedConfig):
    name = "jupiter"
    responsible = "niknik"

    class ReleasePolicy:
        forced = "forced"
        regular = "regular"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_task_owner = "JUPITER"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            RELEASED_TTL_DAYS = "100"

            @property
            def _branch_part(self):

                def add_jupiter_branch_part(branch_part):
                    branch_part.append(
                        jg_build.JobGraphElementYaMakeBuildBranched(
                            task_name="JUPITER_YA_MAKE",
                            job_params={
                                "apiargs": {
                                    "requirements": {
                                        "disk_space": 10 << 30,
                                    },
                                },
                                "job_name_parameter": "JUPITER_CMPY",
                            },
                            ctx={
                                "targets": "robot/jupiter/packages/cmpy",
                                "arts": "robot/jupiter/packages/cmpy",
                                "arts_source": "",
                                "result_rt": "JUPITER_CMPY",
                                "result_rd": "Jupiter cmpy",
                                "result_single_file": True,
                                sandbox_constants.DEFINITION_FLAGS_KEY: "-DDEBUGINFO_LINES_ONLY=yes",
                                sandbox_constants.CHECKOUT_MODE: sandbox_constants.CHECKOUT_MODE_AUTO,
                                sandbox_constants.CHECKOUT: True,
                                sandbox_constants.CLEAR_BUILD_KEY: False,
                                sandbox_constants.CHECK_RETURN_CODE: True,
                                sandbox_constants.TESTS_REQUESTED: False,
                                sandbox_constants.YA_YT_DIR: "//home/jupiter/yamake_cache",
                                sandbox_constants.YA_YT_PROXY: "arnold.yt.yandex.net",
                                sandbox_constants.YA_YT_PUT: True,
                                sandbox_constants.YA_YT_STORE: True,
                                sandbox_constants.YA_YT_TOKEN_VAULT_NAME: "jupiter_yt_token",
                                sandbox_constants.YA_YT_TOKEN_VAULT_OWNER: "JUPITER",
                                "result_released_ttl": self.RELEASED_TTL_DAYS,
                            },
                            out={"JUPITER_CMPY": 30},
                        )
                    )
                    branch_part.append(
                        jg_build.JobGraphElementBuildPackageBranched(
                            task_name="KOSHER_YA_PACKAGE",
                            job_params={
                                "apiargs": {
                                    "requirements": {
                                        "disk_space": 50 << 30,  # 50 Gb
                                    },
                                },
                                "job_name_parameter": "JUPITER_EXPORTED_PACKAGES",
                            },
                            package_names=";".join([
                                "robot/jupiter/packages/canonizer/pkg.json",
                                "robot/jupiter/packages/saas_shard_deploy_bundle/pkg.json",
                                "robot/jupiter/packages/shard_deploy_bundle/pkg.json",
                            ]),
                            resource_names=";".join([
                                "JUPITER_CANONIZER",
                                "JUPITER_SAAS_SHARD_DEPLOY_BUNDLE",
                                "JUPITER_BUNDLE",
                            ]),
                            use_compression=False,
                            ctx={
                                sandbox_constants.YA_YT_DIR: "//home/jupiter/yamake_cache",
                                sandbox_constants.YA_YT_PROXY: "arnold.yt.yandex.net",
                                sandbox_constants.YA_YT_PUT: True,
                                sandbox_constants.YA_YT_STORE: True,
                                sandbox_constants.YA_YT_TOKEN_VAULT_NAME: "jupiter_yt_token",
                                sandbox_constants.YA_YT_TOKEN_VAULT_OWNER: "JUPITER",
                                # TODO(ovknp)(JUPITER-1700): uncomment this (and 2 more) when RMDEV-2458 is finished
                                # "package_released_resource_ttl": self.RELEASED_TTL_DAYS,
                            },
                            out={
                                "JUPITER_BUNDLE": 30,
                                "JUPITER_CANONIZER": 30,
                                "JUPITER_SAAS_SHARD_DEPLOY_BUNDLE": 30,
                            },
                        )
                    )
                    branch_part.append(
                        jg_build.JobGraphElementBuildPackageBranched(
                            task_name="KOSHER_YA_PACKAGE",
                            job_params={
                                "apiargs": {
                                    "requirements": {
                                        "disk_space": 150 << 30,  # 150 Gb
                                    },
                                },
                                "job_name_parameter": "JUPITER_CM_BUNDLES",
                            },
                            package_names=";".join([
                                "robot/jupiter/packages/{}/pkg.json".format(bundle) for bundle in JUPITER_BUNDLES
                            ]),
                            resource_names=";".join([
                                "JUPITER_{}".format(bundle.upper()) for bundle in JUPITER_BUNDLES
                            ]),
                            use_compression=False,
                            ctx={
                                sandbox_constants.YA_YT_DIR: "//home/jupiter/yamake_cache",
                                sandbox_constants.YA_YT_PROXY: "arnold.yt.yandex.net",
                                sandbox_constants.YA_YT_PUT: True,
                                sandbox_constants.YA_YT_STORE: True,
                                sandbox_constants.YA_YT_TOKEN_VAULT_NAME: "jupiter_yt_token",
                                sandbox_constants.YA_YT_TOKEN_VAULT_OWNER: "JUPITER",
                                "package_type": "tarball",
                                "raw_package": True,
                                # "package_released_resource_ttl": self.RELEASED_TTL_DAYS,
                            },
                            out={
                                "JUPITER_{}".format(bundle.upper()): 30 for bundle in JUPITER_BUNDLES
                            },
                        )
                    )

                def add_mercury_branch_part(branch_part):
                    build_params = [
                        (
                            "binaries",
                            50 << 30,  # 50 Gb
                            "union",
                            "robot/mercury/packages/binaries",
                        ),
                        (
                            "configs",
                            2 << 30,  # 2 Gb
                            "union",
                            "robot/mercury/packages/configs",
                        ),
                        (
                            "cmpy",
                            10 << 30,  # 10 Gb
                            "union",
                            "robot/mercury/packages/cmpy",
                        ),
                        (
                            "worker",
                            50 << 30,  # 50 Gb
                            "executable",
                            "robot/mercury/packages/worker/worker",
                        ),
                        (
                            "worker_configs",
                            2 << 30,  # 2 Gb
                            "configs",
                            "robot/mercury/packages/worker_configs/config",
                        ),
                        (
                            "subscriber",
                            50 << 30,  # 50 Gb
                            "executable",
                            "robot/mercury/packages/subscriber/subscriber",
                        ),
                        (
                            "shuffler",
                            50 << 30,  # 50 Gb
                            "executable",
                            "robot/mercury/packages/shuffler/shuffler",
                        ),
                        (
                            "rtdups",
                            50 << 30,  # 50 Gb
                            "executable",
                            "robot/mercury/packages/rtdups/rtdups",
                        ),
                        (
                            "logfetcher",
                            50 << 30,  # 50 Gb
                            "executable",
                            "robot/mercury/packages/logfetcher/logfetcher",
                        ),
                        (
                            "recuperator",
                            50 << 30,  # 50 Gb
                            "executable",
                            "robot/mercury/packages/recuperator/recuperator",
                        ),
                    ]
                    for target, disk_space, art_type, arts in build_params:
                        resource_type = "MERCURY_{}".format(target.upper())
                        branch_part.append(
                            jg_build.JobGraphElementYaMakeBuildBranched(
                                task_name="JUPITER_YA_MAKE",
                                job_params={
                                    "apiargs": {
                                        "requirements": {
                                            "disk_space": disk_space,
                                        },
                                    },
                                    "job_name_parameter": "MERCURY_{}".format(target),
                                    "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                                },
                                ctx={
                                    "targets": "robot/mercury/packages/{}".format(target),
                                    "arts": arts,
                                    "arts_source": "",
                                    "result_rt": resource_type,
                                    "result_rd": "Mercury {} bundle".format(target),
                                    "result_single_file": True,
                                    sandbox_constants.DEFINITION_FLAGS_KEY: "-DDEBUGINFO_LINES_ONLY=yes",
                                    sandbox_constants.CHECKOUT_MODE: sandbox_constants.CHECKOUT_MODE_AUTO,
                                    sandbox_constants.CHECKOUT: True,
                                    sandbox_constants.CLEAR_BUILD_KEY: False,
                                    sandbox_constants.CHECK_RETURN_CODE: True,
                                    sandbox_constants.TESTS_REQUESTED: False,
                                    sandbox_constants.YA_YT_DIR: "//home/jupiter/yamake_cache",
                                    sandbox_constants.YA_YT_PROXY: "arnold.yt.yandex.net",
                                    sandbox_constants.YA_YT_PUT: True,
                                    sandbox_constants.YA_YT_STORE: True,
                                    sandbox_constants.YA_YT_TOKEN_VAULT_NAME: "jupiter_yt_token",
                                    sandbox_constants.YA_YT_TOKEN_VAULT_OWNER: "JUPITER",
                                    "result_released_ttl": self.RELEASED_TTL_DAYS,
                                },
                                out={resource_type: 30},
                            )
                        )
                    branch_part.append(
                        jg_build.JobGraphElementBuildPackageBranched(
                            task_name="KOSHER_YA_PACKAGE",
                            job_params={
                                "job_name_parameter": "MERCURY_EXPORTED_PACKAGES",
                                "apiargs": {
                                    "requirements": {
                                        "disk_space": 50 << 30,  # 50 Gb
                                    },
                                },
                                "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            },
                            use_compression=False,
                            ctx={
                                sandbox_constants.YA_YT_DIR: "//home/jupiter/yamake_cache",
                                sandbox_constants.YA_YT_PROXY: "arnold.yt.yandex.net",
                                sandbox_constants.YA_YT_PUT: True,
                                sandbox_constants.YA_YT_STORE: True,
                                sandbox_constants.YA_YT_TOKEN_VAULT_NAME: "jupiter_yt_token",
                                sandbox_constants.YA_YT_TOKEN_VAULT_OWNER: "JUPITER",
                                # "package_released_resource_ttl": self.RELEASED_TTL_DAYS,
                            },
                            resource_names="MERCURY_SHARD_DEPLOY_BUNDLE",
                            package_names="robot/jupiter/packages/shard_deploy_bundle/pkg.json",
                            out={"MERCURY_SHARD_DEPLOY_BUNDLE": 30},
                        )
                    )

                def add_melter_branch_part(branch_part):
                    build_params = [
                        (
                            "worker",
                            50 << 30,  # 50 Gb
                            "executable",
                            "robot/melter/packages/worker/worker",
                        ),
                        (
                            "configs",
                            2 << 30,  # 2 Gb
                            "union",
                            "robot/melter/packages/configs",
                        ),
                    ]
                    for target, disk_space, art_type, arts in build_params:
                        resource_type = "MELTER_{}".format(target.upper())
                        branch_part.append(
                            jg_build.JobGraphElementYaMakeBuildBranched(
                                task_name="JUPITER_YA_MAKE",
                                job_params={
                                    "apiargs": {
                                        "requirements": {
                                            "disk_space": disk_space,
                                        },
                                    },
                                    "job_name_parameter": "MELTER_{}".format(target),
                                    "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                                },
                                ctx={
                                    "targets": "robot/melter/packages/{}".format(target),
                                    "arts": arts,
                                    "arts_source": "",
                                    "result_rt": resource_type,
                                    "result_rd": "Melter {} bundle".format(target),
                                    "result_single_file": True,
                                    sandbox_constants.DEFINITION_FLAGS_KEY: "-DDEBUGINFO_LINES_ONLY=yes",
                                    sandbox_constants.CHECKOUT_MODE: sandbox_constants.CHECKOUT_MODE_AUTO,
                                    sandbox_constants.CHECKOUT: True,
                                    sandbox_constants.CLEAR_BUILD_KEY: False,
                                    sandbox_constants.CHECK_RETURN_CODE: True,
                                    sandbox_constants.TESTS_REQUESTED: False,
                                    sandbox_constants.YA_YT_DIR: "//home/jupiter/yamake_cache",
                                    sandbox_constants.YA_YT_PROXY: "arnold.yt.yandex.net",
                                    sandbox_constants.YA_YT_PUT: True,
                                    sandbox_constants.YA_YT_STORE: True,
                                    sandbox_constants.YA_YT_TOKEN_VAULT_NAME: "jupiter_yt_token",
                                    sandbox_constants.YA_YT_TOKEN_VAULT_OWNER: "JUPITER",
                                    "result_released_ttl": self.RELEASED_TTL_DAYS,
                                },
                                out={resource_type: 30},
                            )
                        )

                branch_part = super(self.__class__, self)._branch_part
                add_jupiter_branch_part(branch_part)
                add_mercury_branch_part(branch_part)
                add_melter_branch_part(branch_part)
                return branch_part

            @property
            def _release(self):
                def update_ctx(default_ctx, additional_ctx):
                    default_ctx.update(additional_ctx)
                    return default_ctx

                def update_job_arrows(default_arrows, additional_arrows):
                    arrows = list(default_arrows)
                    arrows.extend(list(additional_arrows))
                    return arrows

                def generate_job_arrows(service_type, resource_names):
                    job_arrows = []
                    for res in resource_names:
                        resource_type = "{}_{}".format(service_type.upper(), res.upper())
                        job_arrows.append(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter=resource_type,
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "{}_{}_resource_id".format(service_type, res),
                                        resource_type,
                                    )
                                )
                            )
                        )
                    return job_arrows

                def generate_mercury_job_arrows(resource_names):
                    return generate_job_arrows("mercury", resource_names)

                def generate_melter_job_arrows(resource_names):
                    return generate_job_arrows("melter", resource_names)

                def add_jupiter_cm_releases(release_part):
                    JUPITER_YT_CONFIG = {
                        "stable": {
                            "yt_prefix": "//home/jupiter",
                            "yt_server": "arnold.yt.yandex.net",
                            "vault_yt_token": "jupiter_yt_token_2"
                        },
                        "prestable": {
                            "yt_prefix": "//home/jupiter-test/nightly",
                            "yt_server": "arnold.yt.yandex.net",
                            "vault_yt_token": "jupiter_night_yt_token"
                        },
                        "kwyt": {
                            "yt_prefix": "//home/kwyt-test/rthub_test",
                            "yt_server": "arnold.yt.yandex.net",
                            "vault_yt_token": "kwyt_test_yt_token"
                        },
                        "nightly2": {
                            "yt_prefix": "//home/jupiter-dev/nightly2",
                            "yt_server": "freud.yt.yandex.net",
                            "vault_yt_token": "jupiter_night_yt_token"
                        },
                        "nightly3": {
                            "yt_prefix": "//home/jupiter-dev/nightly3",
                            "yt_server": "socrates.yt.yandex.net",
                            "vault_yt_token": "jupiter_night_yt_token"
                        },
                    }

                    job_arrows_from_build = [
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="JUPITER_CMPY",
                            parent_job_data=(
                                jg_job_data.ParentDataDict(
                                    "component_resources",
                                    "jupiter_cmpy_resource_id",
                                    "JUPITER_CMPY",
                                ),
                            ),
                        ),
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="JUPITER_EXPORTED_PACKAGES",
                            parent_job_data=(
                                jg_job_data.ParentDataDict(
                                    "component_resources",
                                    "jupiter_canonizer_resource_id",
                                    "JUPITER_CANONIZER",
                                ),
                                jg_job_data.ParentDataDict(
                                    "component_resources",
                                    "jupiter_bundle_resource_id",
                                    "JUPITER_BUNDLE",
                                ),
                                jg_job_data.ParentDataDict(
                                    "component_resources",
                                    "jupiter_saas_shard_deploy_bundle_resource_id",
                                    "JUPITER_SAAS_SHARD_DEPLOY_BUNDLE",
                                ),
                            )
                        ),
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="JUPITER_CM_BUNDLES",
                            parent_job_data=tuple(
                                jg_job_data.ParentDataDict(
                                    "component_resources",
                                    "jupiter_{}_resource_id".format(bundle),
                                    "JUPITER_{}".format(bundle.upper()),
                                ) for bundle in JUPITER_BUNDLES
                            )
                        ),
                    ]

                    for release_type in JUPITER_YT_CONFIG.keys():
                        for scheduling_policy in ["forced", "regular"]:
                            release_part.append(
                                jg_release.JobGraphElementReleaseBranched(
                                    task_name="JUPITER_WRITE_RELEASE_META",
                                    release_to=release_type,
                                    release_item="JUPITER_CM_{}".format(scheduling_policy),
                                    job_arrows=update_job_arrows(
                                        (
                                            jg_arrows.ParentsData(
                                                input_key="tag",
                                                triggers=(
                                                    jg_job_triggers.JobTriggerNewTag(
                                                        parent_job_data=(
                                                            jg_job_data.ParentDataOutput(
                                                                input_key="branch_number_for_tag",
                                                                output_key="branch_number_for_tag",
                                                            ),
                                                        )
                                                    ),
                                                    jg_job_triggers.JobTriggerNewTag(
                                                        parent_job_data=(
                                                            jg_job_data.ParentDataOutput(
                                                                input_key="new_tag_number",
                                                                output_key="new_tag_number",
                                                            ),
                                                        ),
                                                    ),
                                                ),
                                                transform=lambda branch_number_for_tag, new_tag_number: "jupiter/stable-{}-{}".format(
                                                    branch_number_for_tag,
                                                    new_tag_number,
                                                ),
                                            ),
                                            jg_job_triggers.JobTriggerNewTag(
                                                parent_job_data=(
                                                    jg_job_data.ParentDataOutput(
                                                        input_key="branch_number_for_tag",
                                                        output_key="branch_number_for_tag",
                                                        transform=lambda x, params: str(x),
                                                    ),
                                                    jg_job_data.ParentDataOutput(
                                                        input_key="new_tag_number",
                                                        output_key="new_tag_number",
                                                        transform=lambda x, params: str(x),
                                                    )
                                                )
                                            )
                                        ),
                                        job_arrows_from_build,
                                    ),
                                    job_params={
                                        "ctx": update_ctx(
                                            {"scheduling_policy": scheduling_policy},
                                            JUPITER_YT_CONFIG[release_type],
                                        ),
                                    },
                                )
                            )
                    for release_stage in [rm_const.ReleaseStatus.stable, rm_const.ReleaseStatus.prestable]:
                        for scheduling_policy in ["forced", "regular"]:
                            release_part.append(
                                jg_release.JobGraphElementActionReleaseBranched(
                                    release_to=release_stage,
                                    release_item="JUPITER_CM_{}".format(scheduling_policy),
                                    job_arrows=(
                                        jg_job_triggers.JobTriggerRelease(
                                            job_name_parameter="JUPITER_CM_{}__KWYT".format(scheduling_policy),
                                        ) if release_stage == rm_const.ReleaseStatus.stable else None,
                                        jg_job_triggers.JobTriggerRelease(
                                            job_name_parameter="JUPITER_CM_{}__NIGHTLY2".format(scheduling_policy),
                                        ) if release_stage == rm_const.ReleaseStatus.stable else None,
                                        jg_job_triggers.JobTriggerRelease(
                                            job_name_parameter="JUPITER_CM_{}__NIGHTLY3".format(scheduling_policy),
                                        ) if release_stage == rm_const.ReleaseStatus.stable else None,
                                    ),
                                )
                            )

                def add_mercury_cm_releases(release_part):
                    MERCURY_YT_CONFIG = {
                        "stable": {
                            "yt_prefix": "//home/callisto/mercury",
                            "yt_server": "arnold.yt.yandex.net",
                            "vault_yt_token": "callisto_yt_token"
                        },
                        "backup": {
                            "yt_prefix": "//home/callisto",
                            "yt_server": "hahn.yt.yandex.net",
                            "vault_yt_token": "callisto_yt_token"
                        },
                        "prestable": {
                            "yt_prefix": "//home/jupiter-test/callisto_nightly",
                            "yt_server": "arnold.yt.yandex.net",
                            "vault_yt_token": "callisto_nightly_yt_token"
                        },
                        "beta": {
                            "yt_prefix": "//home/jupiter-test/callisto_jupiter_beta",
                            "yt_server": "arnold.yt.yandex.net",
                            "vault_yt_token": "callisto_nightly_yt_token"
                        },
                    }

                    cm_release_job_arrows = generate_mercury_job_arrows(["binaries", "cmpy", "configs"])
                    cm_release_job_arrows.append(
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="MERCURY_EXPORTED_PACKAGES",
                            parent_job_data=(
                                jg_job_data.ParentDataDict(
                                    "component_resources",
                                    "mercury_shard_deploy_bundle_resource_id",
                                    "MERCURY_SHARD_DEPLOY_BUNDLE",
                                ),
                            ),
                        )
                    )

                    for release_target in MERCURY_YT_CONFIG.keys():
                        for scheduling_policy in ["forced", "regular"]:
                            release_part.append(
                                jg_release.JobGraphElementReleaseBranched(
                                    task_name="JUPITER_WRITE_RELEASE_META",
                                    release_to=release_target,
                                    release_item="MERCURY_CM_{}".format(scheduling_policy),
                                    job_arrows=update_job_arrows(
                                        (
                                            jg_arrows.ParentsData(
                                                input_key="tag",
                                                triggers=(
                                                    jg_job_triggers.JobTriggerNewTag(
                                                        parent_job_data=(
                                                            jg_job_data.ParentDataOutput(
                                                                input_key="branch_number_for_tag",
                                                                output_key="branch_number_for_tag",
                                                            ),
                                                        )
                                                    ),
                                                    jg_job_triggers.JobTriggerNewTag(
                                                        parent_job_data=(
                                                            jg_job_data.ParentDataOutput(
                                                                input_key="new_tag_number",
                                                                output_key="new_tag_number",
                                                            ),
                                                        ),
                                                    ),
                                                ),
                                                transform=lambda branch_number_for_tag, new_tag_number: "jupiter/stable-{}-{}".format(
                                                    branch_number_for_tag,
                                                    new_tag_number,
                                                ),
                                            ),
                                            jg_job_triggers.JobTriggerNewTag(
                                                parent_job_data=(
                                                    jg_job_data.ParentDataOutput(
                                                        input_key="branch_number_for_tag",
                                                        output_key="branch_number_for_tag",
                                                        transform=lambda x, params: str(x),
                                                    ),
                                                    jg_job_data.ParentDataOutput(
                                                        input_key="new_tag_number",
                                                        output_key="new_tag_number",
                                                        transform=lambda x, params: str(x),
                                                    )
                                                )
                                            )
                                        ),
                                        cm_release_job_arrows,
                                    ),
                                    job_params={
                                        "ctx": update_ctx(
                                            {"scheduling_policy": scheduling_policy},
                                            MERCURY_YT_CONFIG[release_target],
                                        ),
                                    },
                                )
                            )

                    for release_stage in [rm_const.ReleaseStatus.stable, rm_const.ReleaseStatus.prestable]:
                        for scheduling_policy in ["forced", "regular"]:
                            release_part.append(
                                jg_release.JobGraphElementActionReleaseBranched(
                                    release_to=release_stage,
                                    release_item="MERCURY_CM_{}".format(scheduling_policy),
                                    job_arrows=(
                                        jg_job_triggers.JobTriggerRelease(
                                            job_name_parameter="MERCURY_CM_{}__BACKUP".format(scheduling_policy),
                                        ) if release_stage == rm_const.ReleaseStatus.stable else None,
                                        jg_job_triggers.JobTriggerRelease(
                                            job_name_parameter="MERCURY_CM_{}__BETA".format(scheduling_policy),
                                        ) if release_stage == rm_const.ReleaseStatus.prestable else None,
                                    ),
                                )
                            )

                def add_workers_releases(release_part):
                    release_job_arrows = generate_mercury_job_arrows([
                        "logfetcher",
                        "recuperator",
                        "rtdups",
                        "subscriber",
                        "shuffler",
                        "worker",
                        "worker_configs",
                    ]) + generate_melter_job_arrows([
                        "worker",
                        "configs",
                    ])
                    for release_stage in [
                        rm_const.ReleaseStatus.testing,
                        rm_const.ReleaseStatus.prestable,
                        rm_const.ReleaseStatus.stable,
                    ]:
                        release_part.append(
                            jg_release.JobGraphElementReleaseBranched(
                                release_to=release_stage,
                                release_item="WORKERS",
                                job_arrows=release_job_arrows,
                            )
                        )
                    deploy_params = [
                        (
                            rm_const.ReleaseStatus.testing,
                            "staging_deploy",
                            "robot-mercury-knight-nanny-token",
                            "WORKERS_TESTING",
                            (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        ),
                        (
                            rm_const.ReleaseStatus.prestable,
                            "prestable_deploy",
                            "robot-mercury-knight-nanny-token",
                            "WORKERS_PRESTABLE",
                            (jg_utils.TestFrequency.LAZY, None),
                        ),
                        (
                            rm_const.ReleaseStatus.stable,
                            "main_first",
                            "robot-mercury-nanny-token",
                            "WORKERS_STABLE",
                            (jg_utils.TestFrequency.LAZY, None),
                        ),
                    ]
                    for release_stage, recipe, vault_id, job_name_parameter, frequency in deploy_params:
                        release_part.append(
                            jg_release.JobGraphElementActionReleaseBranched(
                                release_to=release_stage,
                                release_item="WORKERS",
                                job_params={
                                    "task_name": "DEPLOY_NANNY_DASHBOARD_RECIPE",
                                    "job_name_parameter": job_name_parameter,
                                    "ctx": {
                                        "dashboard_id": "mercury",
                                        "nanny_token_vault_owner": "JUPITER",
                                        "dashboard_recipe_id": recipe,
                                        "nanny_token_vault_id": vault_id,
                                    },
                                    "frequency": frequency,
                                },
                            )
                        )

                release_part = super(self.__class__, self)._release
                add_jupiter_cm_releases(release_part)
                add_workers_releases(release_part)
                add_mercury_cm_releases(release_part)
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def deactivate(self):
                return [
                    rm_const.JobTypes.rm_job_name(
                        rm_const.JobTypes.ACTION_RELEASE,
                        rm_const.RMNames.JUPITER,
                        "MERCURY_TESTING"
                    ),
                ]  # jobs to deactivate on db clone

            @property
            def change_frequency(self):
                uppercase_name = self.name.upper()
                return {
                    "_LOG_MERGE__{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    rm_const.JobTypes.rm_job_name(
                        rm_const.JobTypes.ACTION_RELEASE,
                        rm_const.RMNames.JUPITER,
                        "MERCURY_TESTING"): rm_const.TestFrequencies.EACH_REV_TEST,
                }

            @property
            def ignore_match(self):
                """List of test names which should be ignored in branch db"""
                uppercase_name = self.name.upper()
                return super(self.__class__, self).ignore_match + [
                    "{}_DELIVERY_BUILD_CLUSTERMASTER_CONFIGS".format(uppercase_name),
                    "{}_PRODUCTION_BUILD_CLUSTERMASTER_CONFIGS".format(uppercase_name),
                    "{}_BUILD_CLUSTERMASTER_CONFIGS".format(uppercase_name),
                    "{}_NIGHTLY_BUILD_CLUSTERMASTER_CONFIGS".format(uppercase_name),
                ]

    class Releases(configs.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        resources_info = [
            configs.ReleasedResourceInfo(
                name="jupiter_{}_resource_id".format(name),
                resource_type="JUPITER_{}".format(name.upper()),
                deploy=[
                    configs.DeployServicesInfo(
                        services=["jupiter_cm_production"],
                        level=rm_const.ReleaseStatus.stable,
                    ),
                    configs.DeployServicesInfo(
                        services=["jupiter_cm_nightly"],
                        level=rm_const.ReleaseStatus.prestable,
                    ),
                ] if name == "yandex_bundle" else None,
            ) for name in [
                "cmpy",
                "canonizer",
                "bundle",
                "saas_shard_deploy_bundle"
            ] + JUPITER_BUNDLES
        ] + [
            configs.ReleasedResourceInfo(
                name="mercury_{}_resource_id".format(name),
                resource_type="MERCURY_{}".format(name.upper()),
                deploy=[
                    configs.DeployServicesInfo(
                        services=["mercury_worker_vla"],
                        level=rm_const.ReleaseStatus.stable,
                    ),
                    configs.DeployServicesInfo(
                        services=["mercury_nightly_worker_vla"],
                        level=rm_const.ReleaseStatus.prestable,
                    ),
                    configs.DeployServicesInfo(
                        services=["mercury_staging_worker_man"],
                        level=rm_const.ReleaseStatus.testing,
                    ),
                ] if name == "worker" else None,
            ) for name in [
                "cmpy",
                "binaries",
                "configs",
                "shard_deploy_bundle",
                "worker",
                "worker_configs",
                "subscriber",
                "shuffler",
                "recuperator",
                "rtdups",
                "logfetcher",
            ]
        ] + [
            configs.ReleasedResourceInfo(
                name="melter_{}_resource_id".format(name),
                resource_type="MELTER_{}".format(name.upper()),
                deploy=[
                    configs.DeployServicesInfo(
                        services=["melter_worker_vla"],
                        level=rm_const.ReleaseStatus.stable,
                    ),
                    configs.DeployServicesInfo(
                        services=["callisto_melter_prestable"],
                        level=rm_const.ReleaseStatus.prestable,
                    ),
                ],
            ) for name in [
                "configs",
                "worker",
            ]
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""

        notifications = [
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="Jupiter",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="MergeCommit",
                chat_name="Jupiter",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="Jupiter",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.STARTREK_TICKET_CREATED_CONDITIONS,
                    join_strategy="OR",
                ),
                message_template_file="notifications/ticket_created.html",
            ),
        ]

        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            """Mail notifications configuration"""
            mailing_list = [
                "search-components-releases@yandex-team.ru",
                "jupiter@yandex-team.ru",
            ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = ["jupiter_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "niknik"
            queue = "JUPITERRELEASE"
            dev_queue = "JUPITER"
            summary_template = u"Jupiter {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = []
            add_commiters_as_followers = True
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "robot/jupiter/changelogs/"
        dirs = [
            "arcadia/robot/jupiter",
            "arcadia/robot/library/oxygen",
            "arcadia/robot/library/yt/static",
            "arcadia/robot/mercury",
        ]
        review_groups = [
            "jupiter",
            "mercury",
        ]

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_prefix = "pre-stable"
