# -*- coding: utf-8 -*-
from sandbox.projects.common.constants import constants as sandbox_constants
from sandbox.projects.release_machine.components import configs as cfg
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
import sandbox.projects.release_machine.components.job_graph.presets as jg_presets
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


class LogisticDispatcherCfg(cfg.ReferenceBranchedConfig):
    name = "logistic_dispatcher"
    display_name = "Logistic dispatcher"
    responsible = "ivanmorozov"

    class Testenv(cfg.ReferenceBranchedConfig.Testenv):
        new_task_owner = "TAXI_LOGISTIC"
        trunk_task_owner = "TAXI_LOGISTIC"
        testenv_db_owners = [
            "ivanmorozov",
            "nsofya",
            "skulik",
        ]

        class JobGraph(cfg.ReferenceBranchedConfig.Testenv.JobGraph):
            OUT_RESOURCES_WITH_TTL = {"TAXI_DISPATCHER": 14}
            TARGET = "taxi/logistic-dispatcher/dispatcher"
            BUILD_PARAMS = {
                "arts": "{}/dispatcher".format(TARGET),
                "targets": "{}/dispatcher".format(TARGET),
                "result_single_file": True,
                "result_rt": "TAXI_DISPATCHER",
                "result_rd": "Taxi logistic dispatcher",
                sandbox_constants.USE_ARC_INSTEAD_OF_AAPI: True,
                sandbox_constants.DEFINITION_FLAGS_KEY: " ".join([
                    "-DUSE_PERL=no",
                    "-DUSE_WIZARD=no",
                    "-DNOLINK=yes",
                ])
            }

            def _create_trunk_build(self, build_type):
                build_params = {"build_type": build_type}
                build_params.update(self.BUILD_PARAMS)
                return jg_build.JobGraphElementBuildTrunk(
                    task_name="KOSHER_YA_MAKE",
                    build_item=build_type.upper(),
                    ctx=build_params,
                    job_params={
                        "apiargs": {
                            "kill_timeout": 3600,  # 1h
                        },
                        "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 10),
                    },
                    filter_targets=[self.TARGET],
                )

            @property
            def _trunk_part(self):
                return super(self.__class__, self)._trunk_part + [
                    self._create_trunk_build("release"),
                    self._create_trunk_build("debug"),
                ]

            _preset = jg_presets.SingleBuildPreset(
                build_task_name="BUILD_TAXI",
                build_ctx={
                    "build_taxi_dispatcher": True,
                    "clear_build": True,
                    "ymake_checkout_flag": True,
                    "ymake_flags": "-fno-omit-frame-pointer -Wno-narrowing -Wno-sign-compare;CXXFLAGS=-Wno-narrowing -Wno-overloaded-virtual -Wno-sign-compare;",
                    "ymake_parameters": "-DUSE_PERL=no -DUSE_WIZARD=no -DNOLINK=yes",
                },
                deploy_system=rm_const.DeploySystem.nanny.name,
                stages=[
                    jg_utils.StageReleaseFrequency(rm_const.ReleaseStatus.stable),
                    jg_utils.StageReleaseFrequency(rm_const.ReleaseStatus.prestable),
                    jg_utils.StageReleaseFrequency(rm_const.ReleaseStatus.testing),
                ],
            )

            @property
            def _branch_part(self):
                return super(self.__class__, self)._branch_part + [
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        build_item="KOSHER",
                        out=self.OUT_RESOURCES_WITH_TTL,
                        ctx=self.BUILD_PARAMS,
                    ),
                ]

    class Releases(cfg.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [ri.ReleasableItem(
                name="dispatcher",
                data=ri.SandboxResourceData("TAXI_DISPATCHER", ttl=14),
                deploy_infos=[
                    ri.NannyDeployInfo(
                        ri.DeployService("taxi_logistic-dispatcher_stable"),
                        stage=rm_const.ReleaseStatus.stable
                    ),
                    ri.NannyDeployInfo(
                        ri.DeployService("taxi_logistic-dispatcher_pre_stable"),
                        stage=rm_const.ReleaseStatus.prestable
                    ),
                ],
            )]

    class Notify(cfg.ReferenceBranchedConfig.Notify):
        class Startrek(cfg.ReferenceBranchedConfig.Notify.Startrek):
            """
            https://wiki.yandex-team.ru/releasemachine/startrekoptions/
            """
            assignee = "ivanmorozov"
            queue = "DISPATCH"
            summary_template = "Logistic dispatcher release {}"
            components = "logistic_dispatcher_release"
            followers = [
                "ivanmorozov",
                "nsofya",
                "skulik",
            ]
            ticket_type = "Release"
            tags = [
                "release-rm-task",
                "logistic-dispatcher",
            ]
            use_task_author_as_assignee = True
            hide_commits_under_cut = True
            notify_on_deploy_to_feature_st_tickets = True
            important_changes_limit = 1000

    class ChangelogCfg(cfg.ReferenceBranchedConfig.ChangelogCfg):
        """
        https://wiki.yandex-team.ru/releasemachine/changelogs/
        """
        observed_paths = [
            "arcadia/taxi/logistic-dispatcher/dispatcher",
            "arcadia/kernel/common_server",
        ]
        wiki_page = "Users/ivanmorozov/logistic-dispatcher/changelog/"
