# -*- coding: utf-8 -*-
import json

from sandbox.projects.release_machine.components import configs as cfg
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow
from sandbox.projects.release_machine.components.config_core.jg.preset import basic_build_presets
from sandbox.projects.release_machine.components.config_core.jg.cube import base as cube_base
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import run_command as run_command_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import build as build_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg.lib.conditions import ci_conditions
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.core import const as rm_const


MAX_BIN_SIZE = 1073741824
ENV_2_RELEASE_STATUS = {
    "prod": "stable",
    "preprod": "prestable",
}


def deploy_cube(build_cube, deploy_to, needs, build_binary=False, build_graph_doc=False, **kwargs):
    return cube_base.Cube(
        title="{} Logos release".format(deploy_to.upper()),
        name="deploy_logos_binary_{}".format(deploy_to),
        task="projects/logos/builder",
        input=cube_base.CubeInput(
            logos_secret="sec-01daxhrvak3kbcd6fxhydp3esp#secret",
            build_binary=build_binary,
            build_graph_doc=build_graph_doc,
            logos_binary_resource=build_cube.output.resources["LOGOS_BINARY"].first().id,
            udf_resource_id=build_cube.output.resources["LOGOS_MONO_NILE_UDF"].first().id,
            deploy_to=deploy_to,
            release_id="release_${context.target_revision.number}_v${context.version_info.full}",
            switch_release=True,
            update_after_switch=True,
        ),
        needs=needs,
        **kwargs
    )


def infra_create_cube(service_id, environment_id, needs, **kwargs):
    return cube_base.Cube(
        name="create_infra",
        title="Create infra event",
        task="common/monitoring/infra_create",
        input=cube_base.CubeInput(
            config={
                "status": {"severity": "MINOR", "type": "MAINTENANCE"},
                "placement": {"serviceId": service_id, "environmentId": environment_id},
                "textual": {
                    "title": "Logos | Release | release_${context.target_revision.number}_v${context.version_info.full}",
                    "description": "Release ${context.version_info.full} from revision ${context.target_revision.number}\nCI url: ${context.ci_url}"
                },
            },
        ),
        needs=needs,
        **kwargs
    )


def infra_close_cube(infra_create_cube, needs, **kwargs):
    return cube_base.Cube(
        name="close_infra",
        title="Close infra event",
        task="common/monitoring/infra_close",
        input=cube_base.CubeInput(
            config={},
            infra_identifier=infra_create_cube.output.infra_identifier,
        ),
        needs=needs,
        **kwargs
    )


def send_prod_notification_cube(needs, **kwargs):
    return cube_base.Cube(
        name="prod_notifications",
        title="Send notifications",
        task="projects/logos/notifications",
        input=cube_base.CubeInput(
            config={
                "team": "Logos",
                "chat_id": "logos-prod",
                "message": "[${context.title}](${context.ci_url}) **FINISHED**\nTriggered by [${context.flow_triggered_by}](https://staff.yandex-team.ru/${context.flow_triggered_by})",
            }
        ),
        needs=needs,
        **kwargs
    )


def build_graph_doc_cube(project, release_to, build_cube, needs, **kwargs):
    return cube_base.Cube(
        name="build_graph_doc_{}".format(release_to),
        title="Build graph doc",
        task="projects/logos/run_script",
        input=cube_base.CubeInput(
            cmdline="{{logos-binary}} --{release_to} --path latest.json".format(release_to=release_to),
            sb_resources={
                "logos_binary": build_cube.output.resources["LOGOS_BINARY"].first().id,
            },
            save_as_resource={
                "latest.json": json.dumps({
                    "type": "LOGOS_GRAPH_DOC",
                    "attrs": {
                        "ttl": 999,
                        "released": ENV_2_RELEASE_STATUS[release_to],
                        "arcadia_revision": "${context.target_revision.number}",
                        "release_name": ENV_2_RELEASE_STATUS[release_to],
                        "project": project,
                    }
                }),
            },
            env_vars={
                "AQINFRA_TOOL": "GRAPH_DOC",
            }
        ),
        needs=needs,
        attributes={
            "requirements": {
                "sandbox": {
                    "container_resource": "2581597439",
                }
            },
        },
        **kwargs
    )


def migrate_graph_doc_cube(release_to, needs, **kwargs):
    return cube_base.Cube(
        name="release_graph_doc_{}".format(release_to),
        title="Release {} graph doc to MDB".format(release_to),
        task="projects/logos/run_script",
        input=cube_base.CubeInput(
            cmdline="{resource}",
            resource_select="search",
            resource_search_conf={
                "state": "READY",
                "type": "LOGOS_VIEWY_MIGRATE_TOOL",
                "attrs": {"released": "stable"},
            },
            env_vars={
                "OAUTH_TOKEN": "$(yav:sec-01egg35h0qjbh0jc03ey7e6tpf:nirvana-secret)",
                "TVM_SECRET": "$(yav:sec-01g3xks72rmt95send404pej29:client_secret)",
                "POSTGRESQL_HOST": "c-mdb88758mi9e10ji44n6.rw.db.yandex.net",
                "POSTGRESQL_PASSWORD": "$(yav:sec-01e63wnvmr2dtve75fdhb083gf:logos-mdb-prod-password)",
            },
        ),
        needs=needs,
        **kwargs
    )


def check_binary_size_cube(needs, **kwargs):
    return run_command_cubes.RunCommand(
        name="check_binary_size",
        title="Check binary size",
        input=cube_base.CubeInput(
            config={
                "cmd_line": '''actualsize=$(wc -c < {logos_tool})
                               echo $actualsize > $RESULT_RESOURCES_PATH/binary_size
                               if [ $actualsize -ge {max_bin_size} ]; then
                                   echo "Binary size $actualsize is over {max_bin_size}"
                                   exit 1
                               else
                                   echo "Binary size $actualsize is OK"
                                   exit 0
                               fi'''.format(max_bin_size=MAX_BIN_SIZE, logos_tool="{logos_tool}"),
                "result_output": [{"path": "binary_size"}],
                "sandbox_resource_placeholders": [{"key": "logos", "resource_type": "LOGOS_BINARY"}]
            }
        ),
        needs=needs,
        **kwargs
    )


def run_arcadia_tests_cube(needs, target=None, test_emily_storage=False, test_emily_storage_transport=False, **kwargs):
    if not target:
        assert test_emily_storage or test_emily_storage_transport, "Please specify target to test"
        if test_emily_storage:
            target = "ads/emily/storage"
        else:
            target = "ads/emily/storage/transport"
    return build_cubes.YaMakeBuildCubeBase(
        name="test_{}".format(target.replace("/", "_")),
        targets=target,
        input=cube_base.CubeInput(
            test=True,
        ),
        needs=needs,
        **kwargs
    )


def wait_for_deadline_cube(needs, **kwargs):
    return cube_base.Cube(
        name="wait_for_deadline",
        title="Wait",
        task="projects/logos/wait",
        needs=needs,
        input=cube_base.CubeInput(
            config={
                "conditions": {
                    "wait_until": ["now().hour < 22 or now() >= now().replace(hour=23, minute=5)", ]
                }
            }
        ),
        **kwargs
    )


def release_hotfix_cube(build_cube, needs, release_status="stable", release_resource=True, **kwargs):
    return cube_base.Cube(
        name="monitorings_update",
        title="Force update monitorings",
        task="projects/logos/build_and_release",
        needs=needs,
        manual=True,
        input=cube_base.CubeInput(
            build_or_select="select",
            resource_id=build_cube.output.resources["LOGOS_BINARY"].first().id,
            update_resource_attrs={
                # Только для NileOverYQL
                "nile_udf_resource_id": build_cube.output.resources["LOGOS_MONO_NILE_UDF"].first().id,
            },
            release_resource=release_resource,
            release_status=release_status,
        ),
        **kwargs
    )


def release_config_factory(logos_project, responsible_abc, responsible_login):

    class LogosReleaseConfig(cfg.ReferenceCIConfig):
        project = logos_project
        name = "logos_{}".format(project)
        display_name = "Logos {} graph".format(project)
        responsible = cfg.Responsible(
            abc=cfg.Abc(service_name=responsible_abc),
            login=responsible_login,
        )

        class CI(cfg.ReferenceCIConfig.CI):
            a_yaml_dir = "logos/projects/{}".format(logos_project)
            secret = "sec-01desry8fbgvnkbeybem81ferv"
            sb_owner_group = "LOGOS"

        class Notify(cfg.ReferenceCIConfig.Notify):
            class Startrek(cfg.ReferenceCIConfig.Notify.Startrek):
                assignee = responsible_login
                queue = "DMPREL"
                summary_template = "Релиз Logos: граф {}".format(logos_project)
                add_commiters_as_followers = False
                use_task_author_as_assignee = False

        class ChangelogCfg(cfg.ReferenceCIConfig.ChangelogCfg):
            wiki_page = None
            dirs = [
                "arcadia/logos/projects/{}".format(logos_project),
            ]

    return LogosReleaseConfig


class BaseLogosJG(basic_build_presets.SingleBuildYaMakeJGCfg):
    build_task = "KOSHER_YA_MAKE"
    build_cube_name = "build_logos_project"
    manual_prod_release = True
    release_manually = False  # Do not change - it is about RELEASE_RM_COMPONENT_2 cubes
    infra_service_id = 0
    infra_environment_id = 0
    deploy_to = ["preprod", "prod"]
    monitorings_hotfix = False
    manual_skip_preprod = False
    preprod_just_for_first_tag = True
    build_graph_doc = True

    additional_preprod_cubes = []
    additional_testing_cubes = []

    @property
    def project(self):
        return self.root_cfg.project

    @jg_flow.release_flow(
        stages=[stage.set_rollback(False) for stage in basic_build_presets.DEFAULT_RELEASE_FLOW_STAGES]
    )
    def release(self):
        graph = super(BaseLogosJG, self).release(self)

        build_cube = graph.get("build_logos_project")
        release_stage_entry = graph.get("release_stage_entry")
        release_stable_cube = graph.get("release_stable_sandbox")
        release_prestable_cube = graph.get("release_prestable_sandbox")
        release_stable_cube.add_requirement(release_prestable_cube)

        if self.additional_testing_cubes:
            for cube in self.additional_testing_cubes:
                cube.add_requirement(build_cube)
                release_stage_entry.add_requirement(cube)
                graph.add(cube)

        _first_tag_conditions_kwargs = {}
        if self.preprod_just_for_first_tag:
            _first_tag_conditions_kwargs["condition"] = ci_conditions.CI_FIRST_TAG_ONLY

        if "preprod" in self.deploy_to:
            preprod_deploy_cube = deploy_cube(
                build_cube, "preprod",
                needs=[release_stage_entry, ],
                **_first_tag_conditions_kwargs
            )

            for cube in self.additional_preprod_cubes:
                cube.add_requirement(preprod_deploy_cube)
                graph.add(cube)

            graph.add(preprod_deploy_cube)

            if self.build_graph_doc:
                graph_doc_cube = build_graph_doc_cube(
                    self.project,
                    release_to="preprod",
                    build_cube=build_cube,
                    needs=[preprod_deploy_cube, ] + self.additional_preprod_cubes,
                )
                migrate_cube = migrate_graph_doc_cube("preprod", [graph_doc_cube, ])
                graph.add(graph_doc_cube)
                graph.add(migrate_cube)
                release_prestable_cube.add_requirement(migrate_cube)

            release_prestable_cube.add_requirement(preprod_deploy_cube)
            for cube in self.additional_preprod_cubes:
                release_prestable_cube.add_requirement(cube)

        if "prod" in self.deploy_to:
            prod_deploy_needs = [release_prestable_cube, ]

            if self.manual_skip_preprod:
                manual_skip_cube = dummy_cubes.Dummy(
                    title="☠Skip preprod☠",
                    name="skip_preprod",
                    manual=True,
                    needs=[release_stage_entry, ],
                )
                graph.add(manual_skip_cube)
                prod_deploy_needs.append(manual_skip_cube)

            _infra_create_cube = infra_create_cube(
                service_id=self.infra_service_id,
                environment_id=self.infra_environment_id,
                needs_type=cube_base.CubeNeedsType.ANY,
                needs=prod_deploy_needs,
                manual=self.manual_prod_release,
            )

            prod_deploy_cube = deploy_cube(
                build_cube, "prod",
                needs=[_infra_create_cube, ],
                **_first_tag_conditions_kwargs
            )

            if self.build_graph_doc:
                graph_doc_cube = build_graph_doc_cube(self.project, release_to="prod", build_cube=build_cube, needs=prod_deploy_needs)
                migrate_cube = migrate_graph_doc_cube("prod", [graph_doc_cube, ])
                graph.add(graph_doc_cube)
                graph.add(migrate_cube)
                release_stable_cube.add_requirement(migrate_cube)

            release_stable_cube.add_requirement(prod_deploy_cube)

            _infra_close_cube = infra_close_cube(
                _infra_create_cube,
                needs=[release_stable_cube, ]
            )

            graph.add(_infra_create_cube)
            graph.add(prod_deploy_cube)
            graph.add(_infra_close_cube)

        if self.monitorings_hotfix:
            monitorings_cube = release_hotfix_cube(build_cube, needs=[release_stage_entry])
            graph.add(monitorings_cube)

        return graph


class BaseLogosReleases(cfg.ReferenceCIConfig.Releases):
    allow_robots_to_release_stable = True
    deploy_system = rm_const.DeploySystem.sandbox
    logos_binary_name = "logos_tool"
    build_udf = True
    logos_udf_binary_name = "liblogos_nile_udf.so"

    @property
    def project(self):
        return self.root_cfg.project

    @property
    def releasable_items(self):
        items = [
            ri.ReleasableItem(
                name="logos_binary_{}".format(self.project),
                data=ri.SandboxResourceData("LOGOS_BINARY", ttl=360),
                build_data=ri.BuildData(
                    target="logos/projects/{project}/graph/bin".format(project=self.project),
                    artifact="logos/projects/{project}/graph/bin/{logos_bin}".format(
                        project=self.project,
                        logos_bin=self.logos_binary_name,
                    ),
                ),
                deploy_infos=[
                    ri.SandboxInfo(stage=rm_const.ReleaseStatus.prestable),
                    ri.SandboxInfo(stage=rm_const.ReleaseStatus.stable),
                ],
            ),
            ri.ReleasableItem(
                name="logos_udf_binary_{}".format(self.project),
                data=ri.SandboxResourceData("LOGOS_MONO_NILE_UDF", ttl=360),
                build_data=ri.BuildData(
                    target="logos/projects/{project}/graph/nile_udf".format(project=self.project),
                    artifact="logos/projects/{project}/graph/nile_udf/{udf_bin}".format(
                        project=self.project,
                        udf_bin=self.logos_udf_binary_name,
                    ),
                ),
                deploy_infos=[
                    ri.SandboxInfo(stage=rm_const.ReleaseStatus.prestable),
                    ri.SandboxInfo(stage=rm_const.ReleaseStatus.stable),
                ],
            ),
        ]

        if not self.build_udf:
            items = items[:1]

        return items
