# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.common import constants as sandbox_constants
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


class LyncherCfg(configs.ReferenceBranchedConfig):
    name = "lyncher"
    responsible = "granilace"

    class MergesCfg(configs.ReferenceBranchedConfig.MergesCfg):
        number_last_branches_to_merge = 1

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_db = "modadvert-lyncher-trunk"
        trunk_task_owner = "MODADVERT_RELEASE"
        branch_db_template = "modadvert-lyncher-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="MODADVERT_BUILD_TASK",
                        ctx={
                            "packages": ";".join([
                                "modadvert/programs/lyncher/package.json",
                                "modadvert/programs/qlyncher/docker_package.json",
                                "modadvert/programs/lyncher/config_package.json",
                                "modadvert/programs/offline_lyncher/package.json",
                            ]),
                            "resource_type": ";".join([
                                "MODADVERT_LYNCHER_BINARIES",
                                "MODADVERT_QLYNCHER_BINARIES",
                                "MODADVERT_LYNCHER_CONFIG",
                                "MODADVERT_OFFLINE_LYNCHER_BINARIES",
                            ]),
                            "package_type": "tarball",
                            "use_new_format": True,
                        },
                        out={
                            "MODADVERT_LYNCHER_BINARIES": 100,
                            "MODADVERT_QLYNCHER_BINARIES": 100,
                            "MODADVERT_LYNCHER_CONFIG": 100,
                            "MODADVERT_OFFLINE_LYNCHER_BINARIES": 100,
                        },
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="MODADVERT_PUSH_DOCKER_IMAGE",
                        job_params={
                            "job_name_parameter": "PUSH_MODADVERT_QLYNCHER_BINARIES",
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="docker_package_checkout_arcadia_from_url",
                                        output_key=sandbox_constants.ARCADIA_URL_KEY,
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="packaged_resource_id",
                                        resource_name="MODADVERT_QLYNCHER_BINARIES",
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            "resource_type": "tarball",
                            "resource_type_name": "MODADVERT_QLYNCHER_BINARIES",
                            "registry_login": "robot-modadvert-test",
                            "vault_item_owner": "robot-modadvert-test",
                            "vault_item_name": "registry-token",
                        }
                    )
                )
                for run_qlyncher_test in [True, False]:
                    branch_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="MODADVERT_TEST_LYNCHER",
                            job_params={
                                "job_name_parameter": "RUN_{}_B2B".format("QLYNCHER" if run_qlyncher_test else "LYNCHER"),
                                "frequency": (jg_utils.TestFrequency.LAZY if run_qlyncher_test else jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="feature_lyncher_binaries_resource",
                                            resource_name="MODADVERT_LYNCHER_BINARIES",
                                        ),
                                        jg_job_data.ParentDataResource(
                                            input_key="feature_qlyncher_binaries_resource",
                                            resource_name="MODADVERT_QLYNCHER_BINARIES",
                                        ),
                                        jg_job_data.ParentDataResource(
                                            input_key="feature_config_resource",
                                            resource_name="MODADVERT_LYNCHER_CONFIG",
                                        ),
                                    )
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="PUSH_MODADVERT_QLYNCHER_BINARIES",
                                ),
                                jg_arrows.ParamsData(
                                    input_key="release_number",
                                    transform=jg_utils.get_major_release_number,
                                )
                            ),
                            ctx={
                                "yt_proxy_url": "hahn",
                                "vault_user": "robot-modadvert-test",
                                "tokens": {"YT_TOKEN": "yt-token", "SANDBOX_TOKEN": "sandbox-token"},
                                "use_qlyncher": run_qlyncher_test,
                                "max_tables_age": 168,  # 7 days
                                "sampling_rate": 0.1,
                                "max_rows_count": 0,  # After sampling all requests will be merged to a single table. Specify `sampling_rate` to limit amount of rows
                            },
                        )
                    )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                for environment in ["HAHN", "ARNOLD", "OFFLINE"]:
                    release_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="MODADVERT_DEPLOY_LYNCHER",
                            job_params={
                                "job_name_parameter": "DEPLOY_TO_{}".format(environment),
                            },
                            job_arrows=(
                                jg_arrows.ParamsData(
                                    input_key="release_number",
                                    transform=jg_utils.get_major_release_number,
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="config_resource",
                                            resource_name="MODADVERT_LYNCHER_CONFIG",
                                        ),
                                        jg_job_data.ParentDataResource(
                                            input_key="binaries_resource",
                                            resource_name="MODADVERT_LYNCHER_BINARIES",
                                        ),
                                        jg_job_data.ParentDataResource(
                                            input_key="offline_binaries_resource",
                                            resource_name="MODADVERT_OFFLINE_LYNCHER_BINARIES",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="PUSH_MODADVERT_QLYNCHER_BINARIES",
                                    parent_job_data=(
                                        jg_job_data.ParentDataCtx(
                                            input_key="release_tag_number",
                                            output_key="release_tag_number",
                                            transform=lambda x, params: str(x),
                                        ),
                                    ),
                                ),
                            ),
                            ctx={
                                "vault_user": "robot-modadvert-test",
                                "st_token_vault_name": "st-token",
                                "release_environment": environment,
                            }
                        )
                    )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def change_frequency(self):
                uppercase_name = self.name.upper()
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "_{}_CHANGELOG_MINOR".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_{}_NEW_TAG".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_RUN_{}_B2B".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

    class Releases(configs.ReferenceBranchedConfig.Releases):
        resources_info = [
            configs.ReleasedResourceInfo(
                name="lyncher",
                resource_type="MODADVERT_LYNCHER_BINARIES",
                resource_name="lyncher_executable_id",
            ),
            configs.ReleasedResourceInfo(
                name="lyncher",
                resource_type="MODADVERT_LYNCHER_CONFIG",
                resource_name="lyncher_config_id",
            ),
        ]
        release_followers_permanent = [
            "losev",
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = []
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "losev"
            queue = "MODADVRELEASE"
            dev_queue = "MODADVERT"
            summary_template = u"Приемка нового контура автомодерации (lyncher) {}"
            workflow = {}
            followers = [
                "losev",
                "granilace",
                "losev",
                "vsuvorov",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "modadvert/releases/lyncher/"
        dirs = [
            "arcadia/modadvert/libs/lyncher",
            "arcadia/modadvert/programs/lyncher",
            "arcadia/modadvert/programs/offline_lyncher",
        ]
        review_groups = [
            "modadvert",
        ]

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "modadvert/lyncher/"
