# -*- coding: utf-8 -*-
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import yappy as yappy_cfg
from sandbox.projects.common.constants import constants as common_const
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
from sandbox.projects.release_machine.core import releasable_items as ri
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.config_core.statistics_page as statistics_page
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows


_NOT_GRANTED_MESSAGE_FORMAT = """Merger {} is not allowed to merge into {}.
Please, visit middle releases chat: https://t.me/joinchat/CaUODkN2zNWU7HSBHK8V2Q
or contact current release engineer on duty: https://abc.yandex-team.ru/services/mmeta-web/duty/
"""
MMETA_SUBTYPE_RESOURCES = (
    ("QUICK", ("FRESH_RANKING_MIDDLESEARCH_EXECUTABLE", "mmeta_fresh")),
    ("IMAGES", ("IMAGES_MIDDLESEARCH_EXECUTABLE", "mmeta_images")),
    ("WEB", ("RANKING_MIDDLESEARCH_EXECUTABLE", "mmeta_web")),
)
BUILD_MIDDLESEARCH_2_TASK = "BUILD_MIDDLESEARCH_2"
BUILD_MIDDLESEARCH_IMAGES_TASK = "BUILD_MIDDLESEARCH_IMAGES"


class MiddleCfg(configs.ReferenceBranchedConfig):
    component_group = "reference_component_metricsed"
    name = "middle"
    display_name = u"Middlesearch"
    responsible = "eeight"

    class MergesCfg(configs.ReferenceBranchedConfig.MergesCfg):
        permissions = configs.MergePermissions(
            permission_type=rm_const.PermissionType.ALLOWED,
            people_groups=configs.PeopleGroups(
                # https://staff.yandex-team.ru/departments/yandex_search_tech_quality_component/
                staff_groups=["yandex_search_tech_quality_component_8875"],
                # https://abc.yandex-team.ru/services/mmeta-web/
                abc_services=[configs.Abc(component_id=2100, role_id=None)],
                logins=None,
            ),
        )

        def not_granted_message(self, author, responsible):
            return _NOT_GRANTED_MESSAGE_FORMAT.format(
                author, self.name,
            )

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_db = "ws-middle-trunk"
        trunk_task_owner = "SEARCH-RELEASERS"
        branch_db_template = "ws-middle-{testenv_db_num}"
        merge_on_clone = ["images-middle-trunk"]

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            _DEFAULT_RESOURCE_TTL = 365

            def _generate_output_resources(self, target_resources):
                return dict([
                    (resource, self._DEFAULT_RESOURCE_TTL) for resource in target_resources
                ])

            @property
            def _trunk_part(self):
                trunk_part = super(self.__class__, self)._trunk_part

                trunk_part.append(
                    jg_test.JobGraphElementRunIceFlame(
                        config_file="sas_jupiter_mmeta.conf",
                        job_params={
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.DEFINED_BY_CODE, None),
                            "next_revision_custom_schedule": jg_utils.CustomSchedule(
                                time_interval_list=[(7, 9)],
                                seconds_from_last_run_commit=1.5 * 3600,
                                once_a_day=True,
                            ),
                            "job_name_parameter": "SAS_JUPITER_MMETA_ICEFLAME",
                        },
                        ctx={
                            "release_item_name": "mmeta_web",
                            "service_name": "sas_jupiter_mmeta_yp",
                            "fail_if_service_version_cannot_be_determined": True,
                            "check_production_version": True,
                        },
                    ),
                )

                return trunk_part

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part

                component_to_build_resources = {
                    "WEB": {
                        "task": BUILD_MIDDLESEARCH_2_TASK,
                        "resources": [
                            "RANKING_MIDDLESEARCH_EXECUTABLE",
                            "LOGANALYZE_EXECUTABLE",
                            "EVLOGDUMP_EXECUTABLE",
                            "EVLOGSAMPLE_EXECUTABLE",
                        ]
                    },
                    "QUICK": {
                        "task": BUILD_MIDDLESEARCH_2_TASK,
                        "resources": [
                            "EVLOGDUMP_EXECUTABLE",
                            "LOGANALYZE_EXECUTABLE",
                            "FRESH_RANKING_MIDDLESEARCH_EXECUTABLE",
                            "EVLOGSAMPLE_EXECUTABLE",
                        ]
                    },
                    "IMAGES": {
                        "task": BUILD_MIDDLESEARCH_IMAGES_TASK,
                        "resources": [
                            "IMAGES_MIDDLESEARCH_EXECUTABLE",
                            "L1_INTSEARCH_EXECUTABLE",
                            "INTSEARCH_EXECUTABLE",
                            "EVLOGDUMP_EXECUTABLE",
                            "EVLOGSAMPLE_EXECUTABLE",
                            "LOGANALYZE_EXECUTABLE",
                        ]
                    }
                }
                all_resources = []
                for component in component_to_build_resources:
                    all_resources += component_to_build_resources[component]["resources"]
                all_resources = list(set(all_resources))

                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_MIDDLESEARCH_2",
                        ctx={
                            "target_resources": all_resources,
                            "build_system": common_const.SEMI_DISTBUILD_BUILD_SYSTEM,
                            "notify_if_failed": "SEARCH-RELEASERS",
                            "thinlto": True,
                        },
                        out=self._generate_output_resources(all_resources),
                    )
                )

                for component, component_info in component_to_build_resources.items():
                    output_resources = self._generate_output_resources(component_info["resources"])

                    build_ctx = {
                        "target_resources": component_info["resources"],
                        "build_system": common_const.SEMI_DISTBUILD_BUILD_SYSTEM,
                        "notify_if_failed": "SEARCH-RELEASERS",
                        "thinlto": True,
                    }
                    if component in ("WEB",):
                        build_ctx[common_const.DEFINITION_FLAGS_KEY] = '-DCFLAGS="-fno-omit-frame-pointer"'

                    branch_part.append(
                        jg_build.JobGraphElementYaMakeBuildBranched(
                            task_name=component_info["task"],
                            job_params={
                                "job_name_parameter": component,
                            },
                            ctx=build_ctx,
                            out=output_resources,
                        )
                    )
                beta_params = [
                    ("IMAGES_MIDDLESEARCH_EXECUTABLE", "images"),
                    ("RANKING_MIDDLESEARCH_EXECUTABLE", "web"),
                    ("FRESH_RANKING_MIDDLESEARCH_EXECUTABLE", "quick"),
                ]
                for res_type, beta_conf_type in beta_params:
                    branch_part.append(
                        jg_test.JobGraphElementYappyBetaGeneratorBranched(
                            beta_conf_type=beta_conf_type,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "mmeta_web",
                                            res_type,
                                        ),
                                    ),
                                ),
                            ),
                        )
                    )

                # Metrics SLA-project "SLA Release Machine", more info at https://metrics.yandex-team.ru/admin/sla
                MIDDLE_METRICS_SLA_PROJECT = "aa8286386850f0df016874cc48882d44"
                branch_part.append(
                    jg_test.JobGraphElementLaunchMetrics(
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="checked_beta",
                                        output_key="beta_name",
                                        transform=lambda x, params: "{}.hamster".format(x),
                                    )
                                ),
                                job_name_parameter="WEB",
                            )
                        ),
                        ctx={
                            "beta_conf_type": "web",
                            "sample_beta": "hamster",
                            "scraper_over_yt_pool": "middle_web_priemka",
                            "custom_template_name": "common.json",  # METRICS-6369
                            "sla_project": MIDDLE_METRICS_SLA_PROJECT,
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementLaunchMetrics(
                        search_subtype="images",
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="checked_beta",
                                        output_key="beta_name",
                                        transform=lambda x, params: "{}.hamster".format(x),
                                    )
                                ),
                                job_name_parameter="IMAGES",
                            )
                        ),
                        ctx={
                            "beta_conf_type": "images",
                            "custom_template_name": "common_wizard.json",
                            "sample_beta": "priemka-img.hamster",
                            "scraper_over_yt_pool": "middle_images_priemka",
                            "sla_project": MIDDLE_METRICS_SLA_PROJECT,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="PRIEMKA_ALL_MIDDLESEARCH_BINARIES",
                        job_params={
                            "job_name_parameter": "DAILY_PRIEMKA",
                            "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 1440),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataId(
                                        input_key="build_task_new",
                                    )
                                ),
                            ),
                            jg_arrows.ParamsData("release_number", jg_utils.get_major_release_number),
                            jg_arrows.ParamsData("revision_tested", transform=lambda x, rm_config: x.revision),
                        ),
                        ctx={
                            "set_hamster_config_source": True,
                            "compare_binaries_result_resource_ttl": self._DEFAULT_RESOURCE_TTL,
                            "priemka_web": False,
                            "priemka_video": False,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="PRIEMKA_ALL_MIDDLESEARCH_BINARIES",
                        job_params={
                            "job_name_parameter": "DAILY_PRIEMKA_WEB",
                            "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 1440),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataId(
                                        input_key="build_task_new",
                                    )
                                ),
                                job_name_parameter="WEB"
                            ),
                            jg_arrows.ParamsData("release_number", jg_utils.get_major_release_number),
                            jg_arrows.ParamsData("revision_tested", transform=lambda x, rm_config: x.revision),
                        ),
                        ctx={
                            "set_hamster_config_source": True,
                            "compare_binaries_result_resource_ttl": self._DEFAULT_RESOURCE_TTL,
                            "priemka_web": True,
                            "priemka_img": False,
                            "priemka_cbir": False,
                            "priemka_imgquick": False,
                            "priemka_quick": False,
                            "priemka_video": False,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestXMLSearch(
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="beta_url",
                                        output_key="beta_name",
                                        transform=lambda x, params: "https://{}.hamster.yandex.ru".format(x),
                                    )
                                ),
                                job_name_parameter="WEB",
                            )
                        ),
                    )
                )

                umbrella_parents = (
                    jg_job_triggers.JobTriggerLaunchMetrics(job_name_parameter="WEB"),
                    jg_job_triggers.JobTriggerLaunchMetrics(job_name_parameter="IMAGES"),
                    jg_job_triggers.JobTriggerTestXMLSearch(),
                    jg_job_triggers.JobTriggerActionRelease(
                        job_name_parameter=rm_const.ReleaseStatus.testing,
                    ),
                )

                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedScheduled(
                        job_arrows=umbrella_parents,
                        frequency=(jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 2880),
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedByMarker(
                        job_arrows=umbrella_parents,
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                for release_stage in [rm_const.ReleaseStatus.testing, rm_const.ReleaseStatus.stable]:
                    release_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "mmeta_web",
                                            "RANKING_MIDDLESEARCH_EXECUTABLE",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="IMAGES",
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "mmeta_images",
                                            "IMAGES_MIDDLESEARCH_EXECUTABLE",
                                        ),
                                    ),
                                ),
                            ),
                        )
                    )
                    release_part.append(
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                        )
                    )
                for s_type, p_res in MMETA_SUBTYPE_RESOURCES:
                    res_type, res_ctx_key = p_res
                    release_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=rm_const.ReleaseStatus.stable,
                            release_item=s_type,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict("component_resources", res_ctx_key, res_type),
                                    ),
                                    job_name_parameter=s_type,
                                )
                            )
                        )
                    )
                    release_part.append(
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=rm_const.ReleaseStatus.stable,
                            release_item=s_type,
                        )
                    )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                uppercase_name = self.name.upper()
                tests.update({
                    "_BUILD__BASE": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD__BASE__DEBUG": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_RELEASE_IMAGES_{}SEARCH".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_RELEASE_QUICK_{}SEARCH".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_RELEASE_WEB_{}SEARCH".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_RELEASE_{}SEARCH_2".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_{}_UMBRELLA_ACCEPT_SCHEDULED".format(uppercase_name):
                        rm_const.TestFrequencies.EVERY_TWO_DAYS_TEST,
                })
                return tests

            @property
            def ignore_match(self):
                return super(self.__class__, self).ignore_match + [
                    "BUILD_MIDDLESEARCH_FOR_PRIEMKA",
                    "PERFORMANCE_MIDDLESEARCH_SINGLE_HOST_CPROXY",
                    "MMETA_PERFORMANCE_SINGLE_HOST_PROFILE",
                    "FUZZY_TEST_MIDDLESEARCH",
                    "BUILD_MIDDLESEARCH_FUZZER",
                    "PREPARE_WEB_MIDDLE_FUZZY_INPUT",
                ]  # SEARCH-4376

            @property
            def ignore_prefix(self):
                """List of test prefixes which should be ignored in branch db"""
                return super(self.__class__, self).ignore_prefix + [
                    "WEB_MMETA_FUZZY",
                ]

    class ReleaseViewer(configs.ReferenceConfig.ReleaseViewer):
        kpi_alert = 4
        statistics_page_charts = statistics_page.DEFAULT + [statistics_page.PredefinedCharts.TIMESPECTRE_METRICS]

    class Releases(configs.ReferenceBranchedConfig.Releases):
        allow_old_releases = True
        release_followers_permanent = [
            "kulikov",
            "eeight",
        ]
        responsible = configs.Responsible(
            abc=configs.Abc(component_id=2100, schedule_slug="onduty-web-mmeta-releases"),
            login="eeight"
        )

        @property
        def releasable_items(self):
            mmeta_prod_services = [
                "sas_jupiter_mmeta_yp",
                "vla_jupiter_mmeta_yp",
            ]
            mmeta_nightly_services = [
                "nightlytest_jupiter_mmeta",
                "nightlybaseline_jupiter_mmeta",
            ]
            mmeta_pip_services = [
                "cajuper_mmeta_pip",
                "vla_jupiter_mmeta_pip_jprod",
            ]
            mmeta_beta_services = [
                "vla_callisto_beta_mmeta",
            ]
            mmeta_heater_services = [
                "sas_jupiter_mmeta_heater_yp",
                "vla_jupiter_mmeta_heater_yp",
            ]
            mmeta_hamster_services = [
                "sas_jupiter_mmeta_hamster_prs",
                "sas_jupiter_mmeta_hamster_yp",
                "vla_jupiter_mmeta_hamster_yp",
                "vla_jupiter_mmeta_hamster_prs",
            ]
            return [
                ri.ReleasableItem(
                    name="mmeta_web",
                    data=ri.SandboxResourceData("RANKING_MIDDLESEARCH_EXECUTABLE"),
                    deploy_infos=[
                        ri.NannyDeployInfo(
                            [
                                ri.DeployService(i, tags=["prod"]) for i in mmeta_prod_services
                            ] + [
                                ri.DeployService(i, tags=["pip"]) for i in mmeta_pip_services
                            ] + [
                                ri.DeployService(i, tags=["heater"]) for i in mmeta_heater_services
                            ] + [
                                ri.DeployService(i, tags=["beta"]) for i in mmeta_beta_services
                            ],
                            stage=rm_const.ReleaseStatus.stable
                        ),
                        ri.NannyDeployInfo(
                            [ri.DeployService(i, tags=["hamster"]) for i in mmeta_hamster_services],
                            stage=rm_const.ReleaseStatus.testing
                        ),
                        ri.NannyDeployInfo(
                            [ri.DeployService(i, tags=["nightly"]) for i in mmeta_nightly_services],
                            stage=rm_const.ReleaseStatus.unstable
                        ),
                    ],
                ),
                ri.ReleasableItem(
                    name="mmeta_fresh",
                    data=ri.SandboxResourceData("FRESH_RANKING_MIDDLESEARCH_EXECUTABLE"),
                ),
                ri.ReleasableItem(
                    name="mmeta_images",
                    data=ri.SandboxResourceData("IMAGES_MIDDLESEARCH_EXECUTABLE"),
                ),
            ]

    class Notify(configs.ReferenceBranchedConfig.Notify):

        notifications = [
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="middle_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewTag",
                chat_name="middle_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="BuildTest",
                chat_name="middle_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="KpiAlert",
                chat_name="middle_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="middle_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=[
                        rm_notifications.NotificationConditionItem(
                            field="task_data.status",
                            operator="TEXT_EXACTLY_IS",
                            value="SUCCESS",
                        ),
                        rm_notifications.NotificationConditionItem(
                            field="ticket_history_data.ticket_history_latest_status",
                            operator="TEXT_EXACTLY_IS",
                            value="open",
                        ),
                    ],
                    join_strategy="AND",
                ),
                tags="issue r{{event.ticket_history_data.scope_number}}",
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="middle_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewBetaGeneration",
                chat_name="middle_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="ReleaseFailed",
                chat_name="middle_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="MergeCommit",
                chat_name="middle_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="middle_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_deployed_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="middle_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
        ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["middle_comp", "middle_priemka"]
            config = configs.RmTelegramNotifyConfig(chats=chats)
            invite_link = u"https://t.me/joinchat/CaUODkN2zNWU7HSBHK8V2Q"

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "eeight"
            queue = "MIDDLEREL"
            dev_queue = "MIDDLE"
            summary_template = u"Приемка среднего метапоиска {}"
            components = u"Средний метапоиск".encode("utf-8")
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "kulikov",
                "sankear",
                "eeight",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 4
            nanny_reports = False

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "jandekspoisk/kachestvopoiska/middlesearch/releases/"
        ya_make_targets = [
            "arcadia/search/daemons/ranking_middlesearch",
            "arcadia/search/meta",
            "arcadia/search/web/rearrs_middle",
        ]
        observed_paths = [
            # INCLUDEd from search/web/rearrs_middle/antidup/ya.make
            "arcadia/search/web/rearrs_upper/rearrange.dynamic/antidup/text_source",
        ]
        testenv_dbs = False
        commit_importance_threshold = 2
        review_groups = [
            "middle",
            "rearrange",
        ]

    class MetricsCfg(configs.ReferenceBranchedConfig.MetricsCfg):
        run_bisect = True

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        allow_autobranches = True

    class Yappy(yappy_cfg.YappyBaseCfg):
        _mmeta_res = yappy_cfg.YappyParametrizedResource(
            local_path="mmeta.executable",
            param_name="mmeta_web",
            checkconfig_name="mmiddle-linux-bin-md5",
        )
        _mmeta_models_res = yappy_cfg.YappyStaticResource(local_path="mmeta.models", manage_type="BC_DEFAULT")

        betas = {
            "web": yappy_cfg.YappyTemplateCfg(
                template_name="mmeta-web",
                patches=[
                    yappy_cfg.YappyTemplatePatch(patch_dir="mmeta-web", resources=[_mmeta_res, _mmeta_models_res]),
                ],
                new_yappy=True,
            ),
            "images": yappy_cfg.YappyTemplateCfg(
                template_name="mmeta-imgs",
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="mmeta-imgs",
                        resources=[
                            _mmeta_res,
                            _mmeta_models_res,
                            yappy_cfg.YappyStaticResource(local_path="mmeta.rearrange_data", manage_type="BC_DEFAULT"),
                            yappy_cfg.YappyStaticResource(local_path="mmeta.rearrange_index", manage_type="BC_DEFAULT"),
                        ]
                    ),
                ],
                new_yappy=True,
            ),
            "quick": yappy_cfg.YappyTemplateCfg(
                template_name="mmeta-frozen-quick",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="saas-refresh-quick-frozen-mmeta",
                        resources=[
                            yappy_cfg.YappyParametrizedResource(
                                local_path="httpsearch",
                                param_name="mmeta_web",
                                checkconfig_name="mmiddle-linux-bin-md5",
                            ),
                        ],
                        parent_service="saas-refresh-yp-shmick-frozen-mmeta",
                    ),
                ],
            ),
        }
