# -*- coding: utf-8 -*-

from sandbox.projects.common.constants import constants as common_const
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers


class PodAgentCfg(configs.ReferenceBranchedConfig):
    """
        pod_agent for Y.Deploy.
    """

    name = "pod_agent"
    responsible = "amich"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_task_owner = "DRUG-RELEASERS"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="BUILD_POD_AGENT_BINARY",
                        out={
                            "POD_AGENT_BINARY": 90,
                        },
                        ctx={
                            "targets": "infra/pod_agent",
                            "arts": "infra/pod_agent",
                            "result_rt": "POD_AGENT_BINARY",
                            "result_rd": "pod_agent_binary",
                            "result_single_file": True,
                            common_const.USE_AAPI_FUSE: True,
                            common_const.ALLOW_AAPI_FALLBACK: True,
                        },
                        job_params={
                            "apiargs": {
                                "hidden": False,
                                "requirements": {
                                    "platform": "linux",
                                    "disk_space": (15 << 30),  # 15 Gb
                                },
                            },
                        }
                    ),
                )
                branch_part.append(
                    jg_test.JobGraphElementMetaTestBranched(
                        job_params={
                            "job_name_parameter": "BUILD_TRUNK_META_LINUX_DISTBUILD",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        ctx={
                            "targets": "infra/pod_agent",
                        },
                    )
                )

                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release

                for release_stage, release_tag in [
                    (rm_const.ReleaseStatus.stable, "XDC"),
                    (rm_const.ReleaseStatus.prestable, "XDC_ACCEPTANCE"),
                    (rm_const.ReleaseStatus.prestable, "MAN_PRE"),
                    (rm_const.ReleaseStatus.testing, "SAS_TEST"),
                ]:
                    release_part.append(jg_release.JobGraphElementReleaseBranched(
                        release_to=release_stage,
                        release_item=release_tag,
                        job_params={
                            "ctx": {
                                "additional_release_parameters": {
                                    "release_tag": release_tag.lower(),
                                },
                            },
                        },
                        job_arrows=[
                            jg_job_triggers.JobTriggerBuild([
                                jg_job_data.ParentDataDict("component_resources", "pod_agent_binary", "POD_AGENT_BINARY"),
                            ]),
                            jg_job_triggers.JobTriggerNewTag([
                                jg_job_data.ParentDataOutput("major_release_num", "branch_number_for_tag"),
                                jg_job_data.ParentDataOutput("minor_release_num", "new_tag_number"),
                                jg_job_data.ParentDataCtx("release_number", "scope_number", override=True),
                            ]),
                        ],
                    ))
                    release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                        release_to=release_stage,
                        release_item=release_tag,
                    ))

                return release_part

    class Releases(configs.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        resources_info = [
            configs.ReleasedResourceInfo(
                name="pod_agent_binary",
                resource_type="POD_AGENT_BINARY",
            ),
        ]
        release_followers_permanent = [
            "amich",
            "dkochetov",
        ]
        allow_robots_to_release_stable = True

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""

        notifications = [
            rm_notifications.Notification(
                event_type="MergeCommit",
                chat_name="pod_agent_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="pod_agent_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewTag",
                chat_name="pod_agent_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TestenvDbClone",
                chat_name="pod_agent_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewWikiPage",
                chat_name="pod_agent_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="pod_agent_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="pod_agent_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.STARTREK_TICKET_CREATED_CONDITIONS,
                    join_strategy="OR",
                ),
                message_template_file="notifications/ticket_created.html",
            ),
            rm_notifications.Notification(
                event_type="KpiAlert",
                chat_name="pod_agent_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="pod_agent_comp",
                message_template_file='notifications/release_created_oldschool.html',
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="pod_agent_comp",
                message_template_file='notifications/release_deployed_oldschool.html',
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
        ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = ["pod_agent_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "amich"
            queue = "PODAGENTREL"
            dev_queue = "DEPLOY"
            summary_template = u"Приемка pod_agent {}"
            workflow = {}
            followers = [
                "amich",
                "dkochetov",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "infra/pod_agent/releases/"
        dirs = [
            "arcadia/infra/pod_agent",
            "arcadia/yp/yp_proto/yp/client/api/proto/pod_agent.proto",
        ]
