# -*- coding: utf-8 -*-

from sandbox.projects.release_machine.components import configs
import sandbox.projects.release_machine.core.const as rm_const

import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_job_arrows
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils

from sandbox.projects.quasar.release_machine_jobs.common import JobRunTasklet

GB = 1 << 30

COMPONENT_NAME = 'quasar_backend'

AUTOSHOOT_TASKLET_NAME = 'LunaparkAutoshoot'
NANNY_RELEASE_TASKLET_NAME = 'NannyBackendRelease'
WARMUP_AUTOSHOOT_JOB_NAME = 'Warmup_{}'.format(AUTOSHOOT_TASKLET_NAME)

BASELINE_WARMUP_AUTOSHOOT_JOB_NAME = 'Baseline_{}'.format(WARMUP_AUTOSHOOT_JOB_NAME)
BASELINE_AUTOSHOOT_JOB_NAME = 'Baseline_{}'.format(AUTOSHOOT_TASKLET_NAME)


class JobRunReleaseTasklet(JobRunTasklet):

    @staticmethod
    def get_tasklet_name():
        return NANNY_RELEASE_TASKLET_NAME

    def get_tasklet_input(self, **kwargs):
        tasklet_input = kwargs
        branch = kwargs['branch_number_for_tag']
        tag = kwargs['new_tag_number']
        tasklet_input.update({
            'comment': 'stable-{}-{} RM tasklet release'.format(branch, tag),
        })
        tasklet_input.update(self.extra_input)
        return tasklet_input


class JobRunAutoshootTasklet(JobRunTasklet):

    @staticmethod
    def get_tasklet_name():
        return AUTOSHOOT_TASKLET_NAME

    def get_tasklet_input(self, **kwargs):
        branch = kwargs['branch_number_for_tag']
        tag = kwargs['new_tag_number']
        if 'tasklet_output' in kwargs and kwargs['tasklet_output']:
            # main autoshoot job, needs baseline shootings's tasklet_output to compare the results
            tasklet_input = {
                'version': 'stable-{}-{}'.format(branch, tag),
                'description': 'stable-{}-{}'.format(branch, tag),
                'component_name': COMPONENT_NAME,
                'branch_number': branch,  # branch_number present = startrek comment
                'compare_with': kwargs['tasklet_output']['lunaparkId']
            }

            self.extra_input['name'] += 'stable-{}-{}'.format(branch, tag)
        else:
            # otherwise, it's a baseline shootings and we shouldn't comment or have a version
            tasklet_input = {
                'version': 'baseline version',
                'description': 'baseline shootings',
            }

            self.extra_input['name'] += 'baseline'

        tasklet_input.update(self.extra_input)
        return tasklet_input


class QuasarBackendCfg(configs.ReferenceBranchedConfig):
    name = COMPONENT_NAME
    responsible = 'grmammaev'

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""

        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            """Mail notifications configuration"""
            mailing_list = [
                'quasar-dev@yandex-team.ru',
            ]

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = 'grmammaev'
            queue = 'QUASARRELEASE'
            dev_queue = 'QUASARINFRA'
            summary_template = u'Отведение ветки quasar-backend-{}'
            workflow = {}
            followers = []
            add_commiters_as_followers = True
            important_changes_limit = 1000

    class Releases(configs.ReferenceBranchedConfig.Releases):
        resources_info = [
            configs.ReleasedResourceInfo(
                name='quasar_backend_binaries',
                resource_type='QUASAR_BACKEND_BINARIES',
                deploy=[
                    (rm_const.ReleaseStatus.testing, 'quasar_backend_load'),
                    (rm_const.ReleaseStatus.prestable, 'quasar_backend_dev'),
                    (rm_const.ReleaseStatus.stable, 'quasar_backend_vla'),
                ]
            )
        ]
        allow_old_releases = True

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = ''
        dirs = ['arcadia/quasar/backend']

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = 'quasar/backend'
        tag_name = 'quasar/backend'
        use_arc = True
        use_zipatch_only = True

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_task_owner = 'QUASAR'
        quasar_component = 'backend'
        trunk_db = 'quasar-backend-trunk'
        branch_db_template = 'quasar-backend-{testenv_db_num}'

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):

            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrek(use_arc=self._use_arc),
                ]

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(jg_build.JobGraphElementBuildBranched(
                    task_name='YA_MAKE_RELEASE_TO_NANNY',
                    build_item='binaries',
                    job_params={
                        'apiargs': {
                            'requirements': {
                                'disk_space': 30 * GB,  # in bytes
                            },
                        },
                    },
                    ctx={
                        'targets': 'quasar/backend/packages/binaries',
                        'arts': 'quasar/backend/packages/binaries',
                        'result_rd': 'Quasar backend binaries package',
                        'result_rt': 'QUASAR_BACKEND_BINARIES',
                        'result_single_file': True,
                        'use_aapi_fuse': True,
                        'aapi_fallback': True,
                        'checkout_mode': 'auto',
                    },
                    out={'QUASAR_BACKEND_BINARIES': 90},
                ))

                job_trigger_new_tag = jg_job_triggers.JobTriggerNewTag(
                    parent_job_data=(
                        jg_job_data.ParentDataOutput(
                            input_key='new_tag_number',
                            output_key='new_tag_number'
                        ),
                        jg_job_data.ParentDataOutput(
                            input_key='branch_number_for_tag',
                            output_key='branch_number_for_tag'
                        )
                    )
                )

                # start register warmup on loadbase
                branch_part.append(
                    JobRunAutoshootTasklet(
                        job_type=rm_const.JobTypes.DEPLOY,
                        job_triggers=(
                            job_trigger_new_tag,
                        ),
                        frequency=jg_utils.TestFrequency.CHECK_EACH_COMMIT,
                        extra_input={
                            'target': {'service': 'quasar_backend_loadbase', 'tags': ['a_dc_vla']},
                            'tank': {'service': 'quasar_yandex_tank', 'tags': ['a_dc_vla']},
                            'load_scheme': 'const(30, 205s)',
                            'name': 'Quasar backend stress (registration warmup) ',
                            'operator': 'robot-quasar',
                            'load_conf_resource_id': '1426807827',
                            'is_register_warmup': True,
                            'task': 'QUASARINFRA-176',
                        },
                        job_params={'job_name_parameter': BASELINE_WARMUP_AUTOSHOOT_JOB_NAME}
                    )
                )

                # start main ammo fire on loadbase
                baseline_autoshoot_tasklet_triggers = [
                    job_trigger_new_tag,
                    jg_job_arrows.JobTrigger(
                        job_type=rm_const.JobTypes.DEPLOY,
                        job_name_parameter=BASELINE_WARMUP_AUTOSHOOT_JOB_NAME
                    ),
                ]

                branch_part.append(
                    JobRunAutoshootTasklet(
                        job_type=rm_const.JobTypes.DEPLOY,
                        job_triggers=baseline_autoshoot_tasklet_triggers,
                        frequency=jg_utils.TestFrequency.CHECK_EACH_COMMIT,
                        extra_input={
                            'target': {'service': 'quasar_backend_loadbase', 'tags': ['a_dc_vla']},
                            'tank': {'service': 'quasar_yandex_tank', 'tags': ['a_dc_vla']},
                            'load_scheme': 'step(200, 2000, 400, 10s) const (2000, 10m)',
                            'name': 'Quasar backend stress ',
                            'operator': 'robot-quasar',
                            'load_conf_resource_id': '1426807827',
                            'task': 'QUASARINFRA-176',
                        },
                        job_params={'job_name_parameter': BASELINE_AUTOSHOOT_JOB_NAME}
                    )
                )

                # release binaries and configs to nanny load
                release_tasklet_triggers = [
                    jg_job_triggers.JobTriggerBuild(
                        parent_job_data=(
                            jg_job_data.ParentDataId(
                                input_key='quasar_backend_task_id',
                            ),
                            jg_job_data.ParentDataResource(
                                input_key='quasar_backend_resource_id',
                                resource_name='QUASAR_BACKEND_BINARIES',
                            )
                        ),
                        job_name_parameter='binaries',
                    ),
                    job_trigger_new_tag,
                ]

                branch_part.append(
                    JobRunReleaseTasklet(
                        job_type=rm_const.JobTypes.DEPLOY,
                        job_triggers=release_tasklet_triggers,
                        frequency=jg_utils.TestFrequency.CHECK_EACH_COMMIT,
                        extra_input={
                            'service_id': 'quasar_backend_load',
                        },
                        job_params={'job_name_parameter': NANNY_RELEASE_TASKLET_NAME}
                    )
                )

                # start register warmup and compare to baseline
                registration_autoshoot_tasklet_triggers = [
                    job_trigger_new_tag,
                    jg_job_arrows.JobTrigger(
                        job_type=rm_const.JobTypes.DEPLOY,
                        job_name_parameter=NANNY_RELEASE_TASKLET_NAME
                    ),
                    jg_job_arrows.JobTrigger(
                        job_type=rm_const.JobTypes.DEPLOY,
                        job_name_parameter=BASELINE_WARMUP_AUTOSHOOT_JOB_NAME,
                        parent_job_data=(
                            jg_job_data.ParentDataOutput(
                                input_key='tasklet_output',
                                output_key='tasklet_output'
                            ),
                        )
                    )
                ]

                branch_part.append(
                    JobRunAutoshootTasklet(
                        job_type=rm_const.JobTypes.DEPLOY,
                        job_triggers=registration_autoshoot_tasklet_triggers,
                        frequency=jg_utils.TestFrequency.CHECK_EACH_COMMIT,
                        extra_input={
                            'target': {'service': 'quasar_backend_load', 'tags': ['a_dc_vla']},
                            'tank': {'service': 'quasar_yandex_tank', 'tags': ['a_dc_vla']},
                            'load_scheme': 'const(30, 205s)',
                            'name': 'Quasar backend stress (registration warmup) ',
                            'operator': 'robot-quasar',
                            'load_conf_resource_id': '1426807827',
                            'is_register_warmup': True,
                        },
                        job_params={'job_name_parameter': WARMUP_AUTOSHOOT_JOB_NAME}
                    )
                )

                # start main ammo fire and compare to baseline
                autoshoot_tasklet_triggers = [
                    job_trigger_new_tag,
                    jg_job_arrows.JobTrigger(
                        job_type=rm_const.JobTypes.DEPLOY,
                        job_name_parameter=WARMUP_AUTOSHOOT_JOB_NAME
                    ),
                    jg_job_arrows.JobTrigger(
                        job_type=rm_const.JobTypes.DEPLOY,
                        job_name_parameter=BASELINE_AUTOSHOOT_JOB_NAME,
                        parent_job_data=(
                            jg_job_data.ParentDataOutput(
                                input_key='tasklet_output',
                                output_key='tasklet_output'
                            ),
                        )
                    )
                ]

                branch_part.append(
                    JobRunAutoshootTasklet(
                        job_type=rm_const.JobTypes.DEPLOY,
                        job_triggers=autoshoot_tasklet_triggers,
                        frequency=jg_utils.TestFrequency.CHECK_EACH_COMMIT,
                        extra_input={
                            'target': {'service': 'quasar_backend_load', 'tags': ['a_dc_vla']},
                            'tank': {'service': 'quasar_yandex_tank', 'tags': ['a_dc_vla']},
                            'load_scheme': 'step(200, 2000, 400, 10s) const (2000, 10m)',
                            'name': 'Quasar backend stress ',
                            'operator': 'robot-quasar',
                            'load_conf_resource_id': '1426807827'
                        },
                        job_params={'job_name_parameter': AUTOSHOOT_TASKLET_NAME}
                    )
                )

                return branch_part

            @property
            def _release(self):
                stages = [
                    rm_const.ReleaseStatus.testing,
                    rm_const.ReleaseStatus.prestable,
                    rm_const.ReleaseStatus.stable
                ]
                job_arrows = [(
                    jg_job_triggers.JobTriggerBuild(
                        job_name_parameter='binaries',
                        parent_job_data=(
                            jg_job_data.ParentDataDict(
                                'component_resources',
                                'quasar_backend_binaries',
                                'QUASAR_BACKEND_BINARIES',
                            ),
                        )
                    )
                )]

                release_part = super(self.__class__, self)._release
                for release_stage in stages:
                    release_part.append(jg_release.JobGraphElementReleaseBranched(
                        release_to=release_stage,
                        job_arrows=job_arrows,
                    ))
                    release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                        release_to=release_stage,
                    ))

                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def ignore_match(self):
                """List of test names which should be ignored in branch db"""
                return super(self.__class__, self).ignore_match + [
                    "QUASAR_BACKEND_BUILD",
                    "QUASAR_BACKEND_BUILD_PORTO_LAYER",
                ]

            deactivate = [
                rm_const.JobTypes.rm_job_name(
                    rm_const.JobTypes.ACTION_RELEASE,
                    COMPONENT_NAME.upper(),
                    rm_const.ReleaseStatus.testing
                ),
                rm_const.JobTypes.rm_job_name(
                    rm_const.JobTypes.DEPLOY,
                    COMPONENT_NAME.upper(),
                    NANNY_RELEASE_TASKLET_NAME
                ),
                rm_const.JobTypes.rm_job_name(
                    rm_const.JobTypes.DEPLOY,
                    COMPONENT_NAME.upper(),
                    AUTOSHOOT_TASKLET_NAME
                ),

            ]  # jobs to deactivate on db clone

            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    rm_const.JobTypes.rm_job_name(
                        rm_const.JobTypes.ACTION_RELEASE,
                        COMPONENT_NAME.upper(),
                        rm_const.ReleaseStatus.testing): rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests
