# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals
from six import add_metaclass

import inspect

from sandbox.projects.release_machine.components import configs
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils

"""
This file contains special parametrized base configs for release machine configuration.
In these classes all the parameters from CfgParam will be replaced with derived values from final configuration classes.
It's based on metaclasses but allows to write quite declarative base and very concise final configs.

Usage of parametrized base config:
1. Base class for CfgParams must contain '__metaclass__ = AddParamName'
2. Base config must contain:
    __metaclass__ = PropagateCfgParamsToInners
        class CfgParams(CfgParamsBase):
            pass
3. Any subtypes with usage of preferences from config must contain:
    class CfgParams(CfgParamsBase):
        pass
4. Final config must be inherited from base config and FinalConfig class

This allows the use of values of final config in a base config (and its inner types) in the following ways:
    1. self.CfgParams.PREFERENCE_NAME
    2. class_field_name = CfgParams.PREFERENCE_NAME  (ONLY for class fields without any containers or calculations)
"""


class PropagateCfgParamsToInners(type):
    """
    In classes with this metaclass IF it's derived from FinalConfig:
    1. CfgParams will be finalized: every NoneParamValue will be replaced with its default value
       (or exception will be raised)
    2. All inner types with attributes CfgParams will be replaced with a class derived from this type
       (so a copy of the inner type will be places instead of the original inner type)
    2. All attributes in inner type copies with a name 'CfgParams' will be replaced with the finalized CfgParams
       (via cascading for types with any CfgParams attribute)
    3. All attributes this class and inner types with value type NoneParamValue will be replaces with the values
       from finalized CfgParams (via cascading for types with any CfgParams attribute)
    """
    def finalize_config(cls):
        config_attrs = inspect.getmembers(cls.CfgParams, lambda a: not inspect.isroutine(a))
        for param_name, param_value in config_attrs:
            if isinstance(param_value, NoneParamValue):
                if param_value._NoneParamValue__default_value is not None:
                    setattr(cls.CfgParams, param_name, param_value._NoneParamValue__default_value)
                else:
                    raise Exception('Parameter {} must be set in {}'.format(param_name, cls.CfgParams))

    def propagate(cls, subclass):
        if not hasattr(subclass, 'CfgParams'):
            return

        subclass.CfgParams = cls.CfgParams
        cls_attrs = inspect.getmembers(subclass, lambda a: not inspect.isroutine(a))
        for attr_name, attr_info in cls_attrs:
            if not attr_name.startswith('__') and attr_name != 'CfgParams':
                if isinstance(attr_info, NoneParamValue):
                    setattr(subclass, attr_name, getattr(cls.CfgParams, attr_info._NoneParamValue__param_name))
                elif hasattr(attr_info, 'CfgParams'):
                    # it's necessary to make a copy of the type because of replacement may change the original base type
                    attr_info_copy = type(attr_info.__name__, attr_info.__bases__, dict(attr_info.__dict__))
                    attr_info_copy.__module__ = subclass.__module__
                    cls.propagate(attr_info_copy)
                    setattr(subclass, attr_name, attr_info_copy)

    def __init__(cls, name, bases, attrs):
        super(PropagateCfgParamsToInners, cls).__init__(name, bases, attrs)
        if FinalConfig in bases:
            cls.finalize_config()
            cls.propagate(cls)


class AddParamName(type):
    """
    Saves the name of field to param name attribute of NoneParamValue
    """
    def __new__(cls, name, bases, attrs):
        for attr_name, attr_info in attrs.items():
            if isinstance(attr_info, NoneParamValue):
                attr_info._NoneParamValue__param_name = attr_name
        res = super(AddParamName, cls).__new__(cls, name, bases, attrs)
        return res


class NoneParamValue(object):
    def __init__(self, default_value=None):
        self.__param_name = None
        self.__default_value = default_value

    def __getattribute__(self, item):
        if item not in [
            '__class__', '__get__', '__set__', '_NoneParamValue__param_name', '_NoneParamValue__default_value'
        ]:
            raise Exception('Incorrect usage of {}'.format(self.__class__))
        return object.__getattribute__(self, item)


class FinalConfig(object):
    pass


QLOUD_ENV_BY_RELEASE_STATUS = {
    rm_const.ReleaseStatus.unstable: 'unstable',
    rm_const.ReleaseStatus.testing: 'testing',
    rm_const.ReleaseStatus.prestable: 'prestable',
    rm_const.ReleaseStatus.stable: 'production',
}


@add_metaclass(AddParamName)
class CfgParamsBase(object):

    TRUNK_TASK_OWNER = NoneParamValue('RASP')
    DISPLAY_NAME = NoneParamValue()
    NAME = NoneParamValue()

    RESPONSIBLE = NoneParamValue('monitorius')
    FOLLOWERS = NoneParamValue([
        'monitorius',
        'vasbur',
        'frangetta'
    ])

    DOCKER_IMAGE_REPOSITORY = NoneParamValue('rasp')
    PROJECT_PATHS = NoneParamValue([])
    OBSERVED_PATHS = NoneParamValue([])
    PROJECT_PACKAGE_PATH = NoneParamValue()
    APPLICATION_RESOURCE_NAME = NoneParamValue()

    TRUNK_AUTO_DEPLOY_STAGES = NoneParamValue([])
    RELEASE_AUTO_DEPLOY_STAGES = NoneParamValue([
        rm_const.ReleaseStatus.testing
    ])
    RELEASE_BUTTON_DEPLOY_STAGES = NoneParamValue([
        rm_const.ReleaseStatus.prestable,
        rm_const.ReleaseStatus.stable,
    ])

    QLOUD_PROJECT = NoneParamValue('rasp')
    QLOUD_APPLICATION = NoneParamValue()
    QLOUD_COMPONENT = NoneParamValue('*')
    QLOUD_COMPONENT_TO_WATCH = NoneParamValue('main')

    WIKI_PAGE = NoneParamValue()
    ST_QUEUE = NoneParamValue('RASPFRONT')
    ST_RELEASE_SUMMARY_TEMPLATE = NoneParamValue('Релиз {}')
    ST_COMPONENTS = NoneParamValue(u'Релизы расписаний'.encode('utf-8'))


@add_metaclass(PropagateCfgParamsToInners)
class RaspCfg(configs.ReferenceBranchedConfig):

    class CfgParams(CfgParamsBase):
        pass

    display_name = CfgParams.DISPLAY_NAME
    name = CfgParams.NAME
    responsible = CfgParams.RESPONSIBLE

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        class CfgParams(CfgParamsBase):
            pass

        trunk_task_owner = CfgParams.TRUNK_TASK_OWNER

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            class CfgParams(CfgParamsBase):
                pass

            BUILD_APIARGS = {'kill_timeout': 2700}

            @property
            def BUILD_CTX(self):
                return {
                    'package_type': 'docker',
                    'docker_push_image': True,
                    'docker_registry': 'registry.yandex.net',
                    'docker_image_repository': self.CfgParams.DOCKER_IMAGE_REPOSITORY,
                    'packages': self.CfgParams.PROJECT_PACKAGE_PATH,
                    'docker_user': 'robot-rasp',
                    'docker_token_vault_name': 'DOCKER_OAUTH_TOKEN',
                    'docker_build_network': 'host',
                    'resource_type': self.CfgParams.APPLICATION_RESOURCE_NAME,
                    'checkout': False,
                    'use_aapi_fuse': True,
                    'ya_yt_store': False,
                    'ignore_recurses': True,
                }

            def _qloud_environment(self, rm_release_status):
                return '.'.join(
                    [
                        self.CfgParams.QLOUD_PROJECT,
                        self.CfgParams.QLOUD_APPLICATION,
                        QLOUD_ENV_BY_RELEASE_STATUS[rm_release_status],
                        self.CfgParams.QLOUD_COMPONENT,
                    ]
                )

            @property
            def _trunk_part(self):
                # Тут и далее используется явный вызов базового класса.
                # Использовать super(self.__class__, self)._trunk_part нельзя т.к. это приведет к бесконечной рекурсии
                # при наличии наследника.
                # Код super(RaspCfg.Testenv.JobGraph, self)._trunk_part тоже не работает, возможно изза метакласса.
                default_trunk_part = configs.ReferenceBranchedConfig.Testenv.JobGraph._trunk_part.fget(self)
                trunk_part = []
                for release_stage in self.CfgParams.TRUNK_AUTO_DEPLOY_STAGES:
                    trunk_part += [
                        jg_build.JobGraphElementBuildTrunk(
                            task_name='YA_PACKAGE',
                            job_params={
                                'apiargs': self.BUILD_APIARGS,
                            },
                            build_item='TRUNK',
                            ctx=self.BUILD_CTX,
                            out={self.CfgParams.APPLICATION_RESOURCE_NAME: 10},
                        ),
                        jg_release.JobGraphElementReleaseBase(
                            task_name='RELEASE_RM_COMPONENT_2',
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataCtx(
                                            input_key='registry_name',
                                            output_key='output_resource_version',
                                        )
                                    ),
                                    job_name_parameter='TRUNK',
                                ),
                            ),
                            job_params={
                                'frequency': (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 10),
                                'should_add_to_db': jg_utils.should_add_to_db_trunk,
                                'observed_paths': self.CfgParams.PROJECT_PATHS or self.CfgParams.OBSERVED_PATHS,
                                'ctx': {
                                    'deploy_system': 'qloud',
                                    'qloud_vault_owner': 'RASP',
                                    'qloud_vault_name': 'rasp-qloud-oauth',
                                    'qloud_environment': self._qloud_environment(release_stage),
                                }
                            },
                        ),
                    ]
                return default_trunk_part + trunk_part

            @property
            def _branch_part(self):
                default_branch_part = configs.ReferenceBranchedConfig.Testenv.JobGraph._branch_part.fget(self)
                branch_part = [
                    jg_build.JobGraphElementBuildBranched(
                        task_name='YA_PACKAGE',
                        job_params={
                            'apiargs': self.BUILD_APIARGS,
                        },
                        ctx=self.BUILD_CTX,
                        out={self.CfgParams.APPLICATION_RESOURCE_NAME: 60},
                    )
                ]
                return default_branch_part + branch_part

            def _create_release_element(self, release_stage, auto_deploy=False):
                job_params = {
                    'frequency': (jg_utils.TestFrequency.LAZY, None),
                    'observed_paths': self.CfgParams.PROJECT_PATHS or self.CfgParams.OBSERVED_PATHS,
                    'ctx': {
                        'deploy_system': 'qloud',
                        'qloud_vault_owner': 'RASP',
                        'qloud_vault_name': 'rasp-qloud-oauth',
                        'qloud_environment': self._qloud_environment(release_stage),
                    }
                }
                if auto_deploy:
                    # one time autodeploy (next deploy by changes after almost "infinity" minutes)
                    job_params['frequency'] = (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 1000000000)
                else:
                    job_params['frequency'] = (jg_utils.TestFrequency.LAZY, None)

                return jg_release.JobGraphElementReleaseBranched(
                    task_name='RELEASE_RM_COMPONENT_2',
                    release_to=release_stage,
                    job_arrows=(
                        jg_job_triggers.JobTriggerBuild(
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key='registry_name',
                                    output_key='output_resource_version',
                                ),
                                jg_job_data.ParentDataDict(
                                    input_key="component_resources",
                                    dict_key=self.CfgParams.APPLICATION_RESOURCE_NAME,
                                    resource_name=self.CfgParams.APPLICATION_RESOURCE_NAME,
                                )
                            )
                        ),
                        jg_job_triggers.JobTriggerNewTag([
                            jg_job_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                            jg_job_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
                        ]),
                    ),
                    job_params=job_params,
                )

            @property
            def _release(self):
                default_release_part = configs.ReferenceBranchedConfig.Testenv.JobGraph._release.fget(self)
                release_part = []
                for release_stage in self.CfgParams.RELEASE_AUTO_DEPLOY_STAGES:
                    release_part.append(self._create_release_element(release_stage, True))
                    release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                        release_to=release_stage,
                        job_arrows=(
                            jg_job_triggers.JobTriggerRelease(job_name_parameter=release_stage),
                        ),
                    ))
                for release_stage in self.CfgParams.RELEASE_BUTTON_DEPLOY_STAGES:
                    release_part.append(self._create_release_element(release_stage, False))
                    release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                        release_to=release_stage,
                        job_arrows=(
                            jg_job_triggers.JobTriggerRelease(job_name_parameter=release_stage),
                        ),
                    ))
                return default_release_part + release_part

    class Releases(configs.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        class CfgParams(CfgParamsBase):
            pass

        @property
        def resources_info(self):
            return [
                configs.ReleasedResourceInfo(
                    name=self.CfgParams.APPLICATION_RESOURCE_NAME,
                    resource_type=self.CfgParams.APPLICATION_RESOURCE_NAME,
                    deploy=[configs.DeployServicesInfo(services=[
                        '.'.join(map(str, [
                            self.CfgParams.QLOUD_PROJECT,
                            self.CfgParams.QLOUD_APPLICATION,
                            QLOUD_ENV_BY_RELEASE_STATUS[rm_const.ReleaseStatus.stable],
                            self.CfgParams.QLOUD_COMPONENT_TO_WATCH
                        ]))
                    ])]
                ),
            ]

        allow_robots_to_release_stable = True
        release_followers_permanent = CfgParams.FOLLOWERS
        wait_for_deploy_time_sec = 30 * 60  # 30 min
        deploy_system = rm_const.DeploySystem.qloud

        token_vault_owner = "RASP"
        token_vault_name = "rasp-qloud-oauth"

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""
        class CfgParams(CfgParamsBase):
            pass

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            class CfgParams(CfgParamsBase):
                pass

            chats = [
                'rasp',
            ]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            class CfgParams(CfgParamsBase):
                pass

            assignee = CfgParams.RESPONSIBLE
            use_task_author_as_assignee = True
            add_commiters_as_followers = True
            queue = CfgParams.ST_QUEUE
            dev_queue = CfgParams.ST_QUEUE
            summary_template = CfgParams.ST_RELEASE_SUMMARY_TEMPLATE
            followers = CfgParams.FOLLOWERS
            components = CfgParams.ST_COMPONENTS
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        class CfgParams(CfgParamsBase):
            pass

        wiki_page = CfgParams.WIKI_PAGE
        ya_make_targets = CfgParams.PROJECT_PATHS
        observed_paths = CfgParams.OBSERVED_PATHS
        svn_paths_filter = configs.ChangelogPathsFilter(rm_const.PermissionType.ALLOWED, ['arcadia/travel'])
        add_to_release_notes = True
