# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals
from six import add_metaclass

import re

from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.configs.rasp._base as base


DEPLOY_ENV_BY_RELEASE_STATUS = {
    rm_const.ReleaseStatus.unstable: 'unstable',
    rm_const.ReleaseStatus.testing: 'testing',
    rm_const.ReleaseStatus.prestable: 'prestable',
    rm_const.ReleaseStatus.stable: 'production',
}
NoneParamValue = base.NoneParamValue
FinalConfig = base.FinalConfig


@add_metaclass(base.AddParamName)
class DeployCfgParamsBase(object):

    TRUNK_TASK_OWNER = NoneParamValue('RASP')
    DISPLAY_NAME = NoneParamValue()
    NAME = NoneParamValue()

    RESPONSIBLE = NoneParamValue('monitorius')
    FOLLOWERS = NoneParamValue([
        'monitorius',
        'vasbur',
        'frangetta'
    ])

    DOCKER_IMAGE_REPOSITORY = NoneParamValue('rasp')
    PROJECT_PATHS = NoneParamValue([])
    OBSERVED_PATHS = NoneParamValue([])
    PROJECT_PACKAGE_PATH = NoneParamValue()
    APPLICATION_RESOURCE_NAME = NoneParamValue()

    TRUNK_AUTO_DEPLOY_STAGES = NoneParamValue([])
    RELEASE_AUTO_DEPLOY_STAGES = NoneParamValue([
        rm_const.ReleaseStatus.testing
    ])
    RELEASE_BUTTON_DEPLOY_STAGES = NoneParamValue([
        rm_const.ReleaseStatus.prestable,
        rm_const.ReleaseStatus.stable,
    ])

    DEPLOY_PROJECT = NoneParamValue('rasp')
    DEPLOY_APPLICATION = NoneParamValue()

    WIKI_PAGE = NoneParamValue()
    ST_QUEUE = NoneParamValue('RASPFRONT')
    ST_RELEASE_SUMMARY_TEMPLATE = NoneParamValue('Релиз {}')
    ST_COMPONENTS = NoneParamValue(u'Релизы расписаний'.encode('utf-8'))


def get_deploy_stage_name(deploy_project, deploy_application, rm_release_status):
    env = DEPLOY_ENV_BY_RELEASE_STATUS[rm_release_status]
    return '{}-{}-{}'.format(deploy_project, deploy_application, env)


@add_metaclass(base.PropagateCfgParamsToInners)
class RaspDeployCfg(configs.ReferenceBranchedConfig):

    class CfgParams(DeployCfgParamsBase):
        pass

    display_name = CfgParams.DISPLAY_NAME
    name = CfgParams.NAME
    responsible = CfgParams.RESPONSIBLE

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        class CfgParams(DeployCfgParamsBase):
            pass

        trunk_task_owner = CfgParams.TRUNK_TASK_OWNER

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            class CfgParams(DeployCfgParamsBase):
                pass

            BUILD_APIARGS = {'kill_timeout': 2700}

            @property
            def BUILD_CTX(self):
                return {
                    'package_type': 'docker',
                    'docker_push_image': True,
                    'docker_registry': 'registry.yandex.net',
                    'docker_image_repository': self.CfgParams.DOCKER_IMAGE_REPOSITORY,
                    'packages': self.CfgParams.PROJECT_PACKAGE_PATH,
                    'docker_user': 'robot-rasp',
                    'docker_token_vault_name': 'DOCKER_OAUTH_TOKEN',
                    'docker_build_network': 'host',
                    'resource_type': self.CfgParams.APPLICATION_RESOURCE_NAME,
                    'checkout': False,
                    'use_aapi_fuse': True,
                    'ya_yt_store': False,
                    'ignore_recurses': True,
                    'release_to_ya_deploy': True,
                    'yp_token_vault': 'TRAVEL_YP_OAUTH_TOKEN'
                }

            @property
            def _trunk_part(self):
                # Тут и далее используется явный вызов базового класса.
                # Использовать super(self.__class__, self)._trunk_part нельзя т.к. это приведет к бесконечной рекурсии
                # при наличии наследника.
                # Код super(RaspCfg.Testenv.JobGraph, self)._trunk_part тоже не работает, возможно изза метакласса.
                default_trunk_part = configs.ReferenceBranchedConfig.Testenv.JobGraph._trunk_part.fget(self)
                trunk_part = []
                for release_stage in self.CfgParams.TRUNK_AUTO_DEPLOY_STAGES:
                    trunk_part += [
                        jg_build.JobGraphElementBuildTrunk(
                            task_name='YA_PACKAGE',
                            job_params={
                                'apiargs': self.BUILD_APIARGS,
                            },
                            build_item='TRUNK',
                            ctx=self.BUILD_CTX,
                            out={self.CfgParams.APPLICATION_RESOURCE_NAME: 10},
                        ),
                        jg_release.JobGraphElementReleaseBase(
                            task_name='RELEASE_RM_COMPONENT_2',
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataCtx(
                                            input_key='registry_name',
                                            output_key='output_resource_version',
                                        )
                                    ),
                                    job_name_parameter='TRUNK',
                                ),
                            ),
                            job_params={
                                'frequency': (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 10),
                                'should_add_to_db': jg_utils.should_add_to_db_trunk,
                                'observed_paths': self.CfgParams.PROJECT_PATHS or self.CfgParams.OBSERVED_PATHS,
                                'ctx': {
                                    'deploy_system': rm_const.DeploySystem.ya_deploy.name
                                }
                            },
                        ),
                    ]
                return default_trunk_part + trunk_part

            @property
            def _branch_part(self):
                default_branch_part = configs.ReferenceBranchedConfig.Testenv.JobGraph._branch_part.fget(self)
                branch_part = [
                    jg_build.JobGraphElementBuildBranched(
                        task_name='YA_PACKAGE',
                        job_params={
                            'apiargs': self.BUILD_APIARGS,
                        },
                        ctx=self.BUILD_CTX,
                        out={self.CfgParams.APPLICATION_RESOURCE_NAME: 60},
                    )
                ]
                return default_branch_part + branch_part

            def _create_release_element(self, release_stage, auto_deploy=False):
                job_params = {
                    'frequency': (jg_utils.TestFrequency.LAZY, None),
                    'observed_paths': self.CfgParams.PROJECT_PATHS or self.CfgParams.OBSERVED_PATHS,
                    'ctx': {
                        'deploy_system': rm_const.DeploySystem.ya_deploy.name
                    }
                }
                if auto_deploy:
                    # one time autodeploy (next deploy by changes after almost "infinity" minutes)
                    job_params['frequency'] = (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 1000000000)
                else:
                    job_params['frequency'] = (jg_utils.TestFrequency.LAZY, None)

                return jg_release.JobGraphElementReleaseBranched(
                    task_name='RELEASE_RM_COMPONENT_2',
                    release_to=release_stage,
                    job_arrows=(
                        jg_job_triggers.JobTriggerBuild(
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key='registry_name',
                                    output_key='output_resource_version',
                                ),
                                jg_job_data.ParentDataDict(
                                    input_key="component_resources",
                                    dict_key=self.CfgParams.APPLICATION_RESOURCE_NAME,
                                    resource_name=self.CfgParams.APPLICATION_RESOURCE_NAME,
                                )
                            )
                        ),
                        jg_job_triggers.JobTriggerNewTag([
                            jg_job_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                            jg_job_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
                        ]),
                    ),
                    job_params=job_params,
                )

            @property
            def _release(self):
                default_release_part = configs.ReferenceBranchedConfig.Testenv.JobGraph._release.fget(self)
                release_part = []
                for release_stage in self.CfgParams.RELEASE_AUTO_DEPLOY_STAGES:
                    release_part.append(self._create_release_element(release_stage, True))
                    release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                        release_to=release_stage,
                        job_arrows=(
                            jg_job_triggers.JobTriggerRelease(job_name_parameter=release_stage),
                        ),
                    ))
                for release_stage in self.CfgParams.RELEASE_BUTTON_DEPLOY_STAGES:
                    release_part.append(self._create_release_element(release_stage, False))
                    release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                        release_to=release_stage,
                        job_arrows=(
                            jg_job_triggers.JobTriggerRelease(job_name_parameter=release_stage),
                        ),
                    ))
                return default_release_part + release_part

    class Releases(configs.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        class CfgParams(DeployCfgParamsBase):
            pass

        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name=self.CfgParams.APPLICATION_RESOURCE_NAME,
                    data=ri.SandboxResourceData(
                        self.CfgParams.APPLICATION_RESOURCE_NAME,
                        docker_image_data_re=re.compile(r'stable-(?P<branch>.*)-(?P<tag>.*)\.(?P<revision>.*)\.(?P<task_id>.*)')
                    ),
                    deploy_infos=[ri.YaDeployInfo(ri.DeployService(
                        get_deploy_stage_name(
                            self.CfgParams.DEPLOY_PROJECT,
                            self.CfgParams.DEPLOY_APPLICATION,
                            rm_const.ReleaseStatus.stable
                        )
                    ))],
                )
            ]

        allow_robots_to_release_stable = True
        release_followers_permanent = CfgParams.FOLLOWERS
        wait_for_deploy_time_sec = 30 * 60  # 30 min
        deploy_system = rm_const.DeploySystem.ya_deploy

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""
        class CfgParams(DeployCfgParamsBase):
            pass

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            class CfgParams(DeployCfgParamsBase):
                pass

            chats = [
                'rasp',
            ]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            class CfgParams(DeployCfgParamsBase):
                pass

            assignee = CfgParams.RESPONSIBLE
            use_task_author_as_assignee = True
            add_commiters_as_followers = True
            queue = CfgParams.ST_QUEUE
            dev_queue = CfgParams.ST_QUEUE
            summary_template = CfgParams.ST_RELEASE_SUMMARY_TEMPLATE
            followers = CfgParams.FOLLOWERS
            components = CfgParams.ST_COMPONENTS
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        class CfgParams(DeployCfgParamsBase):
            pass

        wiki_page = CfgParams.WIKI_PAGE
        ya_make_targets = CfgParams.PROJECT_PATHS
        observed_paths = CfgParams.OBSERVED_PATHS
        svn_paths_filter = configs.ChangelogPathsFilter(rm_const.PermissionType.ALLOWED, ['arcadia/travel'])
        add_to_release_notes = True
