# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

# alias for base class module is obligatory because of testenv specifics
from sandbox.projects.release_machine.components import configs
import sandbox.projects.release_machine.components.configs.rasp._base as base
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


class RaspAdminCfg(base.RaspCfg, base.FinalConfig):
    class CfgParams(base.RaspCfg.CfgParams):
        DISPLAY_NAME = 'Rasp Admin'
        NAME = 'rasp_admin'

        PROJECT_PACKAGE_PATH = 'travel/rasp/admin/pkg.json'
        APPLICATION_RESOURCE_NAME = 'RASP_ADMIN_BINARY'

        # sandbox release
        QLOUD_APPLICATION = None

        PROJECT_PATHS = ['travel/rasp/admin']
        WIKI_PAGE = 'raspisanija/releases/admin/'
        ST_RELEASE_SUMMARY_TEMPLATE = 'Релиз rasp_admin {}'

        RELEASE_AUTO_DEPLOY_STAGES = [
            rm_const.ReleaseStatus.unstable
        ]

        RELEASE_BUTTON_DEPLOY_STAGES = [
            rm_const.ReleaseStatus.testing,
            rm_const.ReleaseStatus.prestable,
            rm_const.ReleaseStatus.stable
        ]

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        class CfgParams(base.CfgParamsBase):
            pass

        trunk_task_owner = CfgParams.TRUNK_TASK_OWNER

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            class CfgParams(base.CfgParamsBase):
                pass

            BUILD_APIARGS = {'kill_timeout': 3600 * 3}

            @property
            def BUILD_CTX(self):
                return {
                    'package_type': 'docker',
                    'docker_push_image': True,
                    'docker_registry': 'registry.yandex.net',
                    'docker_image_repository': self.CfgParams.DOCKER_IMAGE_REPOSITORY,
                    'packages': self.CfgParams.PROJECT_PACKAGE_PATH,
                    'docker_user': 'robot-rasp',
                    'docker_token_vault_name': 'DOCKER_OAUTH_TOKEN',
                    'docker_build_network': 'host',
                    'resource_type': self.CfgParams.APPLICATION_RESOURCE_NAME,
                    'checkout': False,
                    'use_aapi_fuse': True,
                    'ya_yt_store': False,
                    'ignore_recurses': True,

                    'release_to_ya_deploy': True,
                    'yp_token_vault': 'TRAVEL_YP_OAUTH_TOKEN'
                }

            @property
            def _branch_part(self):
                default_branch_part = super(self.__class__, self)._branch_part
                branch_part = [
                    jg_build.JobGraphElementBuildBranched(
                        task_name='RASP_ADMIN_YA_PACKAGE',
                        job_params={
                            'apiargs': self.BUILD_APIARGS,
                        },
                        ctx=self.BUILD_CTX,
                        out={self.CfgParams.APPLICATION_RESOURCE_NAME: 60},
                    )
                ]
                return default_branch_part + branch_part

            def _create_release_element(self, release_stage, auto_deploy=False):
                job_params = {
                    'frequency': (jg_utils.TestFrequency.LAZY, None),
                    'observed_paths': self.CfgParams.PROJECT_PATHS,
                    'ctx': {
                        "deploy_system": rm_const.DeploySystem.sandbox.name
                    }
                }

                if auto_deploy:
                    # one time autodeploy (next deploy by changes after almost "infinity" minutes)
                    job_params['frequency'] = (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 1000000000)

                return jg_release.JobGraphElementReleaseBranched(
                    task_name='RELEASE_RM_COMPONENT_2',
                    release_to=release_stage,
                    job_arrows=(
                        jg_job_triggers.JobTriggerBuild(
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key='registry_name',
                                    output_key='output_resource_version',
                                ),
                                jg_job_data.ParentDataDict(
                                    input_key="component_resources",
                                    dict_key=self.CfgParams.APPLICATION_RESOURCE_NAME,
                                    resource_name=self.CfgParams.APPLICATION_RESOURCE_NAME,
                                )
                            )
                        ),
                        jg_job_triggers.JobTriggerNewTag([
                            jg_job_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                            jg_job_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
                        ]),
                    ),
                    job_params=job_params,
                )

            @property
            def _release(self):
                default_release_part = super(self.__class__, self)._release
                release_part = []
                for release_stage in self.CfgParams.RELEASE_AUTO_DEPLOY_STAGES:
                    release_part.append(self._create_release_element(release_stage, True))
                    release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                        release_to=release_stage,
                        job_arrows=(
                            jg_job_triggers.JobTriggerRelease(job_name_parameter=release_stage),
                        ),
                    ))
                for release_stage in self.CfgParams.RELEASE_BUTTON_DEPLOY_STAGES:
                    release_part.append(self._create_release_element(release_stage, False))
                    release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                        release_to=release_stage,
                        job_arrows=(
                            jg_job_triggers.JobTriggerRelease(job_name_parameter=release_stage),
                        ),
                    ))
                return default_release_part + release_part

    class Releases(configs.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        class CfgParams(base.CfgParamsBase):
            pass

        @property
        def resources_info(self):
            return [
                configs.ReleasedResourceInfo(
                    name=self.CfgParams.APPLICATION_RESOURCE_NAME,
                    resource_type=self.CfgParams.APPLICATION_RESOURCE_NAME,
                    deploy=[
                        configs.DeployServicesInfo(
                            services=['rasp-admin-service-testing'],
                            level=rm_const.ReleaseStatus.unstable
                        ),
                        configs.DeployServicesInfo(
                            services=['rasp-admin-work-testing'],
                            level=rm_const.ReleaseStatus.testing
                        ),
                        configs.DeployServicesInfo(
                            services=['rasp-admin-service-production'],
                            level=rm_const.ReleaseStatus.prestable
                        ),
                        configs.DeployServicesInfo(
                            services=['rasp-admin-work-production'],
                            level=rm_const.ReleaseStatus.stable
                        ),
                    ]
                )
            ]

        allow_robots_to_release_stable = True
        release_followers_permanent = CfgParams.FOLLOWERS
        wait_for_deploy_time_sec = 30 * 60
        deploy_system = rm_const.DeploySystem.ya_deploy
