# -*- coding: utf-8 -*-

import os

from sandbox.projects.common.constants import constants as common_const
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import release_block
from sandbox.projects.release_machine.components.config_core import release_approvements
import sandbox.projects.release_machine.helpers.arcadia as rm_arc
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.job_graph_element as jg_element
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.stages.run_stage as jg_run
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


def _merge(x, y):
    # wait for python 3.9: implements `x | y`
    result = dict(x)
    result.update(y)
    return result


def _build_job_params(observed_paths, disk_space_gb=10):
    return {
        "apiargs": {
            "requirements": {
                "disk_space": disk_space_gb * (1024 ** 3),
            },
        },
        "observed_paths": observed_paths,
    }


class ReleaseMachineCfg(configs.ReferenceBranchedConfig):
    name = "release_machine"
    responsible = configs.Responsible(
        abc=configs.Abc(service_name="releasemachine", component_id=2676),
        login="ilyaturuntaev",
    )

    class ReleaseViewer(configs.ReferenceBranchedConfig.ReleaseViewer):
        kpi_alert = 7
        kpi_alert_skip_weekends = True

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_db = "release-machine-trunk"
        trunk_task_owner = "RELEASE_MACHINE"
        branch_db_template = "release-machine-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            OBSERVED_PATHS = [
                "release_machine/release_machine/bin",
                "release_machine/release_machine/proto",
                "release_machine/release_machine/src",
            ]
            _BASE_BUILD_PARAMS = {
                "result_single_file": True,
                common_const.USE_AAPI_FUSE: True,
                common_const.USE_ARC_INSTEAD_OF_AAPI: True,
                common_const.ALLOW_AAPI_FALLBACK: True,
                "binary_executor_release_type": "stable",
            }
            BUILD_PARAMS = _merge(_BASE_BUILD_PARAMS, {
                "arts": "release_machine/release_machine/bin/release_machine",
                "targets": "release_machine/release_machine/bin/release_machine",
                "result_rt": "RELEASE_MACHINE_BINARY",
                "result_rd": "RM on horadric2",
            })

            SB_POLLER_OBSERVED_PATHS = [
                "release_machine/release_machine/proto",
                "release_machine/release_machine/src",
            ]
            SB_POLLER_BUILD_PARAMS = _merge(_BASE_BUILD_PARAMS, {
                "arts": "release_machine/release_machine/src/sb_poller/bin/sb_poller",
                "targets": "release_machine/release_machine/src/sb_poller/bin/sb_poller",
                "result_rt": "SB_POLLER_BINARY",
                "result_rd": "RM SB Poller",
            })

            CI_POLLER_OBSERVED_PATHS = [
                "release_machine/release_machine/proto",
                "release_machine/release_machine/src",
            ]
            CI_POLLER_BUILD_PARAMS = _merge(_BASE_BUILD_PARAMS, {
                "arts": "release_machine/release_machine/src/ci_poller/bin/ci_poller",
                "targets": "release_machine/release_machine/src/ci_poller/bin/ci_poller",
                "result_rt": "CI_POLLER_BINARY",
                "result_rd": "RM CI Poller",
            })

            SOLOMON_AGENT_OBSERVED_PATHS = [
                "solomon/agent",
            ]
            SOLOMON_AGENT_BUILD_PARAMS = _merge(_BASE_BUILD_PARAMS, {
                "arts": "solomon/agent/bin/solomon-agent",
                "targets": "solomon/agent/bin/solomon-agent",
                "result_rt": "RM_SOLOMON_AGENT_BINARY",
                "result_rd": "RM Solomon Agent",
            })

            @property
            def _trunk_part(self):
                this_trunk_part = [
                    jg_test.JobGraphElementMetaTestTrunk(
                        job_params={
                            "observed_paths": self.OBSERVED_PATHS + [
                                "release_machine/release_machine/tests",
                            ],
                            "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 60),
                        },
                        ctx={
                            "env_vars": " ".join([
                                "DB_URL_TEST=$(vault:value:SEARCH-RELEASERS:release_engine_db_test_url)",
                                "DB_URL=$(vault:value:SEARCH-RELEASERS:release_engine_db_test_url)",
                            ]),
                            "targets": "release_machine/release_machine",
                            common_const.YA_YT_STORE: False,
                        }
                    ),
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="KOSHER_YA_MAKE",
                        build_item="BACKEND_TRUNK",
                        out={
                            "RELEASE_MACHINE_BINARY": 20,
                        },
                        ctx=self.BUILD_PARAMS,
                        job_params=_build_job_params(observed_paths=self.OBSERVED_PATHS, disk_space_gb=20),
                    ),
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="KOSHER_YA_MAKE",
                        build_item="SB_POLLER_TRUNK",
                        out={
                            "SB_POLLER_BINARY": 20,
                        },
                        ctx=self.SB_POLLER_BUILD_PARAMS,
                        job_params=_build_job_params(observed_paths=self.SB_POLLER_OBSERVED_PATHS, disk_space_gb=5),
                    ),
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="KOSHER_YA_MAKE",
                        build_item="CI_POLLER_TRUNK",
                        out={
                            "CI_POLLER_BINARY": 20,
                        },
                        ctx=self.CI_POLLER_BUILD_PARAMS,
                        job_params=_build_job_params(observed_paths=self.CI_POLLER_OBSERVED_PATHS, disk_space_gb=2),
                    ),
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="KOSHER_YA_MAKE",
                        build_item="SOLOMON_AGENT_TRUNK",
                        out={
                            "RM_SOLOMON_AGENT_BINARY": 20,
                        },
                        ctx=self.SOLOMON_AGENT_BUILD_PARAMS,
                        job_params=_build_job_params(observed_paths=self.SOLOMON_AGENT_OBSERVED_PATHS, disk_space_gb=1),
                    ),
                    jg_release.JobGraphElementReleaseBase(
                        task_name="RELEASE_RM_COMPONENT_2",
                        release_to=rm_const.ReleaseStatus.testing,
                        release_item="BACKEND_TRUNK",
                        job_params={
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 90),
                            "observed_paths": self.OBSERVED_PATHS,
                            "filter_targets": [
                                "release_machine/release_machine/src/services",
                            ],
                            "ctx": {
                                "deploy_system": rm_const.DeploySystem.nanny.name,
                            }
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=[
                                    jg_job_data.ParentDataDict(
                                        input_key="component_resources",
                                        dict_key="release_machine_backend",
                                        resource_name="RELEASE_MACHINE_BINARY",
                                    ),
                                ],
                                job_name_parameter="BACKEND_TRUNK",
                            ),
                            jg_job_triggers.JobTriggerRun(
                                job_name_parameter="MIGRATION_DEV",
                            ),
                        ),
                    ),
                    jg_release.JobGraphElementReleaseBase(
                        task_name="RELEASE_RM_COMPONENT_2",
                        release_to=rm_const.ReleaseStatus.testing,
                        release_item="SB_POLLER_TRUNK",
                        job_params={
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 90),
                            "observed_paths": self.SB_POLLER_OBSERVED_PATHS,
                            "filter_targets": [
                                "release_machine/release_machine/src/sb_poller",
                            ],
                            "ctx": {
                                "deploy_system": rm_const.DeploySystem.nanny.name,
                            }
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=[
                                    jg_job_data.ParentDataDict(
                                        input_key="component_resources",
                                        dict_key="release_machine_sb_poller",
                                        resource_name="SB_POLLER_BINARY",
                                    ),
                                ],
                                job_name_parameter="SB_POLLER_TRUNK",
                            ),
                            jg_job_triggers.JobTriggerRun(
                                job_name_parameter="MIGRATION_DEV",
                            ),
                            jg_job_triggers.JobTriggerRun(
                                job_name_parameter="MIGRATION_TEST",
                            ),
                        ),
                    ),
                    jg_release.JobGraphElementReleaseBase(
                        task_name="RELEASE_RM_COMPONENT_2",
                        release_to=rm_const.ReleaseStatus.testing,
                        release_item="CI_POLLER_TRUNK",
                        job_params={
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 90),
                            "observed_paths": self.CI_POLLER_OBSERVED_PATHS,
                            "filter_targets": [
                                "release_machine/release_machine/src/ci_poller",
                            ],
                            "ctx": {
                                "deploy_system": rm_const.DeploySystem.nanny.name,
                            }
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=[
                                    jg_job_data.ParentDataDict(
                                        input_key="component_resources",
                                        dict_key="release_machine_ci_poller",
                                        resource_name="CI_POLLER_BINARY",
                                    ),
                                ],
                                job_name_parameter="CI_POLLER_TRUNK",
                            ),
                            jg_job_triggers.JobTriggerRun(
                                job_name_parameter="MIGRATION_DEV",
                            ),
                            jg_job_triggers.JobTriggerRun(
                                job_name_parameter="MIGRATION_TEST",
                            ),
                        ),
                    ),
                    jg_release.JobGraphElementReleaseBase(
                        task_name="RELEASE_RM_COMPONENT_2",
                        release_to=rm_const.ReleaseStatus.testing,
                        release_item="SOLOMON_AGENT_TRUNK",
                        job_params={
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 90),
                            "observed_paths": self.SOLOMON_AGENT_OBSERVED_PATHS,
                            "filter_targets": [
                                "solomon/agent",
                            ],
                            "ctx": {
                                "deploy_system": rm_const.DeploySystem.nanny.name,
                            },
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 3 << 30,  # 3 GB
                                },
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=[
                                    jg_job_data.ParentDataDict(
                                        input_key="component_resources",
                                        dict_key="release_machine_solomon_agent",
                                        resource_name="RM_SOLOMON_AGENT_BINARY",
                                    ),
                                ],
                                job_name_parameter="SOLOMON_AGENT_TRUNK",
                            ),
                        ),
                    ),
                    jg_run.JobGraphElementRunTrunkCommon(
                        task_name="RUN_ALEMBIC_MIGRATIONS",
                        job_params={
                            "job_name_parameter": "MIGRATION_DEV",
                        },
                        ctx={
                            "db_url_yav_secret_uuid": "sec-01ejz9hcr8tq4mmwga719ecgt0",
                            "db_url_yav_secret_name": "db_url_dev",
                            "project_root": "release_machine/release_machine",
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key=common_const.ARCADIA_URL_KEY,
                                transform=lambda x, rm_config: "{svn_ssh_url}/arcadia@{revision}".format(
                                    svn_ssh_url=x.svn_ssh_url,
                                    revision=x.revision,
                                ),
                            ),
                        ),
                    ),
                    jg_run.JobGraphElementRunTrunkCommon(
                        task_name="RUN_ALEMBIC_MIGRATIONS",
                        job_params={
                            "job_name_parameter": "MIGRATION_TEST",
                        },
                        ctx={
                            "db_url_yav_secret_uuid": "sec-01ejz9hcr8tq4mmwga719ecgt0",
                            "db_url_yav_secret_name": "db_url_test",
                            "project_root": "release_machine/release_machine",
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key=common_const.ARCADIA_URL_KEY,
                                transform=lambda x, rm_config: "{svn_ssh_url}/arcadia@{revision}".format(
                                    svn_ssh_url=x.svn_ssh_url,
                                    revision=x.revision,
                                ),
                            ),
                        ),
                    ),
                    jg_test.JobGraphElementRunIceFlame(
                        config_file="release_machine__prod__solomon_agent.conf",
                        job_params={
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.DEFINED_BY_CODE, None),
                            "next_revision_custom_schedule": jg_utils.CustomSchedule(
                                time_interval_list=[(7, 9)],
                                seconds_from_last_run_commit=1.5 * 3600,
                                once_a_day=True,
                            ),
                        },
                        ctx={
                            "release_item_name": "release_machine_backend",
                            "service_name": "release_machine_prod",
                        },
                    ),
                ]
                return this_trunk_part + super(self.__class__, self)._trunk_part

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part

                test_env_vars = " ".join([
                    "DB_URL_TEST=$(vault:value:SEARCH-RELEASERS:release_engine_db_test_url)",
                    "DB_URL=$(vault:value:SEARCH-RELEASERS:release_engine_db_test_url)",
                ])

                branch_part.append(jg_build.JobGraphElementYaMakeBuildBranched(
                    build_item="BACKEND",
                    out={
                        "RELEASE_MACHINE_BINARY": 40,
                    },
                    ctx=self.BUILD_PARAMS,
                ))
                branch_part.append(jg_build.JobGraphElementYaMakeBuildBranched(
                    build_item="SB_POLLER",
                    out={
                        "SB_POLLER_BINARY": 40,
                    },
                    ctx=self.SB_POLLER_BUILD_PARAMS,
                ))
                branch_part.append(jg_build.JobGraphElementYaMakeBuildBranched(
                    build_item="CI_POLLER",
                    out={
                        "CI_POLLER_BINARY": 40,
                    },
                    ctx=self.CI_POLLER_BUILD_PARAMS,
                ))
                branch_part.append(jg_build.JobGraphElementYaMakeBuildBranched(
                    build_item="SOLOMON_AGENT",
                    out={
                        "RM_SOLOMON_AGENT_BINARY": 40,
                    },
                    ctx=self.SOLOMON_AGENT_BUILD_PARAMS,
                    job_params={
                        "apiargs": {
                            "requirements": {
                                "disk_space": 3 << 30,  # 3 GB
                            },
                        },
                    },
                ))
                branch_part.append(jg_build.JobGraphElementBuildBranched(
                    task_name="BUILD_RELEASE_ENGINE_UI",
                    build_item="UI",
                    out={
                        "RELEASE_ENGINE_UI": 40,
                    },
                    job_arrows=[
                        jg_arrows.GlobalResourceData("nodejs_archive", "NODE_JS_ARCHIVE"),
                        jg_arrows.ParentsData(
                            "checkout_arcadia_from_url",
                            triggers=[
                                jg_job_triggers.JobTriggerNewTag([
                                    jg_job_data.ParentDataOutput(
                                        "arc_url",
                                        "result_path",
                                        transform=lambda x, params: os.path.join(
                                            x, "arcadia/release_machine/release_machine/ui"
                                        ),
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        "arc_rev",
                                        "result_revision",
                                    )
                                ]),
                            ],
                            transform=lambda arc_url, arc_rev: rm_arc.construct_arc_url(
                                arc_url, arc_rev, add_arcadia=False
                            ),
                            override=True,
                        ),
                    ]
                ))
                branch_part.append(jg_test.JobGraphElementTestBranchCommon(
                    task_name="TEST_RELEASE_ENGINE_BACKEND",
                    job_arrows=[
                        jg_job_triggers.JobTriggerBuild(
                            parent_job_data=[
                                jg_job_data.ParentDataResource(
                                    input_key="release_machine_backend_binary",
                                    resource_name="RELEASE_MACHINE_BINARY",
                                ),
                            ],
                            job_name_parameter="BACKEND",
                        ),
                    ],
                ))
                branch_part.append(jg_run.JobGraphElementRunBranchCommon(
                    task_name="RUN_ALEMBIC_MIGRATIONS",
                    job_params={
                        "job_name_parameter": "MIGRATION",
                    },
                    ctx={
                        "db_url_yav_secret_uuid": "sec-01ejz9hcr8tq4mmwga719ecgt0",
                        "db_url_yav_secret_name": "db_url_prod",
                        "project_root": "release_machine/release_machine",
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerNewTag(
                            parent_job_data=[
                                jg_job_data.ParentDataOutput(
                                    input_key=common_const.ARCADIA_URL_KEY,
                                    output_key=common_const.ARCADIA_URL_KEY,
                                ),
                            ],
                        ),
                    ),
                ))

                branch_part.append(jg_test.JobGraphElementMetaTestTrunk(
                    job_params={
                        "observed_paths": self.OBSERVED_PATHS + [
                            "release_machine/release_machine/pytest",
                        ],
                        "should_add_to_db": {
                            "trunk": jg_utils.should_add_to_db_trunk,
                            "branch": jg_utils.should_add_to_db_branch,
                        },
                        "frequency": {
                            "trunk": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 10),
                            "branch": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        "job_name_parameter": "PYTEST_MAIN",
                    },
                    ctx={
                        "env_vars": test_env_vars,
                        "targets": "release_machine/release_machine/pytest",
                    },
                ))

                branch_part.append(jg_test.JobGraphElementMetaTestTrunk(
                    job_params={
                        "observed_paths": self.OBSERVED_PATHS + [
                            "release_machine/release_machine/tests/proto_dispatcher",
                        ],
                        "should_add_to_db": {
                            "trunk": jg_utils.should_add_to_db_trunk,
                            "branch": jg_utils.should_add_to_db_branch,
                        },
                        "frequency": {
                            "trunk": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 10),
                            "branch": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        "job_name_parameter": "EVENT_DISPATCHER",
                    },
                    ctx={
                        "env_vars": test_env_vars,
                        "targets": "release_machine/release_machine/tests/proto_dispatcher",
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerMetaTest(
                            job_name_parameter="PYTEST_MAIN",
                        ),
                    ),
                ))

                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release

                action_release_params = {
                    "release_to": rm_const.ReleaseStatus.stable,
                    "job_arrows": [
                        jg_arrows.JobTrigger(
                            job_type=rm_const.JobTypes.RELEASE,
                            job_name_parameter="UI",
                        ),
                    ]
                }

                release_params = {
                    "task_name": "RELEASE_RM_COMPONENT_2",
                    "release_to": rm_const.ReleaseStatus.stable,
                    "job_arrows": (
                        jg_job_triggers.JobTriggerBuild(
                            parent_job_data=[
                                jg_job_data.ParentDataDict(
                                    input_key="component_resources",
                                    dict_key="release_machine_backend",
                                    resource_name="RELEASE_MACHINE_BINARY",
                                ),
                            ],
                            job_name_parameter="BACKEND",
                        ),
                        jg_job_triggers.JobTriggerBuild(
                            parent_job_data=[
                                jg_job_data.ParentDataDict(
                                    input_key="component_resources",
                                    dict_key="release_machine_sb_poller",
                                    resource_name="SB_POLLER_BINARY",
                                ),
                            ],
                            job_name_parameter="SB_POLLER",
                        ),
                        jg_job_triggers.JobTriggerBuild(
                            parent_job_data=[
                                jg_job_data.ParentDataDict(
                                    input_key="component_resources",
                                    dict_key="release_machine_ci_poller",
                                    resource_name="CI_POLLER_BINARY",
                                ),
                            ],
                            job_name_parameter="CI_POLLER",
                        ),
                        jg_job_triggers.JobTriggerBuild(
                            parent_job_data=[
                                jg_job_data.ParentDataDict(
                                    input_key="component_resources",
                                    dict_key="release_machine_solomon_agent",
                                    resource_name="RM_SOLOMON_AGENT_BINARY",
                                ),
                            ],
                            job_name_parameter="SOLOMON_AGENT",
                        ),
                        jg_job_triggers.JobTriggerTestBranchCommon(),
                        jg_job_triggers.JobTriggerNewTag(
                            parent_job_data=[
                                jg_job_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                                jg_job_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
                            ],
                        ),
                        jg_job_triggers.JobTriggerRun(
                            job_name_parameter="MIGRATION",
                        ),
                    ),
                    "job_params": {
                        "apiargs": {
                            "kill_timeout": 15 * 60,  # 15 min
                        },
                        "ctx": {
                            "deploy_system": rm_const.DeploySystem.nanny.name,
                        }
                    }
                }

                release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                    job_params={
                        "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 42),
                    },
                    **action_release_params
                ))
                release_part.append(jg_release.JobGraphElementActionPushReleaseBranched(**action_release_params))
                release_part.append(jg_release.JobGraphElementReleaseBranched(**release_params))
                release_part.append(jg_release.JobGraphElementReleasePushBranched(**release_params))
                release_part.append(jg_element.JobGraphElement(
                    job_params={
                        "job_type": rm_const.JobTypes.RELEASE,
                        "task_name": "RELEASE_ZEPHYR_SERVICE_UI",
                        "should_add_to_db": jg_utils.should_add_to_db_branch,
                        "cancel_fallbehind_runs_on_fix": False,
                        "job_name_parameter": "UI",
                        "ctx": {"access_secret_key": "rm_prod_ui_access_key"}
                    },
                    job_arrows=[
                        jg_job_triggers.JobTriggerBuild(
                            parent_job_data=[
                                jg_job_data.ParentDataResource(
                                    input_key="resource",
                                    resource_name="RELEASE_ENGINE_UI",
                                ),
                            ],
                            job_name_parameter="UI",
                        ),
                        jg_job_triggers.JobTriggerTestBranchCommon(),
                        jg_arrows.ParentsData(
                            "version",
                            triggers=[
                                jg_arrows.ParamsData(
                                    input_key="release_num",
                                    transform=lambda x, rm_config: jg_utils.get_major_release_number(x, rm_config),
                                ),
                                jg_job_triggers.JobTriggerNewTag([
                                    jg_job_data.ParentDataOutput(
                                        input_key="tag_num",
                                        output_key="new_tag_number",
                                    )
                                ]),
                            ],
                            transform=lambda release_num, tag_num: "{}-{}".format(release_num, tag_num),
                        ),
                    ],
                ))
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def ignore_match(self):
                return super(self.__class__, self).ignore_match + [
                    "_BUILD__RELEASE_MACHINE__BACKEND_TRUNK",
                    "_BUILD__RELEASE_MACHINE__SB_POLLER_TRUNK",
                    "_BUILD__RELEASE_MACHINE__CI_POLLER_TRUNK",
                    "_RELEASE__RELEASE_MACHINE__BACKEND_TRUNK__TESTING",
                    "_RELEASE__RELEASE_MACHINE__SB_POLLER_TRUNK__TESTING",
                    "_RELEASE__RELEASE_MACHINE__CI_POLLER_TRUNK__TESTING",
                    "_META_TEST__RELEASE_MACHINE",
                ]

            @property
            def ignore_prefix(self):
                """List of test prefixes which should be ignored in branch db"""
                return super(self.__class__, self).ignore_prefix + [
                    "RELEASE_MACHINE_BUILD_UI",
                    "TESTING_RELEASE_RM_UI",
                ]

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            deploy_infos = [
                ri.NannyDeployInfo(ri.DeployService("release_machine_prod")),
                ri.NannyDeployInfo(ri.DeployService("release_machine_dev"), stage=rm_const.ReleaseStatus.testing),
            ]

            return [
                ri.ReleasableItem(
                    name="release_machine_backend",
                    data=ri.SandboxResourceData("RELEASE_MACHINE_BINARY"),
                    deploy_infos=deploy_infos,
                ),
                ri.ReleasableItem(
                    name="release_machine_sb_poller",
                    data=ri.SandboxResourceData("SB_POLLER_BINARY"),
                    deploy_infos=deploy_infos,
                ),
                ri.ReleasableItem(
                    name="release_machine_ci_poller",
                    data=ri.SandboxResourceData("CI_POLLER_BINARY"),
                    deploy_infos=deploy_infos,
                ),
                ri.ReleasableItem(
                    name="release_machine_solomon_agent",
                    data=ri.SandboxResourceData("RM_SOLOMON_AGENT_BINARY"),
                    deploy_infos=deploy_infos,
                ),
            ]

        allow_robots_to_release_stable = True
        block_on_test_results = [
            release_block.block_conf(name_filter="_TEST__*", accept_result_threshold=release_block.OK),
        ]
        approvements = release_approvements.ReleaseApprovementsSettings(
            description="Please approve release {tag_name} of {component_name}",
            stages=["ilyaturuntaev"],
        )

        infra_service_id = 2046
        infra_envs = {
            rm_const.ReleaseStatus.stable: 3189,
        }
        infra_event_duration_sec = 5 * 60  # 5 min

    class Notify(configs.ReferenceBranchedConfig.Notify):
        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["rm_maintainers"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "ilyaturuntaev"
            queue = "RMREL"
            dev_queue = "RMDEV"
            summary_template = u"Release Machine acceptance {}"
            workflow = {}
            followers = [
                "mvel",
            ]
            add_commiters_as_followers = False
            use_task_author_as_assignee = False
            deadline = 7
            commit_importance_threshold = 2
            notify_on_component_versions_change_to_feature_tickets = frozenset(["RMDEV"])

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "ReleaseMachine/releases/"
        ya_make_targets = [
            "arcadia/release_machine/release_machine",
        ]
        svn_paths_filter = configs.ChangelogPathsFilter(
            rm_const.PermissionType.CHANGED,
            [
                "arcadia/release_machine/release_machine/proto",
                "arcadia/release_machine/release_machine/src",
                "arcadia/release_machine/release_machine/ui",
            ],
            importance=3,
        )

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "release_machine/release_machine_ui"
        tag_name = "release_machine/release_machine_ui"
