# -*- coding: utf-8 -*-

import os
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.config_core.statistics_page as statistics_page
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow
from sandbox.projects.release_machine.components.config_core.jg.preset import basic_build_presets
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import release as release_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube import base as jg_cube
from sandbox.projects.release_machine.components.config_core import yappy as yappy_cfg
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import yappy as yappy_cubes
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.configs.report import ReportTestsBundle


HAMSTER_SERVICES = [
    "hamster_report_man_web_yp",
    "hamster_report_sas_web_yp",
    "hamster_report_vla_web_yp",
]

PRODUCTION_SERVICES = [
    "production_report_man_web_yp",
    "production_report_sas_web_yp",
    "production_report_vla_web_yp",
]

PRESTABLE_SERVICES = [
    "prestable_report_sas_web_yp",
]

PARENT_SERVICE = "hamster_report_vla_web_yp"


class ReportCICfg(configs.ReferenceCIConfig):
    name = "report_ci"
    display_name = "Report CI"
    responsible = configs.Responsible(
        login="elshiko",
        abc=configs.Abc(service_name="upper", schedule_slug="srcsetup_mapping_report_releases")
    )

    # metrics launch
    component_group = "reference_component_metricsed"

    class CI(configs.ReferenceCIConfig.CI):
        a_yaml_dir = "web/report"
        secret = "sec-01desry8fbgvnkbeybem81ferv"
        sb_owner_group = "REPORT_CORE"

        ya_make_abs_paths_glob = [
            os.path.join("web/report", "**"),
        ]

    class JG(basic_build_presets.SingleBuildGeneralJGCfg):
        build_task = "projects/upper_search/report/build_task"

        # configuring auto-deploy
        release_manually = False

        @jg_flow.release_flow(
            stages=basic_build_presets.JOINED_BUILD_RELEASE_FLOW_STAGES,
        )
        def release(self):
            graph = super(self.__class__, self).release(self)

            main_stage_entry_cube = graph.get("main_graph_entry")
            beta_cube = graph.get("generate_beta__report_ci__web")
            release_entry_cube = graph.get("release_stage_entry")

            # build task for instancectl configs
            build_configs_pva = jg_cube.Cube(
                name="build_configs_with_pva",
                task="projects/upper_search/report/build_configs_pva",
                input=jg_cube.CubeInput(
                    arcadia_url=rm_const.CIJMESPathCommon.ARCADIA_URL_SVN,
                    config_types="\n".join([
                        "REPORT_PRODUCTION_INSTANCECTL_CONF",
                        "REPORT_YAPPY_INSTANCECTL_CONF",
                        "REPORT_PUSHCLIENT_PRODUCTION_CONF",
                        "REPORT_PUSHCLIENT_YAPPY_CONF",
                    ]),
                ),
                needs=[
                    graph.get(dummy_cubes.RMMainGraphEntry.NAME),
                ],
            )

            graph.add(build_configs_pva)
            release_entry_cube.add_requirement(build_configs_pva)

            #  tests init
            tests = ReportTestsBundle.get_bundle(
                entry_cube=main_stage_entry_cube,
                beta_cube=beta_cube,
                soy_pool="report_web_priemka",
            )

            # add cubes to graph
            for test in tests:
                release_entry_cube.add_requirement(test)
                graph.add(test)

            return graph

        def _check_releasable_item_is_suitable(self, ri):
            return hasattr(ri.data, "resource_type") and ri.data.resource_type != "SCHEDULED_RM_RELEASE_DATA" and ri.name == "report"

        # activating build cube
        def _get_build_cube(self, graph):
            return jg_cube.Cube(
                name=self.build_cube_name,
                task=self.build_task,
                input=jg_cube.CubeInput(
                    report_arcadia_url=rm_const.CIJMESPathCommon.ARCADIA_URL_SVN,
                    project="WEB",
                ),
                needs=[
                    graph.get(dummy_cubes.RMMainGraphEntry.NAME),
                ],
            )

        # yappy beta with self resources from build
        def _get_generate_yappy_beta_cubes(self, graph, build):
            component_resources = {
                ri.name: build.output.resources[ri.data.resource_type].first().id
                for ri in self.suitable_releasable_items
            }

            # will be changed in https://st.yandex-team.ru/SEARCH-11943
            component_resources.update({
                "instancectl_conf_yappy": "${tasks.build_configs_with_pva.resources | [?type == 'REPORT_YAPPY_INSTANCECTL_CONF'] | [0].id}",
                "pushclient_conf_yappy": "${tasks.build_configs_with_pva.resources | [?type == 'REPORT_PUSHCLIENT_YAPPY_CONF'] | [0].id}",
            })

            return [
                yappy_cubes.GenerateYappyBeta(
                    component_name=self.component_name,
                    beta_conf_type="web",
                    input=jg_cube.CubeInput(
                        component_resources=component_resources,
                    ),
                    needs=[build],
                ),
            ]

        # activate Morty release
        def _get_release_cubes_for_deploy_system(self, deploy_system, where, releasable_items, graph, build):
            component_resources = {
                ri.name: build.output.resources[ri.data.resource_type].first().id
                for ri in releasable_items
            }

            # will be changed in https://st.yandex-team.ru/SEARCH-11943
            component_resources.update({
                "instancectl_conf_production": "${tasks.build_configs_with_pva.resources | [?type == 'REPORT_PRODUCTION_INSTANCECTL_CONF'] | [0].id}",
                "pushclient_conf_production": "${tasks.build_configs_with_pva.resources | [?type == 'REPORT_PUSHCLIENT_PRODUCTION_CONF'] | [0].id}",
            })

            return [
                release_cubes.ScheduleRelease(
                    name="release_{}_{}".format(where, deploy_system),
                    component_name=self.component_name,
                    where_to_release=where,
                    flows=["deploy"],
                    warden_component_name="report",
                    warden_parent_component_name="web",
                    deploy_system=deploy_system,
                    schedule_mode="create_and_schedule",
                    input=jg_cube.CubeInput(
                        component_resources=component_resources,
                    ),
                    manual=self.release_manually,
                ),
            ]

    class Releases(configs.ReferenceCIConfig.Releases):
        """Releases configuration"""
        allow_robots_to_release_stable = True
        allow_old_releases = True
        release_followers_permanent = [
            "elshiko",
        ]

        @property
        def releasable_items(self):
            release_deploy_infos = [
                ri.NannyDeployInfo(
                    [
                        ri.DeployService(service, tags=["hamster"]) for service in HAMSTER_SERVICES
                    ] + [
                        ri.DeployService(service, tags=["prestable"]) for service in PRESTABLE_SERVICES
                    ] + [
                        ri.DeployService(service, tags=["production"]) for service in PRODUCTION_SERVICES
                    ]
                ),
            ]

            return [
                ri.ReleasableItem(
                    name="report",
                    data=ri.SandboxResourceData("REPORT_CORE_PACKAGE", dst_path="report_core.tar"),
                    deploy_infos=release_deploy_infos,
                ),
                ri.ReleasableItem(
                    name="instancectl_conf_production",
                    data=ri.SandboxResourceData("REPORT_PRODUCTION_INSTANCECTL_CONF", dst_path="instancectl.conf"),
                    deploy_infos=release_deploy_infos,
                ),
                ri.ReleasableItem(
                    name="pushclient_conf_production",
                    data=ri.SandboxResourceData("REPORT_PUSHCLIENT_PRODUCTION_CONF", dst_path="push-client.conf.tmpl"),
                    deploy_infos=release_deploy_infos,
                ),
                ri.ReleasableItem(
                    name="instancectl_conf_yappy",
                    data=ri.SandboxResourceData("REPORT_YAPPY_INSTANCECTL_CONF", dst_path="instancectl.conf"),
                ),
                ri.ReleasableItem(
                    name="pushclient_conf_yappy",
                    data=ri.SandboxResourceData("REPORT_PUSHCLIENT_YAPPY_CONF", dst_path="push-client.conf.tmpl"),
                ),
                ri.DynamicReleasableItem(
                    name="scheduled_bundle",
                    data=ri.SandboxResourceData("SCHEDULED_RM_RELEASE_DATA"),
                    deploy_infos=[ri.NannyDeployInfo()],
                ),
            ]

    class Yappy(yappy_cfg.YappyBaseCfg):
        betas = {
            "beta": yappy_cfg.YappyBetaCfg(
                beta_name="report-web-1",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyBetaPatch(
                        patch_file="report-report-web-1",
                        resources=[
                            yappy_cfg.YappyParametrizedResource(
                                local_path="report_core.tar", param_name="report"
                            ),
                            yappy_cfg.YappyParametrizedResource(
                                local_path="instancectl.conf", param_name="instancectl_conf_yappy"
                            ),
                            yappy_cfg.YappyParametrizedResource(
                                local_path="push-client.conf.tmpl", param_name="pushclient_conf_yappy"
                            ),
                        ],
                        parent_service=PARENT_SERVICE,
                    ),
                ],
            ),
        }

    class ReleaseViewer(configs.ReferenceCIConfig.ReleaseViewer):
        """Release viewer configuration"""
        kpi_alert = 3
        kpi_alert_skip_weekends = True
        statistics_page_charts = statistics_page.DEFAULT + [statistics_page.PredefinedCharts.TIMESPECTRE_METRICS]

    class Notify(configs.ReferenceCIConfig.Notify):
        """Notifications configuration"""

        notifications = [
            rm_notifications.Notification(
                event_type="MergeCommit",
                chat_name="report_from_rm_const",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="report_from_rm_const",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewTag",
                chat_name="report_from_rm_const",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="KpiAlert",
                chat_name="report_from_rm_const",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="report_from_rm_const",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.STARTREK_TICKET_CREATED_CONDITIONS,
                    join_strategy="OR",
                ),
                message_template_file="notifications/ticket_created.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="report_from_rm_const",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="report_from_rm_const",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_deployed_oldschool.html",
            ),
        ]

        class Telegram(configs.ReferenceCIConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = ["report"]
            config = configs.RmTelegramNotifyConfig(chats=chats)
            invite_link = "https://t.me/joinchat/CaUODkYSN2MyE5jE9xgJzg"

        class Startrek(configs.ReferenceCIConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = configs.Responsible(login="elshiko", abc=configs.Abc(component_id=1609, role_id=None, schedule_slug="srcsetup_mapping_report_releases"))
            queue = "REPORTREL"
            dev_queue = "WEBREPORT"
            summary_template = u"Приемка web report {}"
            components = u"web_report"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "elshiko",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 4
            nanny_reports = False

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "serp/report/releases/web-report/"
        observed_paths = [
            "arcadia/web/report",
            "arcadia/search/garden/runtime_tests",
        ]

    class MetricsCfg(configs.ReferenceCIConfig.MetricsCfg):
        """Metrics configuration"""

    class SvnCfg(configs.ReferenceCIConfig.SvnCfg):
        tag_prefix = "r"
        branch_prefix = "r"
        branch_folder_template = "{branch_prefix}{branch_num}"
        tag_folder_template = "{tag_prefix}{branch_num}/1.{tag_num}"
        tag_folder_pattern = r"r([0-9]+)/1\.([0-9]+)"
        allow_autobranches = True
