
from sandbox.projects.release_machine.components import configs

from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.core import const as rm_const

from sandbox.projects.release_machine.components.config_core.jg import base as jg_base
from sandbox.projects.release_machine.components.config_core.jg import cube as jg_cube
from sandbox.projects.release_machine.components.config_core.jg.graph import base as graph_base
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import build as build_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import release as release_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import db as db_cubes
from sandbox.projects.release_machine.components.config_core.jg import flow


DEPLOY_INFO_TESTING = ri.NannyDeployInfo(ri.DeployService("release_machine_dev"), stage=rm_const.ReleaseStatus.testing)
DEPLOY_INFO_STABLE = ri.NannyDeployInfo(ri.DeployService("release_machine_prod"))


class RmCfg(configs.ReferenceCIConfig):

    name = "rm"
    display_name = "Release Machine"

    responsible = configs.Responsible(
        abc=configs.Abc(service_name="releasemachine", component_id=2676),
        login="ilyaturuntaev",
    )

    class CI(configs.ReferenceCIConfig.CI):
        a_yaml_dir = "release_machine/release_machine"
        secret = "sec-01desry8fbgvnkbeybem81ferv"
        sb_owner_group = "RELEASE_MACHINE"

        ya_make_abs_paths_glob = [
            "release_machine/release_machine/**",
            "release_machine/common_proto/**",
        ]

    class JG(jg_base.JGCfg):

        def _iter_build_cubes_with_releasable_items(self):

            for releasable_item in self.root_cfg.releases_cfg.releasable_items:

                yield releasable_item, build_cubes.KosherYaMake(
                    name="build__{}".format(releasable_item.name),
                    title="Build " + " ".join(releasable_item.name.split("_")).title(),
                    targets=releasable_item.build_data.target,
                    artifacts=releasable_item.build_data.artifact,
                    resource_types=releasable_item.data.resource_type,
                )

        def _get_release_cube_for_build(self, releasable_item, build_cube, deploy_info, manual=True, trunk=False):

            releasable_item_resource_id = build_cube.output.resources[releasable_item.data.resource_type].first().id

            if trunk:
                release_cube_cls = release_cubes.ReleaseRmComponent2Trunk
            else:
                release_cube_cls = release_cubes.ReleaseRmComponent2

            return release_cube_cls(
                name="release__{where}__{what}".format(
                    what=releasable_item.name,
                    where=deploy_info.stage,
                ),
                title="Release {what} to {where}".format(
                    what=releasable_item.name,
                    where=deploy_info.stage,
                ),
                component_name=self.component_name,
                where_to_release=deploy_info.stage,
                manual=manual,
                input=jg_cube.CubeInput(
                    component_resources={
                        releasable_item.name: releasable_item_resource_id,
                    },
                    deploy_system=deploy_info.deploy_system.name,
                    release_item_name="{}_trunk".format(releasable_item.name),
                ),
                needs=[build_cube],
            )

        @flow.main_release_flow()
        def release(self):
            return graph_base.Graph([dummy_cubes.Dummy("potato")])

        @flow.supplementary_release_flow(
            title="Testing Release RM Backend and Workers",
            auto=True,
            stages=[
                flow.ReleaseActionStageData(
                    "entry",
                    cube_names=["entry"],
                ),
                flow.ReleaseActionStageData(
                    "build_and_release",
                    cube_names=[
                        "build_entry",
                        "migrate_db_dev",
                        "migrate_db_test",
                    ],
                    cube_types=[
                        build_cubes.TYPE_BUILD,
                        release_cubes.TYPE_RELEASE,
                    ],
                    displace=flow.DISPLACE_ON_FAILURE_OR_WAITING,
                ),
            ],
        )
        def testing_release_rm_backend(self):

            all_cubes = []

            entry = dummy_cubes.Dummy(name="entry")

            build_entry = dummy_cubes.Dummy(name="build_entry", needs=[entry])

            migration_cube_dev = db_cubes.RunAlembicMigrations(
                name="migrate_db_dev",
                title="Migrate DB (dev)",
                component_name=self.component_name,
                db_url_yav_secret_uuid="sec-01ejz9hcr8tq4mmwga719ecgt0",
                db_url_yav_secret_name="db_url_dev",
                project_root="release_machine/release_machine",
                input=jg_cube.CubeInput(
                    checkout_arcadia_from_url=rm_const.CIJMESPathCommon.ARCADIA_URL_SVN,
                ),
                needs=[build_entry],
            )

            migration_cube_test = db_cubes.RunAlembicMigrations(
                name="migrate_db_test",
                title="Migrate DB (tests)",
                component_name=self.component_name,
                db_url_yav_secret_uuid="sec-01ejz9hcr8tq4mmwga719ecgt0",
                db_url_yav_secret_name="db_url_test",
                project_root="release_machine/release_machine",
                input=jg_cube.CubeInput(
                    checkout_arcadia_from_url=rm_const.CIJMESPathCommon.ARCADIA_URL_SVN,
                ),
                needs=[build_entry],
            )

            all_cubes.extend([entry, build_entry, migration_cube_dev, migration_cube_test])

            for releasable_item, build_cube in self._iter_build_cubes_with_releasable_items():

                build_cube.add_requirement(build_entry)

                release_cube = self._get_release_cube_for_build(
                    releasable_item,
                    build_cube,
                    DEPLOY_INFO_TESTING,
                    manual=False,
                    trunk=True,
                )

                release_cube.add_requirement(migration_cube_dev)

                all_cubes.extend([build_cube, release_cube])

            return graph_base.Graph(all_cubes)

        @flow.supplementary_release_flow(
            title="Testing Release RM UI",
            auto=True,
            filters=[
                {
                    "discovery": "dir",
                    "abs-paths": [
                        "release_machine/release_machine/ui/**",
                        "release_machine/release_machine/frontend/**",
                    ],
                },
            ],
        )
        def testing_release_rm_frontend(self):

            entry = dummy_cubes.Dummy(name="entry")

            build_entry = dummy_cubes.Dummy(name="build_entry", needs=[entry])

            build_ui = build_cubes.BuildReleaseEngineUi(
                input=jg_cube.CubeInput(
                    checkout_arcadia_from_url=(
                        "arcadia:"
                        "/arc/trunk/arcadia/release_machine/release_machine/ui@${context.target_revision.number}"
                    ),
                    nodejs_archive="678521955",
                    npm_build_common="build_production",
                ),
                needs=[build_entry],
            )

            release_to_testing = release_cubes.ReleaseZephyrServiceUi(
                component_name=self.component_name,
                where_to_release="testing",
                resource=build_ui.output.resources["RELEASE_ENGINE_UI"].first().id,
                access_secret_key="rm_testing_ui_access_key",
                needs=[build_ui],
                manual=False,
            )

            return graph_base.Graph([entry, build_entry, build_ui, release_to_testing])

    class Releases(configs.ReferenceCIConfig.Releases):

        @property
        def releasable_items(self):

            deploy_infos = [
                DEPLOY_INFO_STABLE,
                DEPLOY_INFO_TESTING,
            ]

            return [
                ri.ReleasableItem(
                    name="backend",
                    data=ri.SandboxResourceData("RELEASE_MACHINE_BINARY"),
                    deploy_infos=deploy_infos,
                    build_data=ri.BuildData(
                        target="release_machine/release_machine/bin/release_machine",
                        artifact="release_machine/release_machine/bin/release_machine",
                    ),
                ),
                ri.ReleasableItem(
                    name="sb_poller",
                    data=ri.SandboxResourceData("SB_POLLER_BINARY"),
                    deploy_infos=deploy_infos,
                    build_data=ri.BuildData(
                        target="release_machine/release_machine/src/sb_poller/bin/sb_poller",
                        artifact="release_machine/release_machine/src/sb_poller/bin/sb_poller",
                    ),
                ),
                ri.ReleasableItem(
                    name="ci_poller",
                    data=ri.SandboxResourceData("CI_POLLER_BINARY"),
                    deploy_infos=deploy_infos,
                    build_data=ri.BuildData(
                        target="release_machine/release_machine/src/ci_poller/bin/ci_poller",
                        artifact="release_machine/release_machine/src/ci_poller/bin/ci_poller",
                    ),
                ),
                ri.ReleasableItem(
                    name="solomon_agent",
                    data=ri.SandboxResourceData("RM_SOLOMON_AGENT_BINARY"),
                    deploy_infos=deploy_infos,
                    build_data=ri.BuildData(
                        target="solomon/agent/bin/solomon-agent",
                        artifact="solomon/agent/bin/solomon-agent",
                    ),
                ),
            ]

        allow_robots_to_release_stable = True

        infra_service_id = 2046
        infra_envs = {
            rm_const.ReleaseStatus.stable: 3189,
        }
        infra_event_duration_sec = 5 * 60  # 5 min

    class Notify(configs.ReferenceCIConfig.Notify):

        class Telegram(configs.ReferenceCIConfig.Notify.Telegram):
            chats = ["rm_maintainers"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceCIConfig.Notify.Startrek):
            assignee = "ilyaturuntaev"
            queue = "RMREL"
            dev_queue = "RMDEV"
            summary_template = u"Release Machine acceptance {}"
            workflow = {}
            followers = [
                "mvel",
            ]
            add_commiters_as_followers = False
            use_task_author_as_assignee = False
            deadline = 7
            commit_importance_threshold = 2
            notify_on_component_versions_change_to_feature_tickets = frozenset(["RMDEV"])

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = "ReleaseMachine/releases/"
        ya_make_targets = [
            "arcadia/release_machine/release_machine",
        ]
        svn_paths_filter = configs.ChangelogPathsFilter(
            rm_const.PermissionType.CHANGED,
            [
                "arcadia/release_machine/release_machine/proto",
                "arcadia/release_machine/release_machine/src",
                "arcadia/release_machine/release_machine/ui",
            ],
            importance=3,
        )
