# -*- coding: utf-8 -*-
from enum import Enum
from functools import partial
from itertools import chain

from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components.config_core.jg import base as jg_base
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow
from sandbox.projects.release_machine.components.config_core.jg import cube as jg_cube
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import deploy as deploy_cubes

from ._resource import (
    BuildableResource,
)


ROBOT_SECRET = 'sec-01g27qf5atjj7ya5251823z53m'

PROJECT_ROOT = 'quality/user_sessions/money_rt'
PROCESSORS_ROOT = '{}/processors'.format(PROJECT_ROOT)


class ReleaseStatus(rm_const.ReleaseStatus):
    hotfix = 'hotfix'


class Resources(Enum):
    resharder = BuildableResource(
        '{}/resharder/money-rt-resharder'.format(PROCESSORS_ROOT),
        'RESHARDER_RT_MONEY_BINARY',
    )
    hit_master = BuildableResource(
        '{}/hit_master/money-rt-hit-master'.format(PROCESSORS_ROOT),
        'HIT_MASTER_RT_MONEY_BINARY',
    )
    hit_profile_state_observers = BuildableResource(
        '{}/hit_profile_state_observers/hit-profile-state-observers'.format(
            PROCESSORS_ROOT,
        ),
        'HIT_PROFILE_STATE_OBSERVERS_RT_MONEY_BINARY',
    )
    event_streamer = BuildableResource(
        '{}/event_streamer/money-rt-event-streamer'.format(PROCESSORS_ROOT),
        'EVENT_STREAMER_RT_MONEY_BINARY',
    )
    sync_profiles = BuildableResource(
        '{}/sync_profiles/money-rt-sync-profiles'.format(PROCESSORS_ROOT),
        'SYNC_PROFILES_RT_MONEY_BINARY',
    )
    delay_for_sync_profiles = BuildableResource(
        '{}/delay_for_sync_profiles/money-rt-delay-for-sync-profiles'.format(
            PROCESSORS_ROOT,
        ),
        'DELAY_FOR_SYNC_PROFILES_RT_MONEY_BINARY',
    )
    queue_daemon = BuildableResource(
        'ads/bsyeti/big_rt/queue_daemon/queue_daemon',
        'QUEUE_DAEMON_RT_MONEY_BINARY',
    )
    table_balancer = BuildableResource(
        'quality/user_sessions/rt/tools/table_balancer/us-rt-table-balancer',
        # TODO(statizarm): uncomment after resource type declaration
        # 'TABLE_BALANCER_RT_MONEY_BINARY',
    )
    config_generator = BuildableResource(
        '{}/packages/rt_money_config_generators/rt_money_config_generators'.format(
            PROJECT_ROOT,
        ),
        'CONFIG_GENERATOR_RT_MONEY_BINARY',
    )
    solomon_agent = BuildableResource(
        'solomon/agent/bin/solomon-agent',
        'SOLOMON_RT_MONEY_BINARY',
    )


DEPLOY_UNIT_RESOURCES = {
    'delay-for-sync-profiles': {
        'binaries': Resources.delay_for_sync_profiles.value,
    },
    'event-streamer': {
        'binaries': Resources.event_streamer.value,
    },
    'hit-master': {
        'binaries': Resources.hit_master.value,
    },
    'hit-profile-state-observers': {
        'binaries': Resources.hit_profile_state_observers.value,
    },
    'queue-daemon': {
        'binaries': Resources.queue_daemon.value,
    },
    'resharder-bs-event-lite-log': {
        'binaries': Resources.resharder.value,
        'solomon_binary': Resources.solomon_agent.value,
    },
    'resharder-bs-hit-rt-log': {
        'binaries': Resources.resharder.value,
        'solomon_binary': Resources.solomon_agent.value,
    },
    'resharder-profile-info': {
        'binaries': Resources.resharder.value,
    },
    'sync-profiles': {
        'binaries': Resources.sync_profiles.value,
    },
    'sync-table-balancer': {
        'binaries': Resources.table_balancer.value,
    },
    'table-balancer': {
        'binaries': Resources.table_balancer.value,
    }
}

COMMON_RESOURCES = {
    'configs': Resources.config_generator.value,
}

RELEASE_STATUS_TO_STAGES = {
    ReleaseStatus.stable: ('money-rt-production',),
    ReleaseStatus.prestable: ('money-rt-prestable',),
    ReleaseStatus.hotfix: ('money-rt-production', 'money-rt-prestable'),
}


class RtMoneyCfg(configs.ReferenceCIConfig):
    name = "rt_money"
    responsible = configs.Responsible(
        abc=configs.Abc(service_name="user-sessions"),
        login="statizarm",
    )

    class JG(jg_base.BaseReleaseMachineJobGraph):

        @classmethod
        def _static_resource_patch(
            cls, resource_type, resource_id, deploy_unit, resource_ref,
        ):
            return dict(
                sandbox=dict(
                    sandbox_resource_type=resource_type,
                    attributes=dict(
                        id=resource_id,
                    ),
                    static=dict(
                        deploy_unit_id=deploy_unit,
                        static_resource_ref=resource_ref,
                    ),
                ),
            )

        @classmethod
        def _patches(cls, deploy_units):
            res = []
            for deploy_unit, resources in deploy_units.items():
                create_patch = partial(
                    cls._static_resource_patch,
                    deploy_unit=deploy_unit,
                )
                for resource_ref, resource in chain(
                    COMMON_RESOURCES.items(), resources.items()
                ):
                    res.append(
                        create_patch(
                            resource_type=resource.type,
                            resource_id=resource.id,
                            resource_ref=resource_ref,
                        )
                    )

            return res

        @classmethod
        def _release_cube(
            cls, stage, release_status, deploy_units, needs=None,
        ):
            needs = [] if needs is None else needs
            where = release_status
            return deploy_cubes.CreateDeployRelease(
                name='release_{}'.format(stage.replace('-', '_')),
                component_name='rt_money',
                where_to_release=where,
                manual=release_status == ReleaseStatus.stable,
                input=jg_cube.CubeInput(
                    config=dict(
                        stage_id=stage,
                        patches=cls._patches(deploy_units),
                    ),
                ),
                needs=needs,
            )

        @classmethod
        def _add_release_cubes(
            cls, graph, release_status, exclude_deploy_units=None, needs=None
        ):
            needs = [] if needs is None else needs
            exclude_deploy_units = set() if exclude_deploy_units is None else exclude_deploy_units
            deploy_units = {
                key: value
                for key, value in DEPLOY_UNIT_RESOURCES.items()
                if key not in exclude_deploy_units
            }

            res = []
            for stage in RELEASE_STATUS_TO_STAGES[release_status]:
                release_cube = cls._release_cube(
                    stage=stage,
                    release_status=release_status,
                    deploy_units=deploy_units,
                    needs=needs,
                )
                graph.add(release_cube)
                res.append(release_cube)

            return res

        def _add_wait_cube(cls, graph, name, wait_for):
            wait_cube = dummy_cubes.Dummy(
                name='wait_{}'.format(name),
                needs=wait_for,
            )
            graph.add(wait_cube)
            return wait_cube

        @jg_flow.release_flow(
            auto=dict(
                conditions=[
                    {
                        'min-commits': 1,
                        'since-last-release': '4h',
                        'schedule': {
                            'time': '9:00 - 21:00 MSK',
                            'days': 'MON, WED',
                        }
                    }
                ]
            ),
        )
        def release(self):
            graph = super(self.__class__, self).release(self)
            main_entry = graph.get(dummy_cubes.RMMainGraphEntry.NAME)

            build_cubes = [
                resource.value.cube(name=resource.name, needs=[main_entry])
                for resource in Resources
            ]

            for cube in build_cubes:
                graph.add(cube)

            wait_build_all_cube = self._add_wait_cube(
                graph, 'build_all', build_cubes,
            )

            release_prestable_cubes = self._add_release_cubes(
                graph=graph,
                release_status=rm_const.ReleaseStatus.prestable,
                needs=[wait_build_all_cube],
            )

            self._add_wait_cube(
                graph, 'release_prestable', release_prestable_cubes,
            )

            release_production_cubes = self._add_release_cubes(
                graph=graph,
                release_status=rm_const.ReleaseStatus.stable,
                needs=[wait_build_all_cube],
            )

            self._add_wait_cube(
                graph, 'release_stable', release_production_cubes,
            )

            return graph

    class CI(configs.ReferenceCIConfig.CI):
        a_yaml_dir = "quality/user_sessions/money_rt"
        secret = ROBOT_SECRET
        sb_owner_group = "USERSESSIONSTOOLS"

        # влюкчаем graph- и dir-discovery по этим путям
        ya_make_abs_paths_glob = [
            "quality/user_sessions/money_rt/**",
        ]

    class Releases(configs.ReferenceCIConfig.Releases):
        @property
        def releasable_items(self):
            infos = [
                ri.YaDeployInfo(
                    ri.DeployService(stage), status,
                )
                for status, stages in RELEASE_STATUS_TO_STAGES.items()
                for stage in stages
            ]

            resource_types = set(
                resource.value.type for resource in Resources
            )

            return [
                ri.ReleasableItem(
                    name="rt-money-{}".format(
                        resource_type.lower().replace('_', '-'),
                    ),
                    data=ri.SandboxResourceData(resource_type),
                    deploy_infos=infos,
                ) for resource_type in resource_types
            ]

        deploy_system = rm_const.DeploySystem.ya_deploy
        allow_robots_to_release_stable = True
        allow_old_releases = True

    class Notify(configs.ReferenceCIConfig.Notify):
        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""

            assignee = configs.Responsible(
                abc=configs.Abc(
                    component_id=33128, role_id=8, schedule_slug='rtyt',
                ),
                login="statizarm",
            )
            queue = "USRTREL"
            dev_queue = "ZBL"
            summary_template = u"[r{}] RTMoney release"
            workflow = {}
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7

            @property
            def banned_people(self):
                return super(self.__class__, self).banned_people | {"pechatnov"}

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = "logs/usersessions/rtmoney/releasehistory/"
        dirs = [
            "quality/user_sessions/money_rt/",
            "ads/bsyeti/big_rt",
        ]
