# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
from sandbox.projects.release_machine.core import releasable_items as ri
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers


RTHUB_TESTS_TYPES = ["HOSTS", "GEO_SEARCH", "IMAGES_FRESH", "IMAGES", "PAGES_FRESH", "PAGES", "SITEMAPS", "APP_DOCS"]
RTHUB_PACKAGES_TYPES = ["HOSTS", "GEO_SEARCH", "IMAGES", "PAGES", "PAGES_FRESH", "SITEMAPS", "APP_DOCS", "LB_BALANCER", "KWYT_VIEWER"]
NANNY_SERVICES = {
    "HOSTS": "rthub-hosts",
    "GEO_SEARCH": None,
    "IMAGES": "rthub-images",
    "PAGES": "rthub-pages",
    "PAGES_FRESH": "rthub-pages-fresh",
    "SITEMAPS": "rthub-sitemaps",
    "APP_DOCS": "rthub-appdocs",
    "LB_BALANCER": "rthub-robot-lb-balancer",
    "KWYT_VIEWER": None,
}


class RTHubCfg(configs.ReferenceBranchedConfig):
    name = "rthub"
    responsible = "zagevalo"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "RTHUB"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                for pkg_type in RTHUB_PACKAGES_TYPES:
                    branch_part.append(
                        jg_build.JobGraphElementYaMakeBuildBranched(
                            task_name="BUILD_RTHUB_PACKAGE",
                            job_params={
                                "job_name_parameter": pkg_type,
                            },
                            ctx={
                                "packages": "robot/rthub/packages/rthub_{}_full_package.json".format(pkg_type.lower()),
                                "resource_type": "RTHUB_{}_FULL_PACKAGE".format(pkg_type),
                                "ya_yt_dir": "//home/kwyt-test/yamake_cache",
                                "ya_yt_proxy": "arnold.yt.yandex.net",
                                "ya_yt_put": True,
                                "ya_yt_store": True,
                                "ya_yt_token_vault_name": "kwyt_test_yt_token",
                                "ya_yt_token_vault_owner": "RTHUB"
                            },
                            out={
                                "RTHUB_{}_FULL_PACKAGE".format(pkg_type): 90,
                            },
                        )
                    )
                for pkg_type in RTHUB_TESTS_TYPES:
                    branch_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="TEST_RTHUB_{}_PACKAGE".format(pkg_type),
                            job_params={
                                "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                                "job_name_parameter": "{}_PACKAGE".format(pkg_type),
                                "diff_resource_type": {"RTHUB_TEST_DIFF": 30},
                                "test_type": jg_utils.TestType.DIFF_TEST,
                                "compare_task_name": "COMPARE_RTHUB_TEST_OUTPUTS",
                                "has_diff": lambda ctx: ctx["has_diff"],
                                "compare_job_triggers": (
                                    jg_arrows.CompareTaskDataResource(
                                        input_key="prev_test_out",
                                        resource_name="RTHUB_TEST_OUTPUT",
                                        task_number=1,
                                    ),
                                    jg_arrows.CompareTaskDataResource(
                                        input_key="new_test_out",
                                        resource_name="RTHUB_TEST_OUTPUT",
                                        task_number=2,
                                    ),
                                    jg_arrows.CompareTaskDataResource(
                                        input_key="diff_tool",
                                        resource_name="RTHUB_DIFF_TOOL",
                                        task_number=2,
                                    ),
                                ),
                                "apiargs": {
                                    "requirements": {
                                        "cpu_model": "e5-2650",
                                    },
                                },
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter=pkg_type.split("_FRESH")[0],
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="rthub_pkg_build",
                                            resource_name="RTHUB_{}_FULL_PACKAGE".format(pkg_type.split("_FRESH")[0]),
                                        )
                                    )
                                ) if pkg_type != "PAGES_FRESH" else None,
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter=pkg_type,
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="rthub_pkg_build",
                                            resource_name="RTHUB_{}_FULL_PACKAGE".format(pkg_type),
                                        )
                                    )
                                ) if pkg_type == "PAGES_FRESH" else None,
                                jg_arrows.GlobalResourceData(
                                    input_key="test_data",
                                    resource_name="RTHUB_{}_TEST_DATA".format(pkg_type),
                                )
                            ),
                            out={
                                "RTHUB_TEST_OUTPUT": 30,
                                "RTHUB_DIFF_TOOL": 30,
                            },
                        )
                    )
                return branch_part

            @property
            def _release(self):
                default_release_part = super(self.__class__, self)._release
                release_part = []
                for release_item in [
                    "PAGES",
                    "HOSTS",
                    "GEO_SEARCH",
                    "SITEMAPS",
                    "IMAGES",
                    "APP_DOCS",
                    "LB_BALANCER",
                    "KWYT_VIEWER",
                ]:
                    for release_stage in [rm_const.ReleaseStatus.prestable, rm_const.ReleaseStatus.stable]:
                        release_part.append(
                            jg_release.JobGraphElementReleaseBranched(
                                release_to=release_stage,
                                release_item=release_item,
                                job_arrows=(
                                    jg_job_triggers.JobTriggerBuild(
                                        job_name_parameter=release_item,
                                        parent_job_data=(
                                            jg_job_data.ParentDataDict(
                                                "component_resources",
                                                "rthub_{}_package".format(release_item.lower()),
                                                "RTHUB_{}_FULL_PACKAGE".format(release_item),
                                            ),
                                        ),
                                    ),
                                    jg_job_triggers.JobTriggerNewTag(
                                        parent_job_data=[
                                            jg_job_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                                            jg_job_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
                                        ],
                                    ),
                                )
                            )
                        )
                        release_part.append(
                            jg_release.JobGraphElementActionReleaseBranched(
                                release_to=release_stage,
                                release_item=release_item,
                            )
                        )
                for release_stage in [
                    rm_const.ReleaseStatus.prestable,
                    rm_const.ReleaseStatus.stable,
                    rm_const.ReleaseStatus.testing,
                ]:
                    release_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=release_stage,
                            release_item="PAGES_FRESH",
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="PAGES_FRESH",
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "rthub_pages_fresh_package",
                                            "RTHUB_PAGES_FRESH_FULL_PACKAGE",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerNewTag(
                                    parent_job_data=[
                                        jg_job_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                                        jg_job_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
                                    ],
                                ),
                            )
                        )
                    )
                    release_part.append(
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                            release_item="PAGES_FRESH",
                        )
                    )
                return default_release_part + release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                freq_patch = {}
                for t in RTHUB_TESTS_TYPES:
                    freq_patch["_TEST_RTHUB_{}".format(t)] = rm_const.TestFrequencies.EACH_REV_TEST
                for b in RTHUB_PACKAGES_TYPES:
                    freq_patch["_BUILD_RTHUB_{}".format(b)] = rm_const.TestFrequencies.LAZY_TEST
                freq_patch["_BUILD_RTHUB_LB_BALANCER"] = rm_const.TestFrequencies.EACH_REV_TEST
                freq_patch["_BUILD_RTHUB_KWYT_VIEWER"] = rm_const.TestFrequencies.EACH_REV_TEST
                tests.update(freq_patch)
                return tests

            @property
            def ignore_match(self):
                common_ignored = super(self.__class__, self).ignore_match
                rthub_ignored = []
                for b in RTHUB_PACKAGES_TYPES:
                    rthub_ignored.append("BUILD_RTHUB_{}_PACKAGE".format(b))
                for t in RTHUB_TESTS_TYPES:
                    rthub_ignored.append("TEST_RTHUB_{}_PACKAGE".format(t))
                rthub_ignored.append("UPDATE_RTHUB_RESOURCES")
                rthub_ignored.append("TEST_RTHUB_DEBUG_KEY_INV_DIFF")
                return common_ignored + rthub_ignored

    class ReleaseViewer(configs.ReferenceConfig.ReleaseViewer):
        kpi_alert = 7

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="rthub_{}_package".format(i.lower()),
                    data=ri.SandboxResourceData("RTHUB_{}_FULL_PACKAGE".format(i)),
                    deploy_infos=[ri.single_nanny_service(NANNY_SERVICES[i])] if NANNY_SERVICES[i] else None
                ) for i in RTHUB_PACKAGES_TYPES
            ]

        release_followers_permanent = [
            "izetag",
            "ulgen4a",
            "zagevalo",
            "itelichko",
            "akhovrychev",
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        notifications = [
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="rthub_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="KpiAlert",
                chat_name="rthub_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="AcceptanceTest",
                chat_name="rthub_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="BuildTest",
                chat_name="rthub_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewBetaGeneration",
                chat_name="rthub_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="ReleaseFailed",
                chat_name="rthub_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="MergeCommit",
                chat_name="rthub_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="rthub_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_deployed_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="rthub_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="rthub_comp",
                message_template_file="notifications/ticket_history.html",
                pin=True,
                conditions=rm_notifications.NotificationCondition(
                    conditions=[
                        rm_notifications.NotificationConditionItem(
                            field="ticket_history_data.ticket_history_latest_status",
                            operator="TEXT_EXACTLY_IS",
                            value="open",
                        ),
                    ],
                    join_strategy="OR",
                ),
            ),

        ]

        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            mailing_list = ["rthub"]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["rthub_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "zagevalo"
            queue = "RTHUBRELEASE"
            dev_queue = "RTHUB"
            summary_template = u"RTHub {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "zagevalo",
                "ulgen4a",
                "izetag",
                "itelichko",
            ]
            add_commiters_as_followers = False
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "robot/rthub/changelogs/"
        dirs = [
            "arcadia/robot/rthub",
            "arcadia/yql",
        ]
        review_groups = [
            "kwyt",
        ]

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_prefix = "pre-stable"
