# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.common import constants as sandbox_constants
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers


RTMR_SAN_TYPES = ["GCC", "ASAN", "MSAN", "TSAN", "UBSAN", "DEBUG", "VALGRIND"]
RTMR_PACKAGES = ["HOST", "SERVER", "PUSHER", "STORAGE", "PROXY", "APPHOST"]


class RtmrCfg(configs.ReferenceBranchedConfig):
    name = "rtmr"
    responsible = "uzhas"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_db = "rtmr_full_trunk"
        trunk_task_owner = "RTMR-DEV"
        branch_db_template = "rtmr-stable-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            DAEMONS = [
                "host",
                "proxy",
                "pusher",
                "server",
                "storage",
                "apphost",
            ]

            SANITIZERS = [
                ("ASAN", "address", "DEFAULT-LINUX-X86_64", "debug"),
                ("DEBUG", "", "DEFAULT-LINUX-X86_64", "debug"),
                ("GCC", "", "GCC8-LINUX-X86_64", "debug"),
                ("MSAN", "memory", "DEFAULT-LINUX-X86_64", "debug"),
                ("TSAN", "thread", "DEFAULT-LINUX-X86_64", "debug"),
                ("UBSAN", "undefined", "DEFAULT-LINUX-X86_64", "debug"),
                ("VALGRIND", "", "DEFAULT-LINUX-X86_64", "valgrind-release"),
            ]

            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(
                        branching_mode=rm_const.BranchingMode.BRANCH_TO_BRANCH,
                    ),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(),
                ]

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                for daemon in self.DAEMONS:
                    branch_part.append(
                        jg_build.JobGraphElementBuildPackageBranched(
                            package_names="rtmapreduce/packages/daemons/yandex-search-rtmr-{}/pkg.json".format(daemon),
                            resource_names="RTMR_{}_DEB".format(daemon.upper()),
                            job_params={
                                "apiargs": {
                                    "requirements": {
                                        "disk_space": 30 << 30,  # 30 Gb
                                    },
                                    "kill_timeout": 28800,
                                },
                                "job_name_parameter": "DAEMONS_{}".format(daemon),
                            },
                            ctx={
                                "ya_timeout": 28000,
                                "package_type": "debian",
                                "publish_to": "search",
                                "publish_package": True,
                                "ya_yt_dir": "//home/rtmr-dev/build-cache",
                                "ya_yt_max_cache_size": 161061273600,
                                "ya_yt_proxy": "freud.yt.yandex.net",
                                "ya_yt_put": True,
                                "ya_yt_replace_result": False,
                                "ya_yt_store": True,
                                "ya_yt_store_codec": "zstd08_1",
                                "ya_yt_token_vault_name": "robot-rtmr-build-yt",
                                "ya_yt_token_vault_owner": "RTMR-DEV",
                            }
                        )
                    )
                for name_param, sanitize, target_platform, build_type in self.SANITIZERS:
                    branch_part.append(
                        jg_test.JobGraphElementMetaTestBranched(
                            task_name="YA_MAKE",
                            job_params={
                                "job_name_parameter": "LIBS_{}_TESTS".format(name_param),
                                "apiargs": {
                                    "kill_timeout": 7200,
                                    "requirements": {
                                        "disk_space": 75 * (1024 ** 3),  # 75 Gb
                                    },
                                },
                                "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            },
                            job_arrows=(
                                jg_arrows.ParamsData(
                                    input_key=sandbox_constants.ARCADIA_URL_KEY,
                                    transform=lambda x, rm_config: "{}/arcadia@{}".format(x.svn_ssh_url, x.revision),
                                    override=True,
                                ),
                            ),
                            ctx={
                                "targets": "rtmapreduce/libs",
                                "test": True,
                                "disable_test_timeout": True,
                                "sanitize": sanitize,
                                "build_type": build_type,
                                "target_platform": target_platform,
                                "kill_timeout": 7200,
                                "checkout_mode": "auto",
                                "check_return_code": False,
                                "test": True,
                                "junit_report": True,
                                "use_aapi_fuse": True,
                                "definition_flags": "-DUSE_EAT_MY_DATA -DDEBUGINFO_LINES_ONLY -DAUTOCHECK",
                                "ya_timeout": 7100,
                            }
                        )
                    )
                umbrella_arrows = [
                    jg_job_triggers.JobTriggerMetaTest(
                        job_name_parameter="LIBS_DEBUG_TESTS",
                    )
                ]
                for daemon in self.DAEMONS:
                    umbrella_arrows.append(
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="DAEMONS_{}".format(daemon),
                        ),
                    )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedScheduled(
                        job_arrows=umbrella_arrows,
                    )
                )
                return branch_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def change_frequency(self):
                return {}

            @property
            def ignore_match(self):
                ignored = super(self.__class__, self).ignore_match
                ignored += ["RTMR_BUILD_DAEMON_PACKAGE_{}".format(s) for s in RTMR_PACKAGES]
                ignored += ["RTMR_BUILD_STATIC_CONFIG"]
                ignored += ["RTMR_BUILD_USERTASKS"]
                ignored += ["RTMR_CONFIG_TESTS"]
                ignored += ["RTMR_DAEMONS_CONFIG"]
                ignored += ["RTMR_DEPLOY_SLICE_9"]
                ignored += ["RTMR_LIBS_TESTS_{}".format(s) for s in RTMR_SAN_TYPES]
                ignored += ["RTMR_PREPARE_USERTASKS_RELEASE_META"]
                ignored += ["RTMR_USERTAKS_TESTS_WITH_STABLE_RTMR"]
                ignored += ["RTMR_USERTASKS_TESTS_{}".format(s) for s in RTMR_SAN_TYPES]
                return ignored

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="rtmr_{}".format(pkg.lower()),
                    data=ri.SandboxResourceData("RTMR_{}_DEB".format(pkg)),
                ) for pkg in RTMR_PACKAGES
            ]

        release_followers_permanent = [
            "uzhas",
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["rtmr_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "uzhas"
            queue = "RTMRREL"
            dev_queue = "RTMR"
            summary_template = u"RTMR release {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "uzhas",
            ]
            add_commiters_as_followers = True
            deadline = 7
            nanny_reports = False
            merged_revisions_order = rm_const.SortOrder.DESCENDING

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "rtmapreduce/rm/changelogs/"
        dirs = [
            "arcadia/kikimr/core/blobstorage",
            "arcadia/kikimr/core/protos",
            "arcadia/library/actors",
            "arcadia/rtmapreduce/config/presets",
            "arcadia/rtmapreduce/config/src",
            "arcadia/rtmapreduce/libs",
            "arcadia/rtmapreduce/packages/configs",
            "arcadia/rtmapreduce/packages/daemons",
            "arcadia/rtmapreduce/protos",
        ]
        review_groups = [
            "rtmr",
        ]
