# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import re
import os

from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core.jg import base as jg_base
from sandbox.projects.release_machine.components.config_core.jg.cube import base as jg_cube
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import build as build_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import release as release_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import internal as internal_cubes
from sandbox.projects.release_machine.components.config_core.jg.graph import base as graph_base
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.core import const as rm_const


class ReleaseSbTaskCubeBase(release_cubes.ReleaseRmComponent2):
    TYPE = "release"

    def _construct_name(self):
        return "{cube_type}__{component_name}".format(
            cube_type=self.TYPE,
            component_name=self._component_name,
        )


class ReleaseSbTaskCubeTesting(ReleaseSbTaskCubeBase):
    TYPE = "release_sb_testing"


class ReleaseSbTaskCubeStable(ReleaseSbTaskCubeBase):
    TYPE = "release_sb_stable"


class TestRunCube(jg_cube.Cube):
    TYPE = "testing"


DISPLACE = {
    "on-status": [
        "FAILURE",
        "RUNNING_WITH_ERRORS",
        "WAITING_FOR_STAGE",
    ],
}

STAGE_NEW_TAG = jg_flow.ReleaseActionStageData(
    "new_tag",
    cube_names=[
        internal_cubes.NewTagCube.NAME,
    ],
    displace=DISPLACE,
)
STAGE_BUILD = jg_flow.ReleaseActionStageData(
    "build",
    cube_names=[
        dummy_cubes.RMMainGraphEntry.NAME,
    ],
    cube_types=[
        build_cubes.TYPE_BUILD,
        internal_cubes.TYPE_CHANGELOG,
        internal_cubes.TYPE_STARTREK,
    ],
    displace=DISPLACE,
)
STAGE_TEST = jg_flow.ReleaseActionStageData(
    "test",
    cube_types=[
        ReleaseSbTaskCubeTesting.TYPE,
        TestRunCube.TYPE,
    ],
    displace=DISPLACE,
)

STAGE_RELEASE = jg_flow.ReleaseActionStageData(
    "release",
    cube_names=[
        dummy_cubes.ReleaseStageEntry.NAME,
    ],
    cube_types=[
        ReleaseSbTaskCubeStable.TYPE,
    ],
    rollback=True,
    displace=DISPLACE,
)


RELEASE_FLOW_STAGES = [
    STAGE_NEW_TAG,
    STAGE_BUILD,
    STAGE_TEST,
    STAGE_RELEASE,
]


RELEASE_FLOW_STAGES_WITH_ROLLBACKABLE_TESTING = [
    STAGE_NEW_TAG,
    STAGE_BUILD,
    STAGE_TEST.set_rollback(True),
    STAGE_RELEASE,
]


class SandboxSingleTaskReleaseCfg(configs.ReferenceCIConfig):
    """
    **Basic Usage**

    ```python
    from sandbox.projects.release_machine.components import configs
    from sandbox.projects.release_machine.components.configs.sb_tasks import _base as sb_cfg_base

    class MyShinyTaskCfg(sb_cfg_base.SandboxSingleTaskReleaseCfg):

        name = "<my_shiny_task_component_name>"
        responsible = configs.Responsible(
            abc=configs.Abc(service_name="<my_shiny_service>"),
            login="<my_shiny_someone>",
        )

        target_task_path = "sandbox/projects/my/shiny/task/bin"

        class CI(sb_cfg_base.SandboxSingleTaskReleaseCfg.CI):
            sb_owner_group = "<MY_SANDBOX_GROUP>"

    ```

    **Settings**

    - `name` — имя компоненты
    - `responsible` — ответственный (ABC-группа обязательна)

    - `target_task_path` — путь до папки с бинарем задачи
    - `target_task_type` — (optional) тип задачи (SCREAMING_SNAKE_CASE, дефолт вычисляется по пути `target_task_path`)
    - `target_bin_name` — (optional) имя бинаря задачи (default: `target_task_type.lower()`)
    - `task_binary_resource` — (optional) тип ресурса с бинарем задачи (default: "SANDBOX_TASKS_BINARY")
    - `test_run_input` — (optional) — инпут для тестового запуска задачи после релиза в тестинг (default: None - не запускать)

    - `add_precommit_check` — добавить прекоммитную проверку
    - `add_prestable_auto_release` — добавить стадию автоматического релиза в prestable
    - `release_to_stable_manually` — релизить в stable только по кнопке

    """

    @property
    def name(self):
        raise NotImplementedError

    @property
    def display_name(self):
        return "{} (Sandbox Task)".format(self.target_task_type)

    @property
    def responsible(self):
        raise NotImplementedError

    @property
    def target_task_path(self):
        """
        Target task path relative to Arcadia root.
        E.g. sandbox/projects/common/build/YaMake2/bin
        """
        raise NotImplementedError

    @property
    def target_task_path_no_bin(self):

        path = self.target_task_path

        if path.endswith("/"):
            path = path[:-1]

        if path.endswith("/bin"):
            path = path[:-4]

        return path

    @property
    def target_task_type(self):
        """Type of the target task (SCREAMING_SNAKE_CASE)"""
        return self.infer_task_type_from_task_path()

    @property
    def target_bin_name(self):
        """Name of the target task binary. E.g. 'ya_make2_runner'"""
        return self.target_task_type.lower()

    @property
    def target_task_ci_registry_location(self):
        return "projects/{project_name}/{task_type}".format(
            project_name=self.responsible.abc.service_name,
            task_type=self.target_task_type.lower(),
        )

    @property
    def task_binary_resource(self):
        return "SANDBOX_TASKS_BINARY"

    @property
    def task_binary_resource_additional_attributes(self):
        return None

    @property
    def test_run_input(self):
        """
        Input for test run. If empty list then no test run is going to be performed. Otherwise the return value
        should be an instance of `jg_cube.CubeInput` or `List[jg_cube.CubeInput]` and provide a complete input
        for the set of task test runs. The run of all tests is performed right after testing release and is run
        with binary_executor_release_type = testing
        """
        return []

    @property
    def add_precommit_check(self):
        return False

    @property
    def add_prestable_auto_release(self):
        return False

    @property
    def release_to_stable_manually(self):
        return False

    def infer_task_type_from_task_path(self):

        path = self.target_task_path_no_bin

        candidate_name = path.rsplit("/", 1)[-1]
        candidate_name = re.sub("([a-z0-9])([A-Z])", r"\1_\2", candidate_name).lower()
        candidate_name = re.sub("([a-z])([0-9]+)", r"\1_\2", candidate_name).upper()

        return candidate_name

    class JG(jg_base.BaseReleaseMachineJobGraph):

        def _get_build_cube(self):
            return build_cubes.DeployBinaryTask(
                target=self.root_cfg.target_task_path,
                target_task_type=self.root_cfg.target_task_type,
                name="build",
                attrs=self.root_cfg.task_binary_resource_additional_attributes,
            )

        def _get_test_run_cube(self, test_run_input, test_index, attributes=None):
            return TestRunCube(
                name="{}_test_run_{}".format(self.root_cfg.target_task_type.lower(), test_index),
                title="{} Test Run #{}".format(self.root_cfg.target_task_type.lower(), test_index),
                task=self.root_cfg.target_task_ci_registry_location,
                input=test_run_input,
                attributes=attributes,
            )

        def _get_test_run_input_list(self):
            test_run_inputs = self.root_cfg.test_run_input

            if isinstance(test_run_inputs, jg_cube.CubeInput):
                return [test_run_inputs]
            else:
                return test_run_inputs

        @jg_flow.release_flow(
            auto=True,
            stages=RELEASE_FLOW_STAGES,
        )
        def release(self):

            graph = super(SandboxSingleTaskReleaseCfg.JG, self).release(self)

            task_releasable_item = self.root_cfg.releases_cfg.releasable_items[0]

            build = self._get_build_cube()
            build.add_requirement(graph.get(dummy_cubes.RMMainGraphEntry.NAME))

            testing_release = ReleaseSbTaskCubeTesting(
                component_name=self.root_cfg.name,
                where_to_release=rm_const.ReleaseStatus.testing,
                manual=False,
                input=jg_cube.CubeInput(
                    component_resources={
                        task_releasable_item.name: build.output.resources[
                            task_releasable_item.data.resource_type
                        ].first().id,
                    },
                    deploy_system=rm_const.DeploySystem.kosher_sandbox_release.name,
                ),
                needs=[graph.get(internal_cubes.CreateChangelog.NAME)],
            )

            if self.root_cfg.notify_cfg.use_startrek:
                testing_release.add_requirement(graph.get(internal_cubes.CreateStartrekTicket.NAME))
                testing_release.add_requirement(graph.get(internal_cubes.FormatChangelog.NAME))
                testing_release.add_requirement(graph.get(internal_cubes.PostChangelogToStartrek.NAME))
                testing_release.add_requirement(graph.get(internal_cubes.LinkFeatureTickets.NAME))

            test_run_inputs = self._get_test_run_input_list()
            test_runs = []

            for test_index, test_run_input in enumerate(test_run_inputs):

                test_run_input.update(
                    binary_executor_release_type="testing",
                )

                test_run = self._get_test_run_cube(
                    test_run_input,
                    test_index,
                    attributes={
                        "requirements": {
                            "sandbox": {
                                "tasks_resource": jg_cube.CubeInput.format_cube_output_value(
                                    build.output.resources[self.root_cfg.task_binary_resource].first().id,
                                ),
                            },
                        },
                    },
                )
                test_run.add_requirement(testing_release)
                test_runs.append(test_run)

            release_needs = [testing_release]

            for test_run in test_runs:
                release_needs.append(test_run)

            if self.root_cfg.add_prestable_auto_release:
                prestable_release = ReleaseSbTaskCubeStable(
                    component_name=self.root_cfg.name,
                    where_to_release=rm_const.ReleaseStatus.prestable,
                    manual=False,
                    input=jg_cube.CubeInput(
                        component_resources={
                            task_releasable_item.name: build.output.resources[
                                task_releasable_item.data.resource_type
                            ].first().id,
                        },
                        deploy_system=rm_const.DeploySystem.kosher_sandbox_release.name,
                    ),
                    needs=release_needs,
                )
                release_needs = [prestable_release]
            else:
                prestable_release = None

            stable_release = ReleaseSbTaskCubeStable(
                component_name=self.root_cfg.name,
                where_to_release=rm_const.ReleaseStatus.stable,
                manual=self.root_cfg.release_to_stable_manually,
                input=jg_cube.CubeInput(
                    component_resources={
                        task_releasable_item.name: build.output.resources[
                            task_releasable_item.data.resource_type
                        ].first().id,
                    },
                    deploy_system=rm_const.DeploySystem.kosher_sandbox_release.name,
                ),
                needs=release_needs,
            )

            graph.add(build)
            graph.add(testing_release)
            for test_run in test_runs:
                graph.add(test_run)
            if prestable_release is not None:
                graph.add(prestable_release)
            graph.add(stable_release)

            return graph

        @jg_flow.precommit_check_flow()
        def precommit_check(self):

            if not self.root_cfg.add_precommit_check:
                return

            test_run_inputs = self._get_test_run_input_list()

            if not test_run_inputs:
                return

            build = self._get_build_cube()

            cubes = [build]

            for test_index, test_run_input in enumerate(test_run_inputs):
                test_run_input.update(
                    binary_executor_release_type="custom",
                )

                test_run = self._get_test_run_cube(
                    test_run_input,
                    test_index,
                    attributes={
                        "requirements": {
                            "sandbox": {
                                "tasks_resource": jg_cube.CubeInput.format_cube_output_value(
                                    build.output.resources[self.root_cfg.task_binary_resource].first().id,
                                ),
                            },
                        },
                    },
                )

                test_run.add_requirement(build)
                cubes.append(test_run)

            return graph_base.Graph(cubes)

    class CI(configs.ReferenceCIConfig.CI):

        secret = "sec-01desry8fbgvnkbeybem81ferv"

        @property
        def a_yaml_dir(self):
            return self.root_cfg.target_task_path_no_bin

        @property
        def ya_make_abs_paths_glob(self):
            return [
                os.path.join(self.root_cfg.target_task_path_no_bin, "**"),
            ]

    class Releases(configs.ReferenceCIConfig.Releases):

        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name=self.root_cfg.target_task_type,
                    data=ri.SandboxResourceData(
                        self.root_cfg.task_binary_resource,
                        ttl=14,
                        attributes={
                            "task_type": self.root_cfg.target_task_type,
                        },
                    ),
                    deploy_infos=[
                        ri.SandboxKosherReleaseInfo(stage=rm_const.ReleaseStatus.stable),
                        ri.SandboxKosherReleaseInfo(stage=rm_const.ReleaseStatus.prestable),
                        ri.SandboxKosherReleaseInfo(stage=rm_const.ReleaseStatus.testing),
                    ],
                ),
            ]

        allow_robots_to_release_stable = True
        allow_old_releases = True
        wait_for_deploy_time_sec = 10 * 60  # 10 min
        main_release_flow_branch_auto_start = True

    class Notify(configs.ReferenceCIConfig.Notify):
        use_startrek = False

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = None

        @property
        def ya_make_targets(self):
            return [
                self.root_cfg.target_task_path,
            ]


class SandboxSingleTaskReleaseWithTestRollbackCfg(SandboxSingleTaskReleaseCfg):
    """
    Makes test stage rollbackable

    The default SandboxSingleTaskReleaseCfg only marks the 'release' stage as rollbackable. Thus the release-to-testing
    cube (which belongs to the 'test' stage) is not triggered in the rollback flow.

    SandboxSingleTaskReleaseWithTestRollbackCfg marks 'test' stage with rollback=True

    Note: It is up to the class users to make sure that any other cube on the 'test' stage supports rollbacks
    """

    class JG(SandboxSingleTaskReleaseCfg.JG):
        @jg_flow.release_flow(
            auto=True,
            stages=RELEASE_FLOW_STAGES_WITH_ROLLBACKABLE_TESTING,
        )
        def release(self):
            return super(SandboxSingleTaskReleaseWithTestRollbackCfg.JG, self).release(self)
