from sandbox.common.format import ident
import sandbox.common.types.misc as ctm

import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build

from sandbox.projects.quasar.platform import ANDROID_PLATFORMS, get_platform_path_key

from sandbox.projects.release_machine.components.configs.smart_devices import _common as common


class BuildDaemonsJob(jg_build.JobGraphElementBuildBranched):
    JOB_TYPE = 'BUILD_DAEMONS'

    def __init__(self, platform, revision=None, factory=False, job_name_parameter=""):
        job_params = dict(
            job_type=self.JOB_TYPE,
            apiargs=dict(
                requirements=dict(
                    disk_space=30 << 30,  # in bytes
                    platform="linux_ubuntu_18.04_bionic",
                ),
            ),
            ignored_paths=['^arcadia/smart_devices/android/tv'],
        )

        if factory:
            packages_path_template = 'smart_devices/platforms/{}/factory/packages'
            resource_type = 'QUASAR_FACTORY_DAEMONS'
            job_params.update(dict(
                frequency=(jg_utils.TestFrequency.LAZY, None),
            ))
        else:
            packages_path_template = 'smart_devices/platforms/{}/packages'
            resource_type = 'QUASAR_DAEMONS'

        packages_path = packages_path_template.format(
            get_platform_path_key(platform, revision)
        )

        super(BuildDaemonsJob, self).__init__(
            task_name='YA_PACKAGE',
            build_item=job_name_parameter,
            job_params=job_params,
            ctx=dict(
                build_type='minsizerel',
                package_type='tarball',
                raw_package=True,
                ignore_recurses=True,
                use_aapi_fuse=True,
                aapi_fallback=True,
                publish_package=False,
                resource_type=resource_type,
                clear_build=True,
                build_system='ya_force',  # force local build
                env_vars="QUASAR_PLATFORM_KEY_PATH='$(vault:file:QUASAR:quasar_daemons_{}_key)'".format(platform),
                packages='{}/daemons.json'.format(packages_path),
            ),
            out={
                resource_type: common.RESOURCE_TTL,
            },
            job_arrows=(
                jg_job_triggers.JobTriggerNewTag((
                    common.ParentDataOutputToDict(
                        input_key='package_resource_attrs',
                        dict_key='major_release_num',
                        output_key='branch_number_for_tag',
                    ),
                    common.ParentDataOutputToDict(
                        input_key='package_resource_attrs',
                        dict_key='minor_release_num',
                        output_key='new_tag_number',
                    ),
                    common.JustPassDataToDict(
                        input_key='package_resource_attrs',
                        dict_key='quasar_platform',
                        value=platform,
                    ),
                    common.JustPassDataToDict(
                        input_key='package_resource_attrs',
                        dict_key='quasar_platform_revision',
                        value=revision,
                    ),
                )),
            ),
        )


class BuildAppJob(jg_build.JobGraphElementBuildBranched):
    JOB_TYPE = 'BUILD_APP'

    def __init__(self, job_name_parameter=""):
        super(BuildAppJob, self).__init__(
            task_name='YA_PACKAGE',
            build_item=job_name_parameter,
            job_params=dict(
                job_type=self.JOB_TYPE,
                apiargs=dict(
                    requirements=dict(
                        disk_space=30 << 30,  # in bytes
                        dns=ctm.DnsType.DNS64,
                        platform="linux_ubuntu_18.04_bionic",
                    ),
                ),
                ignored_paths=['^arcadia/smart_devices/android/tv'],
            ),
            ctx=dict(
                build_type='minsizerel',
                package_type='tarball',
                raw_package=True,
                ignore_recurses=True,
                use_aapi_fuse=True,
                aapi_fallback=True,
                publish_package=False,
                resource_type='QUASAR_APP',
                clear_build=True,
                build_system='ya_force',  # force local build
                env_vars='YANDEX_SIGNER_FILE=$(vault:file:QUASAR:robot-quasar-signer-oauth-token)',
                packages='smart_devices/android/packages/apk/apk.json',
            ),
            out={
                'QUASAR_APP': common.RESOURCE_TTL,
            },
            job_arrows=(
                jg_job_triggers.JobTriggerNewTag((
                    common.ParentDataOutputToDict(
                        input_key='package_resource_attrs',
                        dict_key='major_release_num',
                        output_key='branch_number_for_tag',
                    ),
                    common.ParentDataOutputToDict(
                        input_key='package_resource_attrs',
                        dict_key='minor_release_num',
                        output_key='new_tag_number',
                    ),
                )),
            ),
        )


class BuildTvAppJob(jg_build.JobGraphElementBuildBranched):
    JOB_TYPE = 'BUILD_TV_APP'

    def __init__(self, config_path, build_results_root=None, extra_artifacts=None, env=None, job_name_parameter=""):
        super(BuildTvAppJob, self).__init__(
            task_name='TEAMCITY_APK_BUILD_WRAPPER',
            build_item=job_name_parameter,
            job_params=dict(
                job_type=self.JOB_TYPE,
            ),
            ctx=dict(
                config_path=config_path,
                build_results_root=build_results_root,
                extra_artifacts=extra_artifacts or {},
                ssh_key='QUASAR:robot-quasar-ssh-private-key',
                env=env,
            ),
            out={
                'ANDROID_TV_APP': common.RESOURCE_TTL,
            },
            job_arrows=(
                jg_job_triggers.JobTriggerNewTag((
                    common.ParentDataOutputToDict(
                        input_key='resource_attributes',
                        dict_key='major_release_num',
                        output_key='branch_number_for_tag',
                    ),
                    common.ParentDataOutputToDict(
                        input_key='resource_attributes',
                        dict_key='minor_release_num',
                        output_key='new_tag_number',
                    ),
                    common.JustPassDataToDict(
                        input_key='resource_attributes',
                        dict_key='is_system',
                        value=True,
                    ),
                    jg_job_data.ParentDataCtx(
                        input_key='commit',
                        output_key='arc_hash',
                    ),
                    jg_job_data.ParentDataCtx(
                        input_key="branch",
                        output_key="result_path",
                        transform=lambda x, params: x.replace("arcadia:/arc/", ""),
                    ),
                )),
            ),
        )


class CreateTvAppResourceIdsConfig(jg_build.JobGraphElementBuildBranched):
    JOB_TYPE = 'CREATE_TV_APP_RESOURCE_IDS_CONFIG'

    def __init__(self, job_arrows, job_name_parameter=""):
        super(CreateTvAppResourceIdsConfig, self).__init__(
            task_name='CREATE_TEXT_RESOURCE_2',
            build_item=job_name_parameter,
            job_params=dict(
                job_type=self.JOB_TYPE,
            ),
            ctx=dict(
                resource_type='ANDROID_TV_APP_RESOURCE_IDS_CONFIG',
                created_resource_name='artifact_downloader_dependencies.json',
                input_type='json',
            ),
            out={
                'ANDROID_TV_APP_RESOURCE_IDS_CONFIG': common.RESOURCE_TTL,
            },
            job_arrows=job_arrows,
        )


class BuildCentaurLauncherAppJob(jg_build.JobGraphElementBuildBranched):
    JOB_TYPE = 'BUILD_APP'

    def __init__(self, job_name_parameter="centaur"):
        super(BuildCentaurLauncherAppJob, self).__init__(
            task_name='YA_PACKAGE',
            build_item=job_name_parameter,
            job_params=dict(
                job_type=self.JOB_TYPE,
                apiargs=dict(
                    requirements=dict(
                        disk_space=30 << 30,  # in bytes
                        platform="linux_ubuntu_18.04_bionic",
                        dns=ctm.DnsType.DNS64,
                    ),
                ),
                frequency=(jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                ignored_paths=['^arcadia/smart_devices/android/tv'],
            ),
            ctx=dict(
                build_type='minsizerel',
                package_type='tarball',
                raw_package=True,
                ignore_recurses=True,
                use_aapi_fuse=True,
                aapi_fallback=True,
                publish_package=False,
                resource_type='QUASAR_CENTAUR_DEMO_APP',
                clear_build=True,
                build_system='ya_force',  # force local build
                env_vars='YANDEX_SIGNER_FILE=$(vault:file:QUASAR:robot-quasar-signer-oauth-token)',
                packages='smart_devices/android/packages/centaur/centaur-app.json'
            ),
            out={
                'QUASAR_CENTAUR_DEMO_APP': common.RESOURCE_TTL,
            },
            job_arrows=(
                jg_job_triggers.JobTriggerNewTag((
                    common.ParentDataOutputToDict(
                        input_key='package_resource_attrs',
                        dict_key='major_release_num',
                        output_key='branch_number_for_tag',
                    ),
                    common.ParentDataOutputToDict(
                        input_key='package_resource_attrs',
                        dict_key='minor_release_num',
                        output_key='new_tag_number',
                    ),
                    common.JustPassDataToDict(
                        input_key='package_resource_attrs',
                        dict_key='build_apk_name',
                        value='centaur-app-launcher-v8-release',
                    ),
                    common.JustPassDataToDict(
                        input_key='package_resource_attrs',
                        dict_key='name',
                        value='ru.yandex.quasar.centaur_app',
                    ),
                    common.JustPassDataToDict(
                        input_key='package_resource_attrs',
                        dict_key='verbose_name',
                        value='CentaurApp',
                    ),
                    common.JustPassDataToDict(
                        input_key='package_resource_attrs',
                        dict_key='is_system',
                        value=True,
                    ),
                    common.JustPassDataToDict(
                        input_key='package_resource_attrs',
                        dict_key='generation',
                        value=0,
                    ),
                    common.JustPassDataToDict(
                        input_key='package_resource_attrs',
                        dict_key='comment',
                        value='upload apk from CentaurLauncherAppJob',
                    ),
                    common.JustPassDataToDict(
                        input_key='package_resource_attrs',
                        dict_key='platform',
                        value='centaur',
                    )
                ))
            )
        )


class PublishApkToQuasmodromJob(common.BaseTaskletRunJob):
    JOB_TYPE = 'PUBLISH_APP'

    def __init__(
            self,
            job_arrows,
            job_name_parameter="",
    ):
        requirements = dict(
            disk_space=5 << 30
        )
        super(PublishApkToQuasmodromJob, self).__init__(
            job_type=self.JOB_TYPE,
            tasklet_name='ApkToQuasmodrom',
            job_name_parameter=job_name_parameter,
            job_params=dict(
                apiargs=dict(
                    requirements=requirements,
                ),
            ),
            job_arrows=job_arrows,
        )


class BuildImageJob(jg_build.JobGraphElementBuildBranched):
    JOB_TYPE = 'BUILD_IMAGE'

    def __init__(
            self, platform, revision=None,
            build_type=None, sign=None,
            job_name_parameter="",
            build_daemons_job_name_parameter="",
            build_app_job_name_parameter="",
            build_factory_job_name_parameter=None,
    ):
        ctx = {'keep_resources': True}
        if sign is not None:
            ctx['sign'] = sign
        if build_type is not None:
            ctx['build_type'] = build_type
        if revision is not None:
            ctx['quasar_platform_revision'] = revision

        job_arrows = [
            jg_arrows.JobTrigger(
                BuildDaemonsJob.JOB_TYPE,
                job_name_parameter=build_daemons_job_name_parameter,
                parent_job_data=(
                    jg_job_data.ParentDataResource(
                        input_key='quasar_daemons',
                        resource_name='QUASAR_DAEMONS',
                    ),
                ),
            ),
            jg_job_triggers.JobTriggerNewTag(
                parent_job_data=(
                    jg_job_data.ParentDataOutput(
                        input_key='quasar_minor_version',
                        output_key='new_tag_number',
                    ),
                    jg_job_data.ParentDataOutput(
                        input_key='quasar_major_version',
                        output_key='branch_number_for_tag',
                    ),
                ),
            ),
        ]

        if platform in ANDROID_PLATFORMS:
            job_arrows.append(jg_arrows.JobTrigger(
                BuildAppJob.JOB_TYPE,
                job_name_parameter=build_app_job_name_parameter,
                parent_job_data=(
                    jg_job_data.ParentDataResource(
                        input_key='quasar_app',
                        resource_name='QUASAR_APP',
                    ),
                ),
            ))

        if build_factory_job_name_parameter is not None:
            job_arrows.append(jg_arrows.JobTrigger(
                BuildDaemonsJob.JOB_TYPE,
                job_name_parameter=build_factory_job_name_parameter,
                parent_job_data=(
                    jg_job_data.ParentDataResource(
                        input_key='quasar_daemons_factory',
                        resource_name='QUASAR_FACTORY_DAEMONS',
                    ),
                ),
            ))

        job_arrows.append(jg_arrows.ValueAndParamsData(
            input_key='repository_config_dir',
            value='{{svn_ssh_url}}/arcadia/smart_devices/platforms/{}/firmware@{{revision}}'.format(
                get_platform_path_key(platform),  # add paltform
            ),
            transform=lambda y, x, rm_config: y.format(
                svn_ssh_url=x.svn_ssh_url,
                revision=x.revision,
            ),
        ))

        super(BuildImageJob, self).__init__(
            job_params=dict(
                job_type=self.JOB_TYPE,
                ignored_paths=['^arcadia/smart_devices/android/tv'],
            ),
            task_name='QUASAR_BUILD_{}_IMAGE'.format(ident(platform)),
            build_item=job_name_parameter,
            ctx=ctx,
            out={
                common.ReleaseItemsNames.ota(platform): common.RESOURCE_TTL,
            },
            job_arrows=job_arrows,
        )


class BuildTVImageJob(jg_build.JobGraphElementBuildBranched):
    JOB_TYPE = 'BUILD_IMAGE'

    def __init__(
            self, platform, revision=None,
            sign=None, factory=False,
            build_type=None, apps_build_type=None,
            job_name_parameter="",
            apps_job_arrows=None,
    ):
        ctx = {
            'keep_resources': True,
            'factory': factory,
            'repository_config_dir': 'smart_devices/platforms/{}/firmware'.format(
                get_platform_path_key(platform)),
            'artifact_downloader_arcadia_config': 'smart_devices/platforms/{}/firmware/prebuilt.yaml'.format(
                platform),
        }
        if sign is not None:
            ctx['sign'] = sign
        if build_type is not None:
            ctx['build_type'] = build_type
        if revision is not None:
            ctx['quasar_platform_revision'] = revision

        job_arrows = [
            jg_job_triggers.JobTriggerNewTag(
                parent_job_data=(
                    jg_job_data.ParentDataOutput(
                        input_key='apps_minor_version',
                        output_key='new_tag_number',
                    ),
                    jg_job_data.ParentDataOutput(
                        input_key='apps_major_version',
                        output_key='branch_number_for_tag',
                    ),
                ),
            ),
        ]

        if apps_job_arrows:
            job_arrows.extend(apps_job_arrows)

        super(BuildTVImageJob, self).__init__(
            job_params=dict(
                job_type=self.JOB_TYPE,
            ),
            task_name='QUASAR_BUILD_{}_IMAGE'.format(ident(platform)),
            build_item=job_name_parameter,
            ctx=ctx,
            out={
                common.ReleaseItemsNames.ota(platform): common.RESOURCE_TTL,
            },
            job_arrows=job_arrows,
        )


class PublishImageToS3Job(common.BaseTaskletRunJob):
    JOB_TYPE = 'PUBLISH_TO_S3'

    def __init__(
            self, platform, resource_filters,
            job_name_parameter="",
            build_image_job_name_parameter="",
    ):
        requirements = dict()
        if platform in ANDROID_PLATFORMS:
            requirements['disk_space'] = 5 << 30  # in bytes

        super(PublishImageToS3Job, self).__init__(
            job_type=self.JOB_TYPE,
            tasklet_name='PublishOtaToS3',
            job_name_parameter=job_name_parameter,
            job_params=dict(
                apiargs=dict(
                    requirements=requirements,
                ),
            ),
            job_arrows=(
                jg_arrows.JobTrigger(
                    job_type=BuildImageJob.JOB_TYPE,
                    job_name_parameter=build_image_job_name_parameter,
                    parent_job_data=(
                        common.ParentDataOutputToDict(
                            input_key='tasklet_input',
                            dict_key='resources_ids',
                            output_key='ota_images',
                        ),
                        common.JustPassDataToDict(
                            input_key='tasklet_input',
                            dict_key='resources_filters',
                            value=resource_filters or {},
                        ),
                    ),
                ),
            ),
        )


class PublishImageToQuasmodromJob(common.BaseTaskletRunJob):
    JOB_TYPE = 'PUBLISH_TO_QD'

    def __init__(
            self, platform,
            add_ticket_in_comment=True,
            job_name_parameter="",
            publish_to_s3_job_name_parameter="",
            trunk=False,
    ):
        job_arrows = [
            jg_arrows.JobTrigger(
                job_type=PublishImageToS3Job.JOB_TYPE,
                job_name_parameter=publish_to_s3_job_name_parameter,
                parent_job_data=(
                    common.ParentDataOutputToDict(
                        input_key='tasklet_input',
                        dict_key='resources_ids',
                        output_key='tasklet_output',
                        transform=lambda output, _: output['resourcesLinks'].keys(),
                    ),
                    common.JustPassDataToDict(
                        input_key='tasklet_input',
                        dict_key='rm_platform',
                        value=platform,
                    ),
                    common.JustPassDataToDict(
                        input_key='tasklet_input',
                        dict_key='add_ticket_in_comment',
                        value=add_ticket_in_comment,
                    ),
                    common.JustPassDataToDict(
                        input_key='tasklet_input',
                        dict_key='fail_on_empty_input',
                        value=True,
                    ),
                ),
            ),
        ]

        if not trunk:
            job_arrows.append(
                jg_job_triggers.JobTriggerNewTag((
                    common.ParentDataOutputToDict(
                        input_key='tasklet_input',
                        dict_key='rm_component',
                        output_key='component_name',
                    ),
                    common.ParentDataOutputToDict(
                        input_key='tasklet_input',
                        dict_key='rm_release_number',
                        output_key='branch_number_for_tag',
                    ),
                    common.ParentDataOutputToDict(
                        input_key='tasklet_input',
                        dict_key='comment',
                        output_key='new_tag_number',
                        transform=lambda tag, params: '{}-{}: {}'.format(
                            params.db.db_name, tag, params.revision,
                        ),
                    ),
                )),
            )

        super(PublishImageToQuasmodromJob, self).__init__(
            job_type=self.JOB_TYPE,
            tasklet_name='OtaToQuasmodrom',
            job_name_parameter=job_name_parameter,
            job_arrows=job_arrows,
        )
