# -*- coding: utf-8 -*-
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components import configs as rm_configs

from sandbox.projects.release_machine.components.configs.smart_devices import _common as common
from sandbox.projects.release_machine.components.configs.smart_devices import _release_part as release_part

from sandbox.projects.quasar.platform import get_platform_revisions
from sandbox.projects.quasar.platform import get_effective_revisions
from sandbox.projects.quasar.platform import get_mono_revision
from sandbox.projects.quasar.platform import has_mono_revision
from collections import defaultdict


class DeviceJobGraphBase(object):
    def __init__(self, platform, component_name,
                 create_new_gerrit_branch_on_prerelease=False,
                 rm_branch_name=None, is_factory_rm_branch=False):
        self.platform = platform
        self.create_new_gerrit_branch_on_prerelease = create_new_gerrit_branch_on_prerelease
        self.component_name = component_name
        self.rm_branch_name = rm_branch_name
        self.is_factory_rm_branch = is_factory_rm_branch

    @property
    def release_items(self):
        release_items = []
        if len(self.ota_build_types) > 1:
            release_items.append(
                rm_configs.ReleasedResourceInfo(
                    name=common.ReleaseItemsNames.ota(self.platform),
                    resource_type=common.ReleaseItemsNames.ota(self.platform),
                    attributes=dict(
                        buildtype='user',
                        signed=True,
                    ),
                )
            )
        elif len(self.ota_build_types):
            release_items.append(
                rm_configs.ReleasedResourceInfo(
                    name=common.ReleaseItemsNames.ota(self.platform),
                    resource_type=common.ReleaseItemsNames.ota(self.platform),
                ),
            )
        return release_items

    @property
    def image_build_types(self):
        return [None]

    @property
    def image_build_type_to_release(self):
        return None

    def get_apps_build_types(self, image_build_type):
        return [None]

    def apps_build_type_to_release(self, stage):
        return None

    @property
    def ota_build_types(self):
        return self.image_build_types

    @property
    def platform_revisions(self):
        return get_platform_revisions(self.platform)

    @property
    def effective_revisions(self):
        return get_effective_revisions(self.platform)

    @property
    def release_stages_of_groups(self):
        return {
            'beta': rm_const.ReleaseStatus.prestable,
            'default': rm_const.ReleaseStatus.stable,
            'release_test': rm_const.ReleaseStatus.stable,
            'release_baseline': rm_const.ReleaseStatus.stable,
        }

    def make_build_variant_name(self, image_build_type, apps_build_type, revision):
        build_variant = '_'.join(z for z in (revision, image_build_type, apps_build_type) if z)
        return build_variant if build_variant else None

    def get_all_build_variant_names(self):
        build_variants = []
        for revision in self.platform_revisions:
            for image_build_type in self.ota_build_types:
                for apps_build_type in self.get_apps_build_types(image_build_type):
                    build_variants.append(self.make_build_variant_name(image_build_type, apps_build_type, revision))
        return build_variants

    def get_all_release_build_variant_names(self):
        build_variants = []
        for revision in self.platform_revisions:
            build_variants.append(
                self.make_build_variant_name(
                    self.image_build_type_to_release,
                    self.apps_build_type_to_release(rm_const.ReleaseStatus.testing),
                    revision
                )
            )
        return build_variants

    def revision_job_parameter(self, revision):
        res = [self.platform]
        if revision:
            res.append(revision)
        return '_'.join(res)

    def build_type_job_parameter(self, image_build_type, apps_build_type, revision):
        res = [self.revision_job_parameter(revision)]
        if image_build_type:
            res.append(image_build_type.replace(' ', '_'))
        if apps_build_type:
            res.append(apps_build_type.replace(' ', '_'))
        return '_'.join(res)

    def create_common_release_part(self, upload_symbols=False, has_regress=False, run_kolhoz=False):
        platform_job_parameter = self.platform
        jobs = [
            release_part.PlatformChangelogJob(
                self.platform,
                job_name_parameter=platform_job_parameter,
            ),
            release_part.CreateStartrekReleaseTicketJob(
                self.platform,
                job_name_parameter=platform_job_parameter,
                changelog_job_name_parameter=platform_job_parameter,
            ),
        ]
        if has_regress:
            jobs.append(release_part.CreateStartrekAssessorsTicketJob(
                platform=self.platform,
                release_build_variants=self.get_all_release_build_variant_names(),
                job_name_parameter=self.platform,
                create_st_job_name_parameter=platform_job_parameter,
            ))

        upload_symbols_parameters = []
        if upload_symbols:
            (upload_symbols_jobs, upload_symbols_parameters) = self.create_upload_symbols_jobs()
            jobs.extend(upload_symbols_jobs)

        # prepare storage for mono updates in each group
        mono_revision_group_updates_parameter = defaultdict(list)

        revision_st_updates = []
        mono_revision_st_updates = []
        mono_revision_job_parameter = None
        if has_mono_revision(self.platform):
            mono_revision_job_parameter = self.revision_job_parameter(get_mono_revision(self.platform))

        # create additional jobs only for effective revisions
        for revision in self.effective_revisions:
            revision_job_parameter = self.revision_job_parameter(revision)

            for image_build_type in self.ota_build_types:
                for apps_build_type in self.get_apps_build_types(image_build_type):
                    build_type_job_parameter = self.build_type_job_parameter(image_build_type, apps_build_type, revision)
                    build_variant = self.make_build_variant_name(image_build_type, apps_build_type, revision)
                    jobs.append(release_part.UpdateStartrekReleaseTicketJob(
                        self.platform, self.component_name, build_variant,
                        job_name_parameter=build_type_job_parameter,
                        create_st_job_name_parameter=platform_job_parameter,
                        publish_to_qd_job_name_parameter=build_type_job_parameter,
                    ))
                    revision_st_updates.append(build_type_job_parameter)
                    if mono_revision_job_parameter and revision == get_mono_revision(self.platform):
                        mono_revision_st_updates.append(build_type_job_parameter)

            testing_job_name_parameter = '_'.join((revision_job_parameter, rm_const.ReleaseStatus.testing))
            release_build_type_parameter = self.build_type_job_parameter(
                self.image_build_type_to_release,
                self.apps_build_type_to_release(rm_const.ReleaseStatus.testing),
                revision
            )

            if run_kolhoz:
                jobs.append(
                    release_part.KolhozTestRunJob(
                        platform=self.platform,
                        job_name_parameter=testing_job_name_parameter,
                        publish_to_qd_job_name_parameter=release_build_type_parameter,
                        create_st_job_name_parameter=platform_job_parameter
                    )
                )

            if has_regress:
                release_build_variant = self.make_build_variant_name(
                    self.image_build_type_to_release,
                    self.apps_build_type_to_release(rm_const.ReleaseStatus.testing),
                    revision
                )
                jobs.append(release_part.UpdateStartrekAssessorsTicketJob(
                    platform=self.platform,
                    release_build_variant=release_build_variant,
                    job_name_parameter=testing_job_name_parameter,
                    create_assessors_job_name_parameter=self.platform,
                    publish_to_qd_job_name_parameter=release_build_type_parameter,
                ))

        # loop through all revisions to prepare proper group updates
        for revision in self.platform_revisions:
            revision_job_parameter = self.revision_job_parameter(revision)

            # Create mono revision update jobs if required
            if mono_revision_job_parameter:
                mono_revision = get_mono_revision(self.platform)
                for group, stage in self.release_stages_of_groups.items():
                    revision_group_parameter = group
                    if revision:  # non-mono revision for group name compilation
                        revision_group_parameter = '_'.join((group, revision))
                    mono_group_job_parameter = '_'.join((mono_revision_job_parameter, revision_group_parameter, "MONO"))
                    mono_release_build_type_parameter = self.build_type_job_parameter(
                        self.image_build_type_to_release,
                        self.apps_build_type_to_release(stage),
                        mono_revision  # mono-revision to be deployed
                    )
                    mono_revision_group_updates_parameter[group].append(mono_group_job_parameter)

                    jobs.append(
                        release_part.CreateQuasmodromUpdateJob(
                            revision_group_parameter, mono_group_job_parameter,
                            create_st_job_name_parameter=platform_job_parameter,
                            publish_to_qd_job_name_parameter=mono_release_build_type_parameter,
                        ))
            else:
                # Create regular update jobs
                for group, stage in self.release_stages_of_groups.items():
                    if revision:
                        group = '_'.join((group, revision))

                    group_job_parameter = '_'.join((revision_job_parameter, group))

                    release_build_type_parameter = self.build_type_job_parameter(
                        self.image_build_type_to_release,
                        self.apps_build_type_to_release(stage),
                        revision
                    )

                    jobs.extend((
                        release_part.CreateQuasmodromUpdateJob(
                            group, group_job_parameter,
                            create_st_job_name_parameter=platform_job_parameter,
                            publish_to_qd_job_name_parameter=release_build_type_parameter,
                        ),
                        # This action is shown as Release Item in RM UI (under "Release" button)
                        release_part.ReleaseActionJob(
                            group_job_parameter,
                            release_res_job_name_parameters=("__".join((revision_job_parameter, stage)),),
                            st_updates_job_name_parameters=revision_st_updates,
                            create_update_job_name_parameters=(group_job_parameter,),
                            upload_symbols_job_name_parameters=upload_symbols_parameters,
                        ),
                    ))

            testing_job_name_parameter = '_'.join((revision_job_parameter, rm_const.ReleaseStatus.testing))
            if not mono_revision_job_parameter:
                # This action is shown as Release Item in RM UI (under "Release" button)
                jobs.append(release_part.ReleaseActionJob(
                    job_name_parameter=testing_job_name_parameter,
                    release_res_job_name_parameters=(
                        "__".join((revision_job_parameter, rm_const.ReleaseStatus.testing)),
                    ),
                    st_updates_job_name_parameters=revision_st_updates,
                    regress_updates_job_name_parameter=testing_job_name_parameter if has_regress else None,
                    kolhoz_job_name_parameter=testing_job_name_parameter if run_kolhoz else None
                ))

        # Create release actions for mono-revision updates in RM
        if mono_revision_job_parameter:
            for group, stage in self.release_stages_of_groups.items():
                release_action_name = '_'.join((mono_revision_job_parameter, group, "ALL"))
                jobs.extend((
                    # This action is shown as Release Item in RM UI (under "Release" button)
                    release_part.ReleaseActionJob(
                        release_action_name,
                        release_res_job_name_parameters=("__".join((mono_revision_job_parameter, stage)),),
                        st_updates_job_name_parameters=mono_revision_st_updates,
                        create_update_job_name_parameters=mono_revision_group_updates_parameter[group],
                        upload_symbols_job_name_parameters=upload_symbols_parameters,
                    ),
                ))
            mono_revision_testing_job_name_parameter = '_'.join((mono_revision_job_parameter, rm_const.ReleaseStatus.testing))
            # This action is shown as Release Item in RM UI (under "Release" button)
            jobs.append(release_part.ReleaseActionJob(
                job_name_parameter='_'.join((mono_revision_job_parameter, rm_const.ReleaseStatus.testing)),
                release_res_job_name_parameters=(
                    "__".join((mono_revision_job_parameter, rm_const.ReleaseStatus.testing)),
                ),
                st_updates_job_name_parameters=mono_revision_st_updates,
                regress_updates_job_name_parameter=mono_revision_testing_job_name_parameter if has_regress else None,
                kolhoz_job_name_parameter=mono_revision_testing_job_name_parameter if run_kolhoz else None
            ))

        return jobs

    def create_upload_symbols_jobs(self):
        raise NotImplementedError()
