import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.job_graph_element as jg_element

from sandbox.projects.release_machine.components.configs.smart_devices import _common as common
from sandbox.projects.release_machine.components.configs.smart_devices import _branch_part as branch_part


class PlatformChangelogJob(jg_release.JobGraphElementChangelogFinal):
    JOB_TYPE = rm_const.JobTypes.CHANGELOG_FINAL

    def __init__(self, platform, job_name_parameter=""):
        super(PlatformChangelogJob, self).__init__(
            job_params=dict(
                job_type=self.JOB_TYPE,
                job_name_parameter=job_name_parameter,
                task_name="QUASAR_RELEASE_MACHINE_CHANGELOG",
                ctx=dict(
                    platform=platform,
                ),
            ),
        )


class CreateStartrekReleaseTicketJob(jg_element.JobGraphElement):
    JOB_TYPE = 'ST_CREATE'

    def __init__(
            self, platform,
            job_name_parameter="",
            changelog_job_name_parameter="",
    ):
        super(CreateStartrekReleaseTicketJob, self).__init__(
            job_params=dict(
                job_type=self.JOB_TYPE,
                job_name_parameter=job_name_parameter,
                task_name='CREATE_PLATFORM_STARTREK_TICKET',
                should_add_to_db=jg_utils.should_add_to_db_branch,
                apiargs=dict(
                    hidden=False,
                    requirements=dict(
                        disk_space=2 * (1024 ** 3),  # in Bytes
                    ),
                ),
                ctx=dict(
                    platform=platform,
                    update_issue_if_exists=False,
                )
            ),
            job_arrows=(
                jg_arrows.ParamsData(
                    input_key='additional_info',
                    transform=lambda params, _: params.custom_fields.get('additional_info', '') + '\n'.join([
                        'Intermediate db name: {}'.format(params.db.db_name),
                    ]),
                ),
                jg_arrows.JobTrigger(
                    job_type=PlatformChangelogJob.JOB_TYPE,
                    job_name_parameter=changelog_job_name_parameter,
                    parent_job_data=(
                        jg_job_data.ParentDataResource('changelog', 'RELEASE_MACHINE_CHANGELOG'),
                    ),
                ),
            ),
        )


class UpdateStartrekReleaseTicketJob(common.BaseTaskletRunJob):
    JOB_TYPE = 'ST_UPDATE'

    def __init__(
            self, platform, component_name,
            build_type=None,
            job_name_parameter="",
            create_st_job_name_parameter="",
            publish_to_qd_job_name_parameter="",
            publish_apk_to_qd_job_name_parameter=None,
    ):
        job_arrows = [
            jg_job_triggers.JobTriggerNewTag(
                parent_job_data=(
                    common.ParentDataOutputToDict(
                        input_key='tasklet_input',
                        dict_key='tag_number',
                            output_key='new_tag_number',
                        ),
                    common.ParentDataOutputToDict(
                        input_key='tasklet_input',
                        dict_key='branch_number',
                        output_key='branch_number_for_tag',
                    ),
                    common.JustPassDataToDict(
                        input_key='tasklet_input',
                        dict_key='component_name',
                        value=component_name,
                    ),
                    common.JustPassDataToDict(
                        input_key='tasklet_input',
                        dict_key='build_type',
                        value=build_type,
                    ),
                    common.JustPassDataToDict(
                        input_key='tasklet_input',
                        dict_key='platform',
                        value=platform,
                    ),
                ),
            ),
            jg_arrows.JobTrigger(
                job_type=CreateStartrekReleaseTicketJob.JOB_TYPE,
                job_name_parameter=create_st_job_name_parameter,
            ),
        ]

        if publish_apk_to_qd_job_name_parameter is not None:
            job_arrows.append(
                jg_arrows.JobTrigger(
                    job_type=branch_part.PublishApkToQuasmodromJob.JOB_TYPE,
                    job_name_parameter=publish_apk_to_qd_job_name_parameter,
                    parent_job_data=(
                        common.ParentDataOutputToDict(
                            input_key='tasklet_input',
                            dict_key='firmware_link',
                            output_key='tasklet_output',
                            transform=lambda output, _: output['applicationUrl']
                        ),
                    ),
                ),
            )
        else:
            job_arrows.append(
                jg_arrows.JobTrigger(
                    job_type=branch_part.PublishImageToQuasmodromJob.JOB_TYPE,
                    job_name_parameter=publish_to_qd_job_name_parameter,
                    parent_job_data=(
                        common.ParentDataOutputToDict(
                            input_key='tasklet_input',
                            dict_key='firmware_link',
                            output_key='tasklet_output',
                            transform=lambda output, _: output['firmwares'].values()[0]['url'],
                        ),
                    ),
                ),
            ),

        super(UpdateStartrekReleaseTicketJob, self).__init__(
            job_type=self.JOB_TYPE,
            tasklet_name='FirmwareToSt',
            job_name_parameter=job_name_parameter,
            job_arrows=job_arrows,
        )


class CreateStartrekAssessorsTicketJob(jg_element.JobGraphElement):
    JOB_TYPE = 'ST_CREATE_ASSESSORS'

    def __init__(
            self, platform,
            release_build_variants,
            job_name_parameter="",
            create_st_job_name_parameter="",
    ):
        super(CreateStartrekAssessorsTicketJob, self).__init__(
            job_params=dict(
                job_type=self.JOB_TYPE,
                job_name_parameter=job_name_parameter,
                task_name='SMART_DEVICES_CREATE_REGRESS_TICKET',
                should_add_to_db=jg_utils.should_add_to_db_branch,
                ctx=dict(
                    description="Creates regress runs, assessors ticket and links to release ticket",
                    target_platform=platform,
                    assessors_queue="ALICEASSESSORS",
                    testpalm_project="alice",
                    build_variants=release_build_variants
                )
            ),
            job_arrows=(
                jg_arrows.JobTrigger(
                    job_type=CreateStartrekReleaseTicketJob.JOB_TYPE,
                    job_name_parameter=create_st_job_name_parameter,
                    parent_job_data=(
                        jg_job_data.ParentDataOutput("release_ticket_key", "startrek_issue"),
                    ),
                )
            ),
        )


class UpdateStartrekAssessorsTicketJob(jg_element.JobGraphElement):
    JOB_TYPE = 'ST_UPDATE_ASSESSORS'

    def __init__(
        self, platform,
        release_build_variant,
        job_name_parameter="",
        create_assessors_job_name_parameter="",
        publish_to_qd_job_name_parameter="",
    ):
        super(UpdateStartrekAssessorsTicketJob, self).__init__(
            job_params=dict(
                job_type=self.JOB_TYPE,
                job_name_parameter=job_name_parameter,
                task_name='SMART_DEVICES_UPDATE_REGRESS_TICKET',
                should_add_to_db=jg_utils.should_add_to_db_branch,
                ctx=dict(
                    description="Provides config and firmware to assessors ticket",
                    target_platform=platform,
                    build_variant=release_build_variant
                )
            ),
            job_arrows=(
                jg_arrows.JobTrigger(
                    parent_job_data=(
                        jg_job_data.ParentDataOutput("assessors_ticket_key", "regress_ticket_key"),
                    ),
                    job_type=CreateStartrekAssessorsTicketJob.JOB_TYPE,
                    job_name_parameter=create_assessors_job_name_parameter,
                ),
                jg_job_triggers.JobTriggerNewTag(
                    parent_job_data=(
                        jg_job_data.ParentDataOutput("branch_number", "branch_number_for_tag"),
                        jg_job_data.ParentDataOutput("tag_number", "new_tag_number"),
                    ),
                ),
                jg_arrows.JobTrigger(
                    job_type=branch_part.PublishImageToQuasmodromJob.JOB_TYPE,
                    job_name_parameter=publish_to_qd_job_name_parameter,
                    parent_job_data=(
                        jg_job_data.ParentDataOutput(
                            "firmware_link",
                            "tasklet_output",
                            transform=lambda output, _: output['firmwares'].values()[0]['url']
                        ),
                    ),
                ),
            ),
        )


class CreateQuasmodromUpdateJob(common.BaseTaskletRunJob):
    JOB_TYPE = 'RELEASE_UPDATE'

    def __init__(
            self, group_name,
            job_name_parameter="",
            create_st_job_name_parameter="",
            publish_to_qd_job_name_parameter="",
    ):
        super(CreateQuasmodromUpdateJob, self).__init__(
            job_type=self.JOB_TYPE,
            tasklet_name='UpdateToQuasmodrom',
            job_name_parameter=job_name_parameter,
            job_arrows=(
                jg_arrows.JobTrigger(
                    job_type=CreateStartrekReleaseTicketJob.JOB_TYPE,
                    job_name_parameter=create_st_job_name_parameter,
                    parent_job_data=(
                        common.ParentDataOutputToDict(
                            input_key='tasklet_input',
                            dict_key='ticket',
                            output_key='startrek_issue',
                        ),
                        common.JustPassDataToDict(
                            input_key='tasklet_input',
                            dict_key='group_name',
                            value=group_name,
                        ),
                    ),
                ),
                jg_arrows.JobTrigger(
                    job_type=branch_part.PublishImageToQuasmodromJob.JOB_TYPE,
                    job_name_parameter=publish_to_qd_job_name_parameter,
                    parent_job_data=(
                        common.ParentDataOutputToDict(
                            input_key='tasklet_input',
                            dict_key='firmware_id',
                            output_key='tasklet_output',
                            transform=lambda output, _: output['firmwares'].values()[0]['id'],
                        ),
                    ),
                ),
            ),
        )


class ReleaseResourcesJob(jg_release.JobGraphElementReleaseBranched):
    JOB_TYPE = 'RELEASE_RES'

    def __init__(
            self, platform, release_stage,
            revision=None,
            build_daemons_job_name_parameter=None,
            build_image_job_name_parameter="",
            build_app_job_name_parameter=None,
            build_factory_job_name_parameter=None,
    ):
        job_arrows = [
            jg_arrows.JobTrigger(
                job_type=branch_part.BuildImageJob.JOB_TYPE,
                job_name_parameter=build_image_job_name_parameter,
                parent_job_data=(
                    jg_job_data.ParentDataDict(
                        input_key='component_resources',
                        dict_key=common.ReleaseItemsNames.ota(platform),
                        resource_name=common.ReleaseItemsNames.ota(platform),
                    )
                )
            ),
        ]

        if build_daemons_job_name_parameter is not None:
            job_arrows.append(jg_arrows.JobTrigger(
                job_type=branch_part.BuildDaemonsJob.JOB_TYPE,
                job_name_parameter=build_daemons_job_name_parameter,
                parent_job_data=(
                    jg_job_data.ParentDataDict(
                        input_key='component_resources',
                        dict_key=common.ReleaseItemsNames.daemons(platform),
                        resource_name='QUASAR_DAEMONS',
                    )
                )
            ))

        if build_app_job_name_parameter is not None:
            job_arrows.append(jg_arrows.JobTrigger(
                job_type=branch_part.BuildAppJob.JOB_TYPE,
                job_name_parameter=build_app_job_name_parameter,
                parent_job_data=(
                    jg_job_data.ParentDataDict(
                        input_key='component_resources',
                        dict_key=common.ReleaseItemsNames.app(platform),
                        resource_name='QUASAR_APP',
                    )
                )
            ))

        if build_factory_job_name_parameter is not None:
            job_arrows.append(jg_arrows.JobTrigger(
                job_type=branch_part.BuildDaemonsJob.JOB_TYPE,
                job_name_parameter=build_factory_job_name_parameter,
                parent_job_data=(
                    jg_job_data.ParentDataDict(
                        input_key='component_resources',
                        dict_key=common.ReleaseItemsNames.factory_daemons(platform),
                        resource_name='QUASAR_FACTORY_DAEMONS',
                    ),
                ),
            ))

        super(ReleaseResourcesJob, self).__init__(
            job_params=dict(
                job_type=self.JOB_TYPE,
            ),
            release_to=release_stage,
            release_item='_'.join(z for z in (platform, revision) if z),
            job_arrows=job_arrows,
        )


class ReleaseActionJob(jg_release.JobGraphElementActionReleaseBase):
    def __init__(
            self,
            job_name_parameter="",
            release_res_job_name_parameters=(),
            create_update_job_name_parameters=(),
            st_updates_job_name_parameters=(),
            upload_symbols_job_name_parameters=(),
            regress_updates_job_name_parameter=None,
            kolhoz_job_name_parameter=None
    ):
        job_arrows = []

        job_arrows.extend(
            jg_arrows.JobTrigger(
                job_type=ReleaseResourcesJob.JOB_TYPE,
                job_name_parameter=job_name_parameter
            )
            for job_name_parameter in release_res_job_name_parameters
        )

        job_arrows.extend(
            jg_arrows.JobTrigger(
                job_type=UploadSymbolsJob.JOB_TYPE,
                job_name_parameter=job_name_parameter,
            )
            for job_name_parameter in upload_symbols_job_name_parameters
        )

        job_arrows.extend(
            jg_arrows.JobTrigger(
                job_type=CreateQuasmodromUpdateJob.JOB_TYPE,
                job_name_parameter=job_name_parameter,
            )
            for job_name_parameter in create_update_job_name_parameters
        )

        job_arrows.extend(
            jg_arrows.JobTrigger(
                job_type=UpdateStartrekReleaseTicketJob.JOB_TYPE,
                job_name_parameter=job_name_parameter,
            )
            for job_name_parameter in st_updates_job_name_parameters
        )

        if kolhoz_job_name_parameter is not None:
            job_arrows.append(
                jg_arrows.JobTrigger(
                    job_type=KolhozTestRunJob.JOB_TYPE,
                    job_name_parameter=kolhoz_job_name_parameter,
                )
            )

        if regress_updates_job_name_parameter is not None:
            job_arrows.append(
                jg_arrows.JobTrigger(
                    job_type=UpdateStartrekAssessorsTicketJob.JOB_TYPE,
                    job_name_parameter=regress_updates_job_name_parameter,
                )
            )

        super(ReleaseActionJob, self).__init__(
            job_name_parameter=job_name_parameter,
            job_params={
                "should_add_to_db": jg_utils.should_add_to_db_branch,
                "cancel_fallbehind_runs_on_fix": False,
            },
            job_arrows=job_arrows,
        )


class UploadSymbolsJob(common.BaseTaskletRunJob):
    JOB_TYPE = 'UPLOAD_SYMBOLS'

    def __init__(
            self,
            job_name_parameter="",
            build_daemons_job_name_parameter="",
    ):
        super(UploadSymbolsJob, self).__init__(
            job_type=self.JOB_TYPE,
            tasklet_name='PublishSymbolsToS3',
            job_name_parameter=job_name_parameter,
            job_params=dict(
                apiargs=dict(
                    requirements=dict(
                        disk_space=1 * (1024 ** 3),  # in Bytes
                    ),
                )
            ),
            job_arrows=(
                jg_arrows.JobTrigger(
                    job_type=branch_part.BuildDaemonsJob.JOB_TYPE,
                    job_name_parameter=build_daemons_job_name_parameter,
                    parent_job_data=(
                        common.ParentDataResourceToDict(
                            input_key='tasklet_input',
                            dict_key='daemons_resource_id',
                            output_resource_name='QUASAR_DAEMONS',
                        ),
                    ),
                ),
            ),
        )


class UploadTvSymbolsJob(common.BaseTaskletRunJob):
    JOB_TYPE = 'UPLOAD_SYMBOLS'

    def __init__(
            self,
            job_name_parameter="",
            build_tv_app_job_name_parameter="",
    ):
        super(UploadTvSymbolsJob, self).__init__(
            job_type=self.JOB_TYPE,
            tasklet_name='PublishSymbolsToS3',
            job_name_parameter=job_name_parameter,
            job_params=dict(
                apiargs=dict(
                    requirements=dict(
                        disk_space=1 * (1024 ** 3),  # in Bytes
                    ),
                )
            ),
            job_arrows=(
                jg_arrows.JobTrigger(
                    job_type=branch_part.BuildTvAppJob.JOB_TYPE,
                    job_name_parameter=build_tv_app_job_name_parameter,
                    parent_job_data=(
                        common.ParentDataResourceToDict(
                            input_key='tasklet_input',
                            dict_key='symbols_resource_id',
                            output_resource_name='ANDROID_TV_APP_SYMBOLS',
                        ),
                    ),
                ),
            ),
        )


class UploadCentaurSymbolsJob(common.BaseTaskletRunJob):
    JOB_TYPE = 'UPLOAD_SYMBOLS'

    def __init__(
            self,
            job_name_parameter="",
            build_centaur_app_job_name_parameter="",
    ):
        super(UploadCentaurSymbolsJob, self).__init__(
            job_type=self.JOB_TYPE,
            tasklet_name='PublishSymbolsToS3',
            job_name_parameter=job_name_parameter,
            job_params=dict(
                apiargs=dict(
                    requirements=dict(
                        disk_space=1 * (1024 ** 3),  # in Bytes
                    ),
                )
            ),
            job_arrows=(
                jg_arrows.JobTrigger(
                    job_type=branch_part.BuildCentaurLauncherAppJob.JOB_TYPE,
                    job_name_parameter=build_centaur_app_job_name_parameter,
                    parent_job_data=(
                        common.ParentDataResourceToDict(
                            input_key='tasklet_input',
                            dict_key='daemons_resource_id',
                            output_resource_name='QUASAR_CENTAUR_DEMO_APP',
                        ),
                    ),
                ),
            ),
        )


class PublishApplicationUpdateToQuasmodromJob(common.BaseTaskletRunJob):
    JOB_TYPE = 'PUBLISH_APP_UPDATE'

    def __init__(
        self,
        platform,
        group_name,
        is_critical,
        comment,
        use_testing=False,
        bad=False,
        job_name_parameter="",
        create_st_job_name_parameter="",
        publish_apk_job_name_parameter="",
    ):
        requirements = dict(
            disk_space=5 << 30
        )

        job_arrows = [
            jg_arrows.JobTrigger(
                job_type=branch_part.PublishApkToQuasmodromJob.JOB_TYPE,
                job_name_parameter=publish_apk_job_name_parameter,
                parent_job_data=(
                    common.ParentDataOutputToDict(
                        input_key='tasklet_input',
                        dict_key='application_id',
                        output_key='tasklet_output',
                        transform=lambda output, _: output['applicationId'],
                    ),
                    common.JustPassDataToDict(
                        input_key='tasklet_input',
                        dict_key='platform',
                        value=platform,
                    ),
                    common.JustPassDataToDict(
                        input_key='tasklet_input',
                        dict_key='group_name',
                        value=group_name,
                    ),
                    common.JustPassDataToDict(
                        input_key='tasklet_input',
                        dict_key='is_critical',
                        value=is_critical,
                    ),
                    common.JustPassDataToDict(
                        input_key='tasklet_input',
                        dict_key='comment',
                        value=comment,
                    ),
                    common.JustPassDataToDict(
                        input_key='tasklet_input',
                        dict_key='use_testing',
                        value=use_testing,
                    ),
                    common.JustPassDataToDict(
                        input_key='tasklet_input',
                        dict_key='bad',
                        value=bad,
                    ),
                ),
            )
        ]

        if create_st_job_name_parameter:
            job_arrows.append(
                jg_arrows.JobTrigger(
                    job_type=CreateStartrekReleaseTicketJob.JOB_TYPE,
                    job_name_parameter=create_st_job_name_parameter,
                    parent_job_data=(
                        common.ParentDataOutputToDict(
                            input_key='tasklet_input',
                            dict_key='ticket',
                            output_key='startrek_issue',
                        ),
                    ),
                )
            )

        super(PublishApplicationUpdateToQuasmodromJob, self).__init__(
            job_type=self.JOB_TYPE,
            tasklet_name='ApplicationUpdateToQuasmodrom',
            job_name_parameter=job_name_parameter,
            job_params=dict(
                apiargs=dict(
                    requirements=requirements,
                ),
            ),
            job_arrows=job_arrows
        )


class KolhozTestRunJob(jg_element.JobGraphElement):
    JOB_TYPE = "KOLHOZ_SMART_SPEAKER_REGRESSION"

    def __init__(self, platform, job_name_parameter="", publish_to_qd_job_name_parameter="", create_st_job_name_parameter=""):
        super(KolhozTestRunJob, self).__init__(
            job_params=dict(
                job_type=self.JOB_TYPE,
                job_name_parameter=job_name_parameter,
                should_add_to_db=jg_utils.should_add_to_db_branch,
                task_name=self.JOB_TYPE,
                ctx=dict(
                    description="Runs functional tests in kolhoz",
                    model_name=platform,
                    check_rollback=True
                )
            ),
            job_arrows=(
                jg_arrows.JobTrigger(
                    job_type=branch_part.PublishImageToQuasmodromJob.JOB_TYPE,
                    job_name_parameter=publish_to_qd_job_name_parameter,
                    parent_job_data=(
                        jg_job_data.ParentDataOutput(
                            "version",
                            "tasklet_output",
                            transform=lambda output, _: output['firmwares'].values()[0]['version']
                        ),
                        jg_job_data.ParentDataOutput(
                            "version_id",
                            "tasklet_output",
                            transform=lambda output, _: output['firmwares'].values()[0]['id']
                        ),
                    ),
                ),
                jg_arrows.JobTrigger(
                    job_type=CreateStartrekReleaseTicketJob.JOB_TYPE,
                    job_name_parameter=create_st_job_name_parameter,
                    parent_job_data=(
                        jg_job_data.ParentDataOutput(
                            "ticket",
                            "startrek_issue"
                        ),
                    ),
                )
            )
        )
