# -*- coding: utf-8 -*-
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components import configs as rm_configs

from sandbox.projects.release_machine.components.configs.smart_devices import _common as common
from sandbox.projects.release_machine.components.configs.smart_devices import _branch_part as branch_part
from sandbox.projects.release_machine.components.configs.smart_devices import _release_part as release_part
from sandbox.projects.release_machine.components.configs.smart_devices._device_job_graph_base import DeviceJobGraphBase

from sandbox.projects.quasar.platform import Platform, ANDROID_PLATFORMS, PLATFORMS_WITH_FACTORY, PLATFORMS_WITH_PERCENT_GROUP, PLATFORMS_IN_KOLHOZ


class SmartSpeakerJobGraph(DeviceJobGraphBase):
    @staticmethod
    def common_branch_part():
        return (branch_part.BuildAppJob(), )

    def __init__(self, platform, component_name, create_new_gerrit_branch_on_prerelease=False,
                 rm_branch_name=None, is_factory_rm_branch=False):
        super(SmartSpeakerJobGraph, self).__init__(platform, component_name, create_new_gerrit_branch_on_prerelease,
                                                   rm_branch_name, is_factory_rm_branch)
        self.build_factory = platform in PLATFORMS_WITH_FACTORY

    @property
    def release_items(self):
        release_items = super(SmartSpeakerJobGraph, self).release_items

        release_items.append(
            rm_configs.ReleasedResourceInfo(
                name=common.ReleaseItemsNames.daemons(self.platform),
                resource_type='QUASAR_DAEMONS',
                attributes=dict(
                    quasar_platform=self.platform,
                )
            ),
        )

        if self.build_factory:
            release_items.append(
                rm_configs.ReleasedResourceInfo(
                    name=common.ReleaseItemsNames.factory_daemons(self.platform),
                    resource_type='QUASAR_FACTORY_DAEMONS',
                    attributes=dict(
                        quasar_platform=self.platform,
                    )
                ),
            )

        if self.platform in ANDROID_PLATFORMS:
            release_items.append(
                rm_configs.ReleasedResourceInfo(
                    name=common.ReleaseItemsNames.app(self.platform),
                    resource_type='QUASAR_APP',
                ),
            )

        return release_items

    @property
    def release_stages_of_groups(self):
        groups = super(SmartSpeakerJobGraph, self).release_stages_of_groups

        if self.platform in PLATFORMS_WITH_PERCENT_GROUP:
            groups["percent_test"] = rm_const.ReleaseStatus.stable
            groups["percent_baseline"] = rm_const.ReleaseStatus.stable

        return groups

    @property
    def image_build_types(self):
        if self.platform in ANDROID_PLATFORMS:
            return ['user', 'userdebug', 'eng']
        elif self.platform == Platform.YANDEXMINI_2 or self.platform == Platform.YANDEXMICRO or self.platform == Platform.YANDEXMIDI:
            return ['user', 'eng']
        else:
            return [None]

    @property
    def image_build_type_to_release(self):
        return 'user' if len(self.image_build_types) > 1 else None

    def factory_job_parameter(self, revision):
        return '{}_factory'.format(self.revision_job_parameter(revision))

    def is_signed_image_type(self, image_build_type):
        return (self.platform == Platform.YANDEXMINI or
                self.platform == Platform.YANDEXMIDI or
                self.platform in ANDROID_PLATFORMS)

    @property
    def branch_part(self):
        jobs = []

        for revision in self.effective_revisions:
            revision_job_parameter = self.revision_job_parameter(revision)
            factory_job_parameter = None

            jobs.append(
                branch_part.BuildDaemonsJob(
                    self.platform, revision,
                    job_name_parameter=revision_job_parameter
                )
            )

            if self.build_factory:
                factory_job_parameter = self.factory_job_parameter(revision)
                jobs.append(branch_part.BuildDaemonsJob(
                    self.platform, revision, factory=True,
                    job_name_parameter=factory_job_parameter)
                )

            for image_build_type in self.image_build_types:
                sign = True if self.is_signed_image_type(image_build_type) else None

                for apps_build_type in self.get_apps_build_types(image_build_type):
                    build_type_job_parameter = self.build_type_job_parameter(image_build_type, apps_build_type, revision)

                    jobs.append(
                        branch_part.BuildImageJob(
                            self.platform, revision, image_build_type, sign,
                            job_name_parameter=build_type_job_parameter,
                            build_daemons_job_name_parameter=revision_job_parameter,
                            build_factory_job_name_parameter=factory_job_parameter,
                        )
                    )

                    if image_build_type in self.ota_build_types:
                        jobs.extend((
                            branch_part.PublishImageToS3Job(
                                self.platform,
                                resource_filters=dict(signed='True') if sign else None,
                                job_name_parameter=build_type_job_parameter,
                                build_image_job_name_parameter=build_type_job_parameter,
                            ),
                            branch_part.PublishImageToQuasmodromJob(
                                self.platform,
                                job_name_parameter=build_type_job_parameter,
                                publish_to_s3_job_name_parameter=build_type_job_parameter,
                            ),
                        ))

        return jobs

    @property
    def release_part(self):
        jobs = self.create_common_release_part(upload_symbols=True, has_regress=True, run_kolhoz=self.platform in PLATFORMS_IN_KOLHOZ)

        release_stages = [
            rm_const.ReleaseStatus.testing,
            rm_const.ReleaseStatus.prestable,
            rm_const.ReleaseStatus.stable,
        ]

        for revision in self.effective_revisions:
            for stage in release_stages:
                revision_job_parameter = self.revision_job_parameter(revision)
                factory_job_parameter = self.factory_job_parameter(revision) if self.build_factory else None
                release_build_type_parameter = self.build_type_job_parameter(
                    self.image_build_type_to_release,
                    self.apps_build_type_to_release(stage),
                    revision
                )
                jobs.append(release_part.ReleaseResourcesJob(
                    self.platform, stage, revision,
                    build_daemons_job_name_parameter=revision_job_parameter,
                    build_image_job_name_parameter=release_build_type_parameter,
                    build_app_job_name_parameter="" if self.platform in ANDROID_PLATFORMS else None,
                    build_factory_job_name_parameter=factory_job_parameter,
                ))

        return jobs

    def create_upload_symbols_jobs(self):
        jobs = []
        upload_symbols_parameters = []
        for revision in self.effective_revisions:
            revision_job_parameter = self.revision_job_parameter(revision)

            jobs.append(release_part.UploadSymbolsJob(
                job_name_parameter=revision_job_parameter,
                build_daemons_job_name_parameter=revision_job_parameter
            ))
            upload_symbols_parameters.append(revision_job_parameter)

        return jobs, upload_symbols_parameters
