# -*- coding: utf-8 -*-
from collections import defaultdict

from sandbox.common import enum
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components import configs as rm_configs

from sandbox.projects.release_machine.components.configs.smart_devices import _common as common
from sandbox.projects.release_machine.components.configs.smart_devices import _branch_part as branch_part
from sandbox.projects.release_machine.components.configs.smart_devices import _release_part as release_part

import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release

from sandbox.projects.quasar.platform import APPS_BUILD_TYPES, Platform, TV_PLATFORMS, TV_APPS_FLAVOURED_PLATFORMS
from sandbox.projects.quasar.build_types import TvAppsBuildType

import sandbox.projects.quasar.platform.goya as goya


class AppBuildNames(enum.Enum):
    ALICE = None
    DAYDREAM = None
    DEMO = None
    HOME = None
    MUSIC = None
    SERVICES = None
    SETUP_WIZARD = None
    UPDATER = None
    VIDEO_PLAYER = None


class AppNames(enum.Enum):
    ALICE_APP = None
    HOME_APP = None
    MUSIC_APP = None
    UPDATER_APP = None
    BUGREPORT_SENDER_APP = None
    SERVICES_APP = None
    SERVICES_IOSDK_APP = None
    PLATFORM_SERVICES_APP = None
    SETUP_WIZARD_APP = None
    INPUT_SERVICE_APP = None
    LIVE_TV_APP = None
    VIDEO_PLAYER_APP = None
    WEB_PLAYER_APP = None
    DAYDREAM_APP = None
    DEMO_APP = None


class ReleaseTargetNames(enum.Enum):
    BETA = None
    R_TEST_HISI351_CVTE = None
    R_TEST_MT9632_CV = None
    DEFAULT = None


class AppBuild(object):
    def __init__(self, config_path, build_results_root=None, extra_artifacts=None, env=None):
        self.config_path = config_path
        self.build_results_root = build_results_root or '**/build/outputs/apk'
        self.extra_artifacts = extra_artifacts
        self.env = env


class ReleaseTarget(object):
    def __init__(self, stage, group, platforms):
        self.stage = stage
        self.group = group
        self.platforms = platforms


class AppConfig(object):
    BUILD_TYPE_DICT = {
        TvAppsBuildType.RELEASE: TvAppsBuildType.RELEASE,
        TvAppsBuildType.QA: TvAppsBuildType.QA,
        TvAppsBuildType.DEBUG: TvAppsBuildType.DEBUG,
    }

    def __init__(
            self, build_name, package_name,
            artifact_build_types=('release',),
            artifact_attributes=None,
            release_targets=None,
            upload_symbols=False,
            platform_whitelist=None,
    ):
        self.build_name = build_name
        self.release_targets = release_targets
        self.package_name = package_name
        self.artifact_build_types = artifact_build_types
        self.upload_symbols = upload_symbols
        self.platform_whitelist = platform_whitelist

        self.artifacts = {}
        for build_type in self.artifact_build_types:
            attributes = {'package_name': package_name, 'build_type': build_type}
            attributes.update(artifact_attributes or {})
            self.artifacts[build_type] = attributes

        if self.release_targets is None:
            self.release_targets = [
                ReleaseTargetNames.BETA,
                ReleaseTargetNames.R_TEST_HISI351_CVTE,
                ReleaseTargetNames.R_TEST_MT9632_CV,
                ReleaseTargetNames.DEFAULT,
            ]

    def enabled_for_platform(self, platform):
        return self.platform_whitelist is None or platform in self.platform_whitelist

    @staticmethod
    def resolve_build_type(platform, build_target, build_type, branch_number=None):
        return build_type

    def resolve_release_artifact(self, release_target, platform, build_target):
        build_type = self.resolve_build_type(platform, build_target, 'release')
        assert build_type in self.artifact_build_types, '{} not found in {}'.format(
            build_type, self.artifact_build_types)
        return build_type

    def resolve_image_artifact(self, apps_build_type, platform, build_target, branch_number):
        assert self.enabled_for_platform(platform)
        return self.resolve_build_type(platform, build_target, self.BUILD_TYPE_DICT[apps_build_type], branch_number)


class Module2SpecificAppConfig(AppConfig):
    def __init__(
            self,
            build_name,
            package_name,
            artifact_attributes=None,
            release_targets=None,
            upload_symbols=False,
            platform_whitelist=None
    ):
        super(Module2SpecificAppConfig, self).__init__(
            build_name,
            package_name,
            artifact_build_types=['module2_release', 'tv_release'],
            artifact_attributes=artifact_attributes,
            release_targets=release_targets,
            upload_symbols=upload_symbols,
            platform_whitelist=platform_whitelist,
        )

    def resolve_build_type(self, platform, build_target, build_type, branch_number=None):
        prefix = 'module2_' if platform == Platform.YANDEXMODULE_2 else 'tv_'
        return prefix + build_type


class AliceAppConfig(AppConfig):
    def __init__(self):
        package_name = 'com.yandex.tv.alice'

        super(AliceAppConfig, self).__init__(
            AppBuildNames.ALICE,
            package_name,
            artifact_build_types=['tv_release'],
        )

    def resolve_build_type(self, platform, build_target, build_type, branch_number=None):
        if branch_number is None or branch_number > 127:
            prefix = 'tv_'
            return prefix + build_type
        else:
            return build_type


class DemoAppConfig(AppConfig):
    def __init__(self):
        package_name = 'com.yandex.tv.demo'

        super(DemoAppConfig, self).__init__(
            AppBuildNames.DEMO,
            package_name,
            artifact_build_types=['full_release', 'stub_release'],
            release_targets=[],
            platform_whitelist=[
                Platform.YANDEX_TV_HISI351_CVTE,
                Platform.YANDEX_TV_MT6681_CV,
                Platform.YANDEX_TV_MT6681_CVTE,
                Platform.YANDEX_TV_MT9256_CVTE,
                Platform.YANDEX_TV_MT9632_11_CVTE,
                Platform.YANDEX_TV_MT9632_CV,
                Platform.YANDEX_TV_MT9632_CVTE,
                Platform.YANDEX_TV_RT2842_HIKEEN,
                Platform.YANDEX_TV_RT2861_HIKEEN,
                Platform.YANDEX_TV_RT2871_HIKEEN,
                Platform.GOYA,
                Platform.MAGRITTE
            ],
        )

    def resolve_build_type(self, platform, build_target, build_type, branch_number=None):
        largeSystemPartitionTargets = set(goya.Targets16Gb)
        prefix = 'full_' if build_target in largeSystemPartitionTargets else 'stub_'
        return prefix + build_type


class UpdaterAppConfig(AppConfig):
    BUILD_TYPE_DICT = {
        TvAppsBuildType.RELEASE: 'alexandriaTvUiNolog_release',
        TvAppsBuildType.QA: 'betaTvUiLogged_release',
        TvAppsBuildType.DEBUG: 'betaTvUiLogged_release',
    }

    def __init__(self):
        package_name = 'com.yandex.launcher.updaterapp'

        super(UpdaterAppConfig, self).__init__(
            AppBuildNames.UPDATER,
            package_name,
            artifact_attributes={'signed': 'True'},
            artifact_build_types=['alexandriaTvUiNolog_release'],
        )

    def resolve_release_artifact(self, release_target, platform, build_target, branch_number=None):
        return self.artifact_build_types[0]


class WebPlayerAppConfig(AppConfig):
    def __init__(self):
        package_name = 'com.yandex.tv.webplayer'

        super(WebPlayerAppConfig, self).__init__(
            AppBuildNames.VIDEO_PLAYER,
            package_name,
            artifact_build_types=['arm_release'],
        )

    def resolve_build_type(self, platform, build_target, build_type, branch_number=None):
        return 'arm_' + build_type


APP_BUILDS = {
    AppBuildNames.ALICE: AppBuild(
        'smart_devices/android/ci/tv-alice-app/assemble',
        env={'APPMETRICA_UPLOAD_MAPPING': 'true'},
    ),
    AppBuildNames.DAYDREAM: AppBuild('smart_devices/android/tv/daydream/ci/sandbox-assemble'),
    AppBuildNames.DEMO: AppBuild('smart_devices/android/tv/demo/ci/sandbox-assemble'),
    AppBuildNames.HOME: AppBuild(
        'smart_devices/android/tv/home-app/ci/sandbox-assemble',
        env={'APPMETRICA_UPLOAD_MAPPING': 'true', 'dexCountEnabled': 'true'},
    ),
    AppBuildNames.MUSIC: AppBuild(
        'smart_devices/android/ci/tv-music-app/assemble',
        env={'APPMETRICA_UPLOAD_MAPPING': 'true'},
    ),
    AppBuildNames.SERVICES: AppBuild(
        'smart_devices/android/tv/services/ci/sandbox-assemble',
        extra_artifacts={
            'services-iosdk-app-symbols': {
                'path': 'tv/services/services-iosdk-app/build/outputs/external_build/breakpad_symbols',
                'resource_type': 'ANDROID_TV_APP_SYMBOLS',
                'attributes': {
                    'package_name': 'com.yandex.io.sdk',
                },
            },
        },
        env={'APPMETRICA_UPLOAD_MAPPING': 'true', 'publishRelease': 'false'},
    ),
    AppBuildNames.SETUP_WIZARD: AppBuild('smart_devices/android/tv/setup-wizard/ci/sandbox-assemble'),
    AppBuildNames.UPDATER: AppBuild(
        'smart_devices/android/tv/updater-app/ci/release',
        env={'APPMETRICA_UPLOAD_MAPPING': 'true'},
        build_results_root='apk',
    ),
    AppBuildNames.VIDEO_PLAYER: AppBuild('smart_devices/android/tv/video-player/ci/sandbox-assemble'),
}


APPS = {
    AppNames.ALICE_APP: AliceAppConfig(),
    AppNames.HOME_APP: AppConfig(AppBuildNames.HOME, 'com.yandex.tv.home'),
    AppNames.MUSIC_APP: AppConfig(AppBuildNames.MUSIC, 'com.yandex.tv.music'),
    AppNames.UPDATER_APP: UpdaterAppConfig(),
    AppNames.BUGREPORT_SENDER_APP: AppConfig(AppBuildNames.SERVICES, 'com.yandex.tv.bugreportsender'),
    AppNames.SERVICES_APP: AppConfig(AppBuildNames.SERVICES, 'com.yandex.tv.services'),
    AppNames.SERVICES_IOSDK_APP: Module2SpecificAppConfig(
        AppBuildNames.SERVICES,
        'com.yandex.io.sdk',
        upload_symbols=True,
    ),
    AppNames.PLATFORM_SERVICES_APP: AppConfig(
        AppBuildNames.SERVICES,
        'com.yandex.tv.services.platform',
        release_targets=[],
    ),
    AppNames.SETUP_WIZARD_APP: Module2SpecificAppConfig(
        AppBuildNames.SETUP_WIZARD,
        'com.yandex.tv.setupwizard',
        release_targets=[],
    ),
    AppNames.INPUT_SERVICE_APP: AppConfig(AppBuildNames.VIDEO_PLAYER, 'com.yandex.tv.input.efir'),
    AppNames.LIVE_TV_APP: AppConfig(AppBuildNames.VIDEO_PLAYER, 'com.yandex.tv.live'),
    AppNames.VIDEO_PLAYER_APP: AppConfig(AppBuildNames.VIDEO_PLAYER, 'com.yandex.tv.videoplayer'),
    AppNames.WEB_PLAYER_APP: WebPlayerAppConfig(),
    AppNames.DAYDREAM_APP: AppConfig(AppBuildNames.DAYDREAM, 'com.yandex.tv.daydream'),
    AppNames.DEMO_APP: DemoAppConfig(),
}

RELEASE_PLATFORMS = {
    Platform.YANDEX_TV_HISI351_CVTE,
    Platform.YANDEX_TV_MT6681_CV,
    Platform.YANDEX_TV_MT6681_CVTE,
    Platform.YANDEX_TV_MT9632_CV,
    Platform.YANDEX_TV_MT9632_CVTE,
    Platform.YANDEX_TV_RT2861_HIKEEN,
    Platform.YANDEX_TV_RT2871_HIKEEN,
}

RELEASE_TARGETS = {
    ReleaseTargetNames.BETA: ReleaseTarget(
        stage=rm_const.ReleaseStatus.prestable,
        group='beta',
        platforms=[
            Platform.YANDEX_TV_RT2871_HIKEEN,
            Platform.YANDEX_TV_MT9632_CV,
        ],
    ),
    ReleaseTargetNames.R_TEST_HISI351_CVTE: ReleaseTarget(
        stage=rm_const.ReleaseStatus.stable,
        group='release_test',
        platforms=[Platform.YANDEX_TV_HISI351_CVTE],
    ),
    ReleaseTargetNames.R_TEST_MT9632_CV: ReleaseTarget(
        stage=rm_const.ReleaseStatus.stable,
        group='release_test',
        platforms=[Platform.YANDEX_TV_MT9632_CV],
    ),
    ReleaseTargetNames.DEFAULT: ReleaseTarget(
        stage=rm_const.ReleaseStatus.stable,
        group='default',
        platforms=RELEASE_PLATFORMS & TV_PLATFORMS,
    ),
}


class TvAppsJobGraph:
    @staticmethod
    def build_job_name_parameter(build_name):
        return build_name

    @staticmethod
    def publish_job_name_parameter(app_name, artifact_name):
        return app_name + '_' + artifact_name

    @staticmethod
    def upload_symbols_job_name_parameter(app_name):
        return app_name

    @classmethod
    def release_items(cls):
        release_items = []
        for name, app in APPS.items():
            artifacts = list(app.artifacts.values())
            release_items.append(
                rm_configs.ReleasedResourceInfo(
                    name=common.ReleaseItemsNames.tv_app(name),
                    resource_type='ANDROID_TV_APP',
                    attributes=artifacts[0],
                )
            )
        return release_items

    @classmethod
    def branch_part(cls):
        jobs = []

        for build_name, build_config in APP_BUILDS.items():
            jobs.append(branch_part.BuildTvAppJob(
                job_name_parameter=cls.build_job_name_parameter(build_name),
                config_path=build_config.config_path,
                build_results_root=build_config.build_results_root,
                extra_artifacts=build_config.extra_artifacts,
                env=build_config.env,
            ))

        for app_name, app_params in APPS.items():
            for artifact_name, artifact_attrs in app_params.artifacts.items():
                publish_job_name_parameter = cls.publish_job_name_parameter(app_name, artifact_name)
                jobs.append(branch_part.PublishApkToQuasmodromJob(
                    job_name_parameter=publish_job_name_parameter,
                    job_arrows=(
                        jg_arrows.JobTrigger(
                            job_type=branch_part.BuildTvAppJob.JOB_TYPE,
                            job_name_parameter=cls.build_job_name_parameter(app_params.build_name),
                            parent_job_data=(
                                common.ParentDataResourceToDict(
                                    input_key='tasklet_input',
                                    dict_key='resources_ids',
                                    output_resource_name='ANDROID_TV_APP',
                                    many=True,
                                ),
                                common.JustPassDataToDict(
                                    input_key='tasklet_input',
                                    dict_key='resources_filters',
                                    value=artifact_attrs,
                                ),
                            ),
                        ),
                    ),
                ))

        for platform in TV_APPS_FLAVOURED_PLATFORMS:
            for app_build_type in APPS_BUILD_TYPES[platform]:
                jobs.append(branch_part.CreateTvAppResourceIdsConfig(
                    job_name_parameter='_'.join((app_build_type, platform)),
                    job_arrows=get_apps_job_arrows('resource_file_content_json', platform, app_build_type)
                ))

        return jobs

    @classmethod
    def release_part(cls):
        jobs = []
        release_target_arrows = defaultdict(list)
        for app_name, app_config in APPS.items():
            build_job_name_parameter = cls.build_job_name_parameter(app_config.build_name)
            upload_symbols_arrows = []
            if app_config.upload_symbols:
                upload_symbols_job_name_parameter = cls.upload_symbols_job_name_parameter(app_name)
                jobs.append(release_part.UploadTvSymbolsJob(
                    job_name_parameter=upload_symbols_job_name_parameter,
                    build_tv_app_job_name_parameter=build_job_name_parameter,
                ))
                upload_symbols_arrows.append(jg_arrows.JobTrigger(
                    job_type=release_part.UploadTvSymbolsJob.JOB_TYPE,
                    job_name_parameter=upload_symbols_job_name_parameter,
                ))

            for stage in [rm_const.ReleaseStatus.prestable, rm_const.ReleaseStatus.stable]:
                jobs.append(jg_release.JobGraphElementReleaseBranched(
                    release_item=app_name,
                    release_to=stage,
                    job_arrows=(
                        jg_arrows.JobTrigger(
                            job_type=branch_part.BuildTvAppJob.JOB_TYPE,
                            job_name_parameter=build_job_name_parameter,
                            parent_job_data=(
                                jg_job_data.ParentDataDict(
                                    'component_resources',
                                    common.ReleaseItemsNames.tv_app(app_name),
                                    'ANDROID_TV_APP',
                                )
                            )
                        ),
                    ),
                ))

            for release_target_name in app_config.release_targets:
                release_target = RELEASE_TARGETS[release_target_name]
                release_job_name_parameter = app_name + '_' + release_target_name

                update_arrows = []
                for platform in release_target.platforms:
                    if not app_config.enabled_for_platform(release_target.platforms):
                        continue

                    release_artifact_name = app_config.resolve_release_artifact(
                        release_target_name, platform, build_target=None)
                    publish_job_name_parameter = cls.publish_job_name_parameter(app_name, release_artifact_name)
                    update_job_name_parameter = release_job_name_parameter + '_' + platform
                    jobs.append(release_part.PublishApplicationUpdateToQuasmodromJob(
                        platform=platform,
                        group_name=release_target.group,
                        is_critical=False,
                        comment='RM update',
                        job_name_parameter=update_job_name_parameter,
                        publish_apk_job_name_parameter=publish_job_name_parameter,
                    ))
                    update_arrows.append(jg_arrows.JobTrigger(
                        job_type=release_part.PublishApplicationUpdateToQuasmodromJob.JOB_TYPE,
                        job_name_parameter=update_job_name_parameter,
                    ))

                release_action_job_name_parameter = release_job_name_parameter
                release_arrows = upload_symbols_arrows + update_arrows + [
                    jg_job_triggers.JobTriggerRelease(
                        job_name_parameter='{}__{}'.format(app_name, release_target.stage)),
                ]
                jobs.append(jg_release.JobGraphElementActionReleaseBase(
                    job_name_parameter=release_action_job_name_parameter,
                    job_params={
                        'should_add_to_db': jg_utils.should_add_to_db_branch,
                        'cancel_fallbehind_runs_on_fix': False,
                    },
                    job_arrows=release_arrows,
                ))

                release_target_arrows[release_target_name].extend(release_arrows)

        for release_target_name, release_arrows in release_target_arrows.items():
            jobs.append(jg_release.JobGraphElementActionReleaseBase(
                job_name_parameter='TV_APPS_{}'.format(release_target_name),
                job_params={
                    'should_add_to_db': jg_utils.should_add_to_db_branch,
                    'cancel_fallbehind_runs_on_fix': False,
                },
                job_arrows=release_arrows,
            ))

        return jobs


class ParentAppData(common.JobDataToDict):
    def __init__(
            self,
            input_key,
            dict_key,
            platform,
            app_config,
            apps_build_type,
            build_target=None,
    ):
        super(ParentAppData, self).__init__(input_key, dict_key)
        self.platform = platform
        self.app_config = app_config
        self.apps_build_type = apps_build_type
        self.build_target = build_target

    def data(self, params, parent_job_name):
        if 'smart-devices-' in params.db.db_name:
            branch_number = int(params.db.db_name.split('-')[-1])
        else:
            branch_number = None

        app_build_type = self.app_config.resolve_image_artifact(
            self.apps_build_type, self.platform, build_target=self.build_target, branch_number=branch_number)
        output_key = self.dict_key + '_' + app_build_type

        if output_key in params.parent_test_ctx[parent_job_name]:
            return params.parent_test_ctx[parent_job_name][output_key]


def get_apps_job_arrows(input_key, platform, apps_build_type):
    job_arrows = [
        jg_arrows.JobTrigger(
            job_type=branch_part.BuildTvAppJob.JOB_TYPE,
            job_name_parameter=TvAppsJobGraph.build_job_name_parameter(
                AppBuildNames.SERVICES),
            parent_job_data=(
                common.ParentDataResourceToDict(
                    input_key=input_key,
                    dict_key='io_sdk_symbols',
                    output_resource_name='ANDROID_TV_APP_SYMBOLS',
                )
            ),
        )
    ]

    for app_config in APPS.values():
        if not app_config.enabled_for_platform(platform):
            continue

        app_key = app_config.package_name.replace('.', '_')
        build_target = 'UNIOTA_GOYA_16G' if platform == Platform.GOYA else None
        job_arrows.append(jg_arrows.JobTrigger(
            job_type=branch_part.BuildTvAppJob.JOB_TYPE,
            job_name_parameter=TvAppsJobGraph.build_job_name_parameter(app_config.build_name),
            parent_job_data=(
                ParentAppData(
                    input_key=input_key,
                    dict_key=app_key,
                    platform=platform,
                    app_config=app_config,
                    apps_build_type=apps_build_type,
                    build_target=build_target,
                )
            ),
        ))

    return job_arrows


def collect_artifact_dependencies(apps_build_type, platform, build_target):
    app_job_name = 'build_{}_app'
    artifact_dependencies = {}

    for app_name, app_config in APPS.items():
        if not app_config.enabled_for_platform(platform):
            continue

        app_build_type = app_config.resolve_image_artifact(apps_build_type, platform, build_target, branch_number=None)
        input_key = app_config.package_name.replace('.', '_')

        resource_lookup_string_template = (
            '${{tasks.{job_name}.resources[? '
            "type == 'ANDROID_TV_APP' && "
            "attributes.package_name == '{package_name}' && "
            "attributes.build_type == '{app_build_type}' && "
            "attributes.signed == 'True'].id "
            '| single(@)}}'
        )
        artifact_dependencies[input_key] = resource_lookup_string_template.format(
            job_name=app_job_name.format(app_config.build_name.lower()),
            package_name=app_config.package_name,
            app_build_type=app_build_type
        )

        if app_name == AppNames.SERVICES_IOSDK_APP:
            symbols_lookup_string = "${{tasks.{job_name}.resources[? type == 'ANDROID_TV_APP_SYMBOLS'].id | single(@)}}"
            artifact_dependencies['io_sdk_symbols'] = symbols_lookup_string.format(
                job_name=app_job_name.format(app_config.build_name.lower()))

    return artifact_dependencies
