import re
import six

import sandbox.projects.quasar.platform as quasar_platform

from sandbox.projects.release_machine.components.config_core.jg import base as jg_base
from sandbox.projects.release_machine.components.config_core.jg import cube as jg_cube
from sandbox.projects.release_machine.components.config_core.jg.cube import base as cube_base
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import internal as internal_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import smart_devices as smart_devices_cubes
from sandbox.projects.release_machine.components.config_core.jg.graph import base as jg_graph
from sandbox.projects.release_machine.components.config_core.jg.lib.conditions import ci_conditions

from sandbox.projects.release_machine.components.configs.smart_devices import _tv_apps_job_graph as tv_apps_jg


SUPPORTED_PLATFORMS = {
    quasar_platform.Platform.PLUTO
}


class BuildType(object):
    def __init__(
        self,
        image_build_type,
        tv_apps_build_type
    ):
        self.image_build_type = image_build_type
        self.tv_apps_build_type = tv_apps_build_type


class BuildTvImage(object):
    def build_image_cube(
        self,
        platform,
        platform_name,
        is_factory,
        image_build_type,
        tv_apps_build_type,
        build_apps,
        build_target=None
    ):
        return cube_base.Cube(
            name="build_{}_{}_{}_image".format(platform, image_build_type, tv_apps_build_type),
            title="Image {} {} {} build".format(platform_name, image_build_type, tv_apps_build_type),
            task="projects/smart_devices/quasar_build_{}_image".format(platform),
            input=cube_base.CubeInput(
                build_type=image_build_type,
                tv_apps_build_type=tv_apps_build_type,
                # TODO: does it need configuration? smart_devices/platforms/{}/prebuilt.yaml
                artifact_downloader_arcadia_config="smart_devices/tv/prebuilt.yaml",
                checkout_arcadia_from_url=r"arcadia-arc:/#${context.target_revision.hash}",
                factory=is_factory,
                push_sensors_to_solomon=True,
                artifact_downloader_dependencies=tv_apps_jg.collect_artifact_dependencies(tv_apps_build_type, platform, build_target)
            ),
            needs=build_apps
        )


class BuildGoyaImage(BuildTvImage):
    def __init__(
        self,
        build_product_target_name
    ):
        self._build_product_target_name = build_product_target_name

    def build_image_cube(
        self,
        platform,
        platform_name,
        is_factory,
        image_build_type,
        tv_apps_build_type,
        build_apps
    ):
        cube = super(BuildGoyaImage, self).build_image_cube(
            platform,
            platform_name,
            is_factory,
            image_build_type,
            tv_apps_build_type,
            build_apps,
            self._build_product_target_name
        )
        cube.input.update(build_product_target_name=self._build_product_target_name)
        if is_factory:
            cube.input.update(build_ota=False)
        return cube


class TVBuildGraph(object):
    def __init__(
        self,
        platform,
        build_tv_image,
        build_types,
        is_factory
    ):
        assert platform in SUPPORTED_PLATFORMS
        self._platform = platform
        self._build_tv_image = build_tv_image
        self._is_factory = is_factory
        self._build_types = build_types

    def graph(
        self
    ):
        flow_start = dummy_cubes.Dummy(
            name="flow_start",
            condition=ci_conditions.CI_FIRST_TAG_ONLY
        )
        build_apps = self._build_apps(flow_start)
        build_images = []
        publish_images = []
        for build_type in self._build_types:
            build_images.append(self._build_tv_image.build_image_cube(
                platform=self._platform.lower(),
                platform_name=self._platform_name(),
                is_factory=self._is_factory,
                image_build_type=build_type.image_build_type,
                tv_apps_build_type=build_type.tv_apps_build_type,
                build_apps=build_apps
            ))
            if not self._is_factory:
                publish_images.extend(self._publish_ota(
                    image_build_type=build_type.image_build_type,
                    tv_apps_build_type=build_type.tv_apps_build_type,
                    build_image=build_images[-1]
                ))
        graph = jg_graph.Graph(
            [flow_start] + build_apps + build_images + publish_images
        )
        return graph

    def _build_apps(
        self,
        flow_start
    ):
        build_apps = []
        for app_name, app_config in six.iteritems(tv_apps_jg.APP_BUILDS):
            build_apps.append(
                cube_base.Cube(
                    name="build_{}_app".format(app_name.lower()),
                    title=" ".join(app_name.split('_')).title(),
                    task="projects/smart_devices/teamcity_apk_build_wrapper",
                    input=cube_base.CubeInput(
                        branch="trunk",
                        commit=r"${context.target_revision.hash}",
                        ssh_key="QUASAR:robot-quasar-ssh-private-key",
                        config_path=app_config.config_path,
                        resource_attributes={
                            "is_system": True,
                        },
                        build_results_root=app_config.build_results_root,
                        extra_artifacts=app_config.extra_artifacts,
                        env=app_config.env,
                    ),
                    needs=[flow_start],
                )
            )
        return build_apps

    def _publish_ota(
        self,
        image_build_type,
        tv_apps_build_type,
        build_image
    ):
        cube = build_image
        result = []
        pattern = re.compile(r'(?<!^)(?=[A-Z])')
        build_type = "{}_{}".format(image_build_type, tv_apps_build_type)
        for tasklet in ("PublishOtaToS3", "OtaToQuasmodrom", "UpdateToQuasmodrom"):
            result.append(cube_base.Cube(
                name="{}_{}".format(pattern.sub('_', tasklet).lower(), build_type),
                title="{} {} {} {}".format(self._platform_name(), image_build_type, tv_apps_build_type, tasklet),
                task="common/sandbox/tasklet_executor",
                needs=[cube],
                input=cube_base.CubeInput(
                    binary_executor_release_type=None,
                    resource_owner="QUASAR",
                    resource_type="SANDBOX_TASKS_BINARY",
                    yav_token_vault="QUASAR:robot-quasar-yav-token",
                    tasklet_name=tasklet,
                    resource_attrs={
                        "default_tasklet_name": "tasklet"
                    },
                    tasklet_input={
                        "sandbox_token": {
                            "uui": "sec-01d2ffwrdbwyj37zkj4r8zegsn",
                            "key": "robot-quasar-sandbox-token"
                        },
                        "resources_ids": [
                            "${{tasks.build_{}_".format(self._platform.lower()) +
                            build_type +
                            "_image.resources[? type == 'ANDROID_TV_{}_OTA_IMAGE'].id | single(@)}}".format(self._platform)
                        ]
                    }
                )
            ))
            cube = result[-1]
        return result

    def _platform_name(self):
        return self._platform.title()


class ReleaseFlow(jg_base.BaseReleaseMachineJobGraph):
    def platform(self):
        # TODO: ABC here?
        raise NotImplementedError()

    def _factory_name(self):
        return "{}_factory".format(self.platform())

    def _add_gerrit_subgraph_cubes(self, graph):
        gerrit_branch_cubes = []

        new_branch_number = graph.get(internal_cubes.CreateSvnBranch.NAME).output_params.new_branch_number

        gerrit_cube = smart_devices_cubes.CreateGerritBranchCube(
            platform=self.platform(),
            arcadia_release_branch_number=new_branch_number,
            rm_branch_name=self._factory_name(),
            is_factory_rm_branch=True,
            condition=ci_conditions.CI_FIRST_TAG_ONLY,
        )

        gerrit_branch_cubes.append(gerrit_cube)

        graph.add(gerrit_cube)

        merge_cube = smart_devices_cubes.MergeSystemBranchesCube(
            component_name=self._factory_name(),
            new_branch_number=new_branch_number,
            revisions=[cube.output_params.revision for cube in gerrit_branch_cubes],
            needs=gerrit_branch_cubes,
            condition=ci_conditions.CI_FIRST_TAG_ONLY,
        )

        graph.add(merge_cube)

    def graph(self):
        new_tag = internal_cubes.NewTagCube(
            component_name=self._factory_name(),
        )

        create_svn_branch = internal_cubes.CreateSvnBranch(
            component_name=self._factory_name(),
            input=jg_cube.CubeInput(
                custom_branch_number="${context.version_info.major}",
                revision_for_trunk="${context.target_revision.number}",
            ),
            condition=ci_conditions.CI_FIRST_TAG_ONLY,
            needs=[new_tag],
        )

        pre_release_entry = dummy_cubes.Dummy(
            name="pre_release_entry",
            needs=[create_svn_branch],
            condition=ci_conditions.CI_FIRST_TAG_ONLY,
        )

        changelog = self._get_changelog_cube(
            candidate_path=create_svn_branch.output_params.result_path,
            needs=[pre_release_entry],
            use_previous_branch_as_baseline=True,
        )

        graph = jg_graph.Graph([new_tag, create_svn_branch, pre_release_entry, changelog])

        self._add_startrek_jobs_if_required(graph, entry_cube_name=pre_release_entry.name)

        self._add_gerrit_subgraph_cubes(graph)

        pre_release_finish = dummy_cubes.Dummy(
            name="pre_release_finish",
            needs=graph.all_cubes_list,
            condition=ci_conditions.CI_FIRST_TAG_ONLY,
        )

        graph.add(pre_release_finish)

        return graph
