# -*- coding: utf-8 -*-

import json

import sandbox.projects.quasar.platform as quasar_platform
import sandbox.projects.quasar.build_types as quasar_build_types

from sandbox.projects.release_machine.components.config_core.jg import base as jg_base
from sandbox.projects.release_machine.components.config_core.jg import cube as jg_cube
from sandbox.projects.release_machine.components import configs as rm_configs
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow
from sandbox.projects.release_machine.components.config_core.jg.cube import base as cube_base
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import internal as internal_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import smart_devices as smart_devices_cubes
from sandbox.projects.release_machine.components.config_core.jg.graph import base as jg_graph
from sandbox.projects.release_machine.components.config_core.jg.lib.conditions import ci_conditions

ORIGINAL_COMPONENT = "goya_factory"
GOYA43_16G = "YANDEX_GOYA43.16G.V430DJ1_CV9632-N55"
GOYA50_16G = "YANDEX_GOYA50.16G.V500DJ7_CV9632-N55"


class GoyaFactoryCfg(rm_configs.ReferenceCIConfig):
    """
    This is a CI-based Goya factory release.
    """

    name = "goya_factory"
    display_name = "Goya factory"
    responsible = rm_configs.Responsible(
        abc=rm_configs.Abc(service_name="android_tv"),
        login="ohmikey",
    )

    class CI(rm_configs.ReferenceCIConfig.CI):
        a_yaml_dir = "smart_devices/platforms/goya/rm"  # path to a.yaml directory (relative to Arcadia root)
        secret = "sec-01d1ehgnbf5981b4f8dkg78ndj"       # YaV secret
        sb_owner_group = "ANDROID_TV"                   # Sandbox task owner group

        ya_make_abs_paths_glob = [  # These paths are used for commit graph discovery
            'yandex_io/**',
            'smart_devices/tv/**',
            'smart_devices/platforms/goya/**',
            'speechkit/**',
            'voicetech/vqe/yandex_vqe/yvqelib/**',
            'voicetech/spotter/embedded/**',
        ]

        grant_config_update_permissions_to_release_machine_robot = False

    class JG(jg_base.BaseReleaseMachineJobGraph):
        def platform(self):
            return quasar_platform.Platform.GOYA

        def _factory_name(self):
            return "{}_factory".format(self.platform())

        def _add_gerrit_subgraph_cubes(self, graph):
            gerrit_branch_cubes = []

            new_branch_number = graph.get(internal_cubes.CreateSvnBranch.NAME).output_params.new_branch_number

            gerrit_cube = smart_devices_cubes.CreateGerritBranchCube(
                platform=self.platform(),
                arcadia_release_branch_number=new_branch_number,
                rm_branch_name=self._factory_name(),
                is_factory_rm_branch=True,
                ssh_user="robot-edi",
                ssh_key_vault_owner="ANDROID_TV",
                ssh_key_vault_name="robot-edi-ssh",
                resource_owner="ANDROID_TV",
                yav_token_vault="ANDROID_TV:robot-edi-yav-token",
                condition=ci_conditions.CI_FIRST_TAG_ONLY,
                repository_json_path="smart_devices/tv/ota_releases/cv_android_9"
            )

            gerrit_branch_cubes.append(gerrit_cube)

            graph.add(gerrit_cube)

            merge_cube = smart_devices_cubes.MergeSystemBranchesCube(
                component_name=self._factory_name(),
                new_branch_number=new_branch_number,
                revisions=[cube.output_params.revision for cube in gerrit_branch_cubes],
                needs=gerrit_branch_cubes,
                resource_owner="ANDROID_TV",
                yav_token_vault="ANDROID_TV:robot-edi-yav-token",
                skip_svn_branch=True,
                developers_options=True,
                condition=ci_conditions.CI_FIRST_TAG_ONLY,
            )

            graph.add(merge_cube)

        @jg_flow.release_flow()
        def release(self):
            new_tag = internal_cubes.NewTagCube(
                component_name=self._factory_name(),
            )

            create_svn_branch = internal_cubes.CreateSvnBranch(
                component_name=self._factory_name(),
                input=jg_cube.CubeInput(
                    custom_branch_number="${context.version_info.major}",
                    revision_for_trunk="${context.target_revision.number}",
                ),
                condition=ci_conditions.CI_FIRST_TAG_ONLY,
                needs=[new_tag],
            )

            pre_release_entry = dummy_cubes.Dummy(
                name="pre_release_entry",
                needs=[create_svn_branch],
                condition=ci_conditions.CI_FIRST_TAG_ONLY,
            )

            changelog = self._get_changelog_cube(
                candidate_path=create_svn_branch.output_params.result_path,
                needs=[pre_release_entry],
                use_previous_branch_as_baseline=True,
            )

            graph = jg_graph.Graph([new_tag, create_svn_branch, pre_release_entry, changelog])

            self._add_startrek_jobs_if_required(graph, entry_cube_name=pre_release_entry.name)

            self._add_gerrit_subgraph_cubes(graph)

            pre_release_finish = dummy_cubes.Dummy(
                name="pre_release_finish",
                needs=graph.all_cubes_list,
                condition=ci_conditions.CI_FIRST_TAG_ONLY,
            )

            graph.add(pre_release_finish)

            return graph

        @jg_flow.register_flow(
            title="Build Goya factory image",
            description="Build Goya factory image from release machine"
        )
        def build_goya_factory_image(self):
            flow_start = dummy_cubes.Dummy(
                name="flow_start",
                condition=ci_conditions.CI_FIRST_TAG_ONLY
            )
            build_apps = self._build_apps(flow_start)
            build_images = self._build_images(build_apps)
            graph = jg_graph.Graph(
                [flow_start] + build_apps + build_images
            )
            return graph

        def _build_apps(
            self,
            flow_start
        ):
            read_apps = cube_base.Cube(
                name="read_apps",
                title="Read apps.json from arcadia",
                task="projects/quasar_infra/read_json_from_arcadia",
                input=cube_base.CubeInput(
                    checkout_arcadia_from_url="arcadia-arc:/#${context.target_revision.hash}",
                    json_file_relative_path="smart_devices/platforms/goya/firmware/apps.json"
                ),
                needs=[flow_start]
            )
            read_apps_version = cube_base.Cube(
                name="read_apps_version",
                title="Read app_version.json from arcadia",
                task="projects/quasar_infra/read_json_from_arcadia",
                input=cube_base.CubeInput(
                    checkout_arcadia_from_url="arcadia-arc:/#${context.target_revision.hash}",
                    json_file_relative_path="smart_devices/platforms/goya/firmware/apps_version.json"
                ),
                needs=[flow_start]
            )
            return [read_apps, read_apps_version]

        def _build_images(
            self,
            build_apps
        ):
            uniota_supported_targets=[
                {
                    "brand": "YANDEX",
                    "product": "GOYA43.V430DJ1-Q01-C1"
                },
                {
                    "brand": "YANDEX",
                    "product": "GOYA50.V500DJ7-QE1-D5"
                }
            ]
            build_images = []
            for target in (GOYA43_16G, GOYA50_16G):
                name = target.replace(".", "_").replace("-", "_")
                cube = cube_base.Cube(
                    name="build_goya_{}_image".format(name.lower()),
                    title="Image {} build".format(name.lower()),
                    task="projects/smart_devices/quasar_build_goya_image",
                    input=cube_base.CubeInput(
                        build_type=quasar_build_types.ImageBuildtype.USER,
                        tv_apps_build_type=quasar_build_types.TvAppsBuildType.RELEASE,
                        # TODO: does it need configuration? smart_devices/platforms/{}/prebuilt.yaml
                        artifact_downloader_arcadia_config="smart_devices/tv/prebuilt.yaml",
                        checkout_arcadia_from_url=r"arcadia-arc:/#${context.target_revision.hash}",
                        repository_config_dir="smart_devices/tv/ota_releases/cv_android_9",
                        build_factory=True,
                        push_sensors_to_solomon=True,
                        artifact_downloader_dependencies={},
                        artifact_downloader_dependencies_json=r"${tasks.read_apps.output_params.file_content}",
                        apps_major_version=r"${tasks.read_apps_version.output_params.file_content.major}",
                        apps_minor_version=r"${tasks.read_apps_version.output_params.file_content.minor}",
                        build_product_target_name=target,
                        uniota_supported_targets=json.dumps(uniota_supported_targets),
                        build_ota=False
                    ),
                    needs=build_apps
                )
                build_images.append(cube)
            return build_images

    class Notify(rm_configs.ReferenceCIConfig.Notify):

        class Mail(rm_configs.ReferenceCIConfig.Notify.Mail):
            mailing_list = [
                'quasar-dev@yandex-team.ru',
            ]

        class Startrek(rm_configs.ReferenceCIConfig.Notify.Startrek):
            assignee = 'grmammaev'
            queue = 'QUASAR'
            dev_queue = 'QUASAR'
            release_queue = 'ALICERELEASE'
            summary_template = u'Отведение ветки goya_factory/stable-{}'
            release_summary_template = u'Релиз платформы Goya из ветки goya/stable-{}'
            workflow = {}
            followers = []
            add_commiters_as_followers = False
            hide_commits_under_cut = True
            important_changes_limit = 1000

    class ChangelogCfg(rm_configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = ''
        dirs = [
            'arcadia/yandex_io',
            'arcadia/smart_devices',
            'arcadia/speechkit',
            'arcadia/voicetech/vqe/yandex_vqe/yvqelib',
            'arcadia/voicetech/spotter/embedded',
        ]
        review_groups = [
            'quasar-app',
        ]

    # Leaving it empty for now since we're leaving all release logic at smart_devices at the time
    class Releases(rm_configs.ReferenceCIConfig.Releases):
        resources_info = []

    class SvnCfg(rm_configs.ReferenceCIConfig.SvnCfg):

        start_version = 1
        use_zipatch_only = True

        def __init__(self, *args, **kwargs):
            super(self.__class__, self).__init__(ORIGINAL_COMPONENT)
