# -*- coding: utf-8 -*-

import sandbox.projects.quasar.platform as quasar_platform
import sandbox.projects.quasar.build_types as quasar_build_types

from sandbox.projects.release_machine.components import configs as rm_configs
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow

from sandbox.projects.release_machine.components.configs.smart_devices import _tv_cubes as tv_cubes

ORIGINAL_COMPONENT = "pluto_factory"


class PlutoFactoryCfg(rm_configs.ReferenceCIConfig):
    """
    This is a CI-based pluto factory release. It can only create branches and merge commits.
    """

    name = "pluto_factory"
    display_name = "Pluto factory"
    responsible = rm_configs.Responsible(
        abc=rm_configs.Abc(service_name="vsdev"),
        login="ohmikey",
    )

    class CI(rm_configs.ReferenceCIConfig.CI):
        a_yaml_dir = "smart_devices/platforms/pluto/rm/regular"  # path to a.yaml directory (relative to Arcadia root)
        secret = "sec-01d2ffwrdbwyj37zkj4r8zegsn"                # YaV secret
        sb_owner_group = "QUASAR"                                # Sandbox task owner group

        ya_make_abs_paths_glob = [  # These paths are used for commit graph discovery
            'yandex_io/**',
            'smart_devices/tv/**',
            'speechkit/**',
            'voicetech/vqe/yandex_vqe/yvqelib/**',
            'voicetech/spotter/embedded/**',
        ]

    class JG(tv_cubes.ReleaseFlow):
        def platform(self):
            return quasar_platform.Platform.PLUTO

        @jg_flow.release_flow()
        def release(self):
            return self.graph()

        @jg_flow.register_flow(
            title="Build Pluto factory image",
            description="Build Pluto factory image from release machine"
        )
        def build_pluto_factory_image(self):
            build_image = tv_cubes.BuildTvImage()
            build_types = (
                tv_cubes.BuildType(
                    quasar_build_types.ImageBuildtype.USER, quasar_build_types.TvAppsBuildType.RELEASE),
                tv_cubes.BuildType(
                    quasar_build_types.ImageBuildtype.USERDEBUG, quasar_build_types.TvAppsBuildType.QA),
                tv_cubes.BuildType(
                    quasar_build_types.ImageBuildtype.ENGINEERING, quasar_build_types.TvAppsBuildType.DEBUG)
            )
            build_graph = tv_cubes.TVBuildGraph(
                platform=self.platform(),
                build_tv_image=build_image,
                build_types=build_types,
                is_factory=True
            )
            return build_graph.graph()

    class Notify(rm_configs.ReferenceCIConfig.Notify):

        class Mail(rm_configs.ReferenceBranchedConfig.Notify.Mail):
            mailing_list = [
                'quasar-dev@yandex-team.ru',
            ]

        class Startrek(rm_configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = 'grmammaev'
            queue = 'QUASAR'
            dev_queue = 'QUASAR'
            release_queue = 'ALICERELEASE'
            summary_template = u'Отведение ветки pluto_factory/stable-{}'
            release_summary_template = u'Релиз платформы Pluto из ветки pluto/stable-{}'
            workflow = {}
            followers = [
                'orinichev'
            ]
            add_commiters_as_followers = False
            hide_commits_under_cut = True
            important_changes_limit = 1000

    class ChangelogCfg(rm_configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = ''
        dirs = [
            'arcadia/yandex_io',
            'arcadia/smart_devices',
            'arcadia/speechkit',
            'arcadia/voicetech/vqe/yandex_vqe/yvqelib',
            'arcadia/voicetech/spotter/embedded',
        ]
        review_groups = [
            'quasar-app',
        ]

    # Leaving it empty for now since we're leaving all release logic at smart_devices at the time
    class Releases(rm_configs.ReferenceCIConfig.Releases):
        resources_info = []

    class SvnCfg(rm_configs.ReferenceCIConfig.SvnCfg):

        start_version = 11  # Moved to the CI since 10th branch
        use_zipatch_only = True

        def __init__(self, *args, **kwargs):
            super(self.__class__, self).__init__(ORIGINAL_COMPONENT)
