# -*- coding: utf-8 -*-

import sandbox.projects.quasar.platform as quasar_platform

from sandbox.projects.release_machine.components.config_core.jg import base as jg_base
from sandbox.projects.release_machine.components.config_core.jg import cube as jg_cube
from sandbox.projects.release_machine.components import configs as rm_configs
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow
from sandbox.projects.release_machine.components.config_core.jg.cube import base as cube_base
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import internal as internal_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import smart_devices as smart_devices_cubes
from sandbox.projects.release_machine.components.config_core.jg.graph import base as jg_graph
from sandbox.projects.release_machine.components.config_core.jg.lib.conditions import ci_conditions


class SaturnImageBuild(cube_base.Cube):
    def __init__(self, type, daemons_build_cube, factory_daemons_build_cube, **kwargs):
        super(SaturnImageBuild, self).__init__(
            name="build_{}_saturn_image".format(type),
            title="Build saturn {} image".format(type),
            task="projects/smart_devices/quasar_build_saturn_image",
            input=cube_base.CubeInput(
                checkout_arcadia_from_url="arcadia-arc:/#${context.target_revision.hash}",
                build_type=type,
                quasar_daemons="${{tasks.{}.resources[? type == 'QUASAR_DAEMONS'].id | single(@)}}".format(
                    daemons_build_cube.name
                ),
                quasar_daemons_factory="${{tasks.{}.resources[? type == 'QUASAR_FACTORY_DAEMONS'].id | single(@)}}".format(
                    factory_daemons_build_cube.name
                ),
            ),
            needs=[daemons_build_cube, factory_daemons_build_cube],
            **kwargs
        )


class SaturnFactoryCfg(rm_configs.ReferenceCIConfig):
    """
    This is a CI-based Saturn factory release.
    """

    name = "saturn_factory"
    display_name = "Saturn factory"
    responsible = rm_configs.Responsible(
        abc=rm_configs.Abc(service_name="vsdev", role_id=3271),
        login="slavashel",
    )
    robot = "robot-quasar"

    class CI(rm_configs.ReferenceCIConfig.CI):
        a_yaml_dir = "smart_devices/platforms/saturn/rm"  # path to a.yaml directory (relative to Arcadia root)
        secret = "sec-01d2ffwrdbwyj37zkj4r8zegsn"  # robot-quasar secrets
        sb_owner_group = "QUASAR"  # Sandbox task owner group

        ya_make_abs_paths_glob = [  # These paths are used for commit graph discovery
            'yandex_io/**',
            'smart_devices/**',
            'speechkit/**',
        ]

        grant_config_update_permissions_to_release_machine_robot = False

    class JG(jg_base.BaseReleaseMachineJobGraph):

        @property
        def platform(self):
            return quasar_platform.Platform.SATURN

        def _factory_name(self):
            return "{}_factory".format(self.platform)

        def _add_gerrit_subgraph_cubes(self, graph):
            gerrit_branch_cubes = []

            new_branch_number = graph.get(internal_cubes.CreateSvnBranch.NAME).output_params.new_branch_number

            gerrit_cube = smart_devices_cubes.CreateGerritBranchCube(
                platform=self.platform,
                arcadia_release_branch_number=new_branch_number,
                rm_branch_name=self._factory_name(),
                is_factory_rm_branch=True,
                condition=ci_conditions.CI_FIRST_TAG_ONLY,
            )

            gerrit_branch_cubes.append(gerrit_cube)

            graph.add(gerrit_cube)

            merge_cube = smart_devices_cubes.MergeSystemBranchesCube(
                component_name=self._factory_name(),
                new_branch_number=new_branch_number,
                revisions=[cube.output_params.revision for cube in gerrit_branch_cubes],
                needs=gerrit_branch_cubes,
                skip_svn_branch=True,
                developers_options=True,
                condition=ci_conditions.CI_FIRST_TAG_ONLY,
            )

            graph.add(merge_cube)

        @jg_flow.release_flow()
        def release(self):
            new_tag = internal_cubes.NewTagCube(
                component_name=self._factory_name(),
            )

            create_svn_branch = internal_cubes.CreateSvnBranch(
                component_name=self._factory_name(),
                input=jg_cube.CubeInput(
                    custom_branch_number="${context.version_info.major}",
                    revision_for_trunk="${context.target_revision.number}",
                ),
                condition=ci_conditions.CI_FIRST_TAG_ONLY,
                needs=[new_tag],
            )

            pre_release_entry = dummy_cubes.Dummy(
                name="pre_release_entry",
                needs=[create_svn_branch],
                condition=ci_conditions.CI_FIRST_TAG_ONLY,
            )

            changelog = self._get_changelog_cube(
                candidate_path=new_tag.output.svn_data.svn_paths.tag,
                needs=[new_tag],
                use_previous_branch_as_baseline=True,
            )

            graph = jg_graph.Graph([new_tag, create_svn_branch, pre_release_entry, changelog])

            self._add_startrek_jobs_if_required(graph, entry_cube_name=pre_release_entry.name)

            self._add_gerrit_subgraph_cubes(graph)

            pre_release_finish = dummy_cubes.Dummy(
                name="pre_release_finish",
                needs=graph.all_cubes_list,
                condition=ci_conditions.CI_FIRST_TAG_ONLY,
            )

            graph.add(pre_release_finish)

            # TODO(slavahshel) move this graph generation to lib as common pattern
            # BUILD stage
            build_stage_start = dummy_cubes.Dummy(name="build_stage_start", needs=[pre_release_finish])
            graph.add(build_stage_start)

            builds_cubes = self._build_image(build_stage_start)
            for cube in builds_cubes:
                graph.add(cube)

            build_stage_finish = dummy_cubes.Dummy(name="build_stage_end", needs=builds_cubes)
            graph.add(build_stage_finish)

            # TESTING stage
            testing_stage_start = dummy_cubes.Dummy(name="testing_stage_start", needs=[build_stage_finish], manual=True)
            graph.add(testing_stage_start)
            publish_image_cubes = self._publish_image(testing_stage_start, builds_cubes[-2], builds_cubes[-1])
            prod_image = None
            for cube in publish_image_cubes:
                if isinstance(cube, smart_devices_cubes.PublishImageToQdCube) and cube.is_prod_build:
                    prod_image = cube
                graph.add(cube)
            testing_stage_finish = dummy_cubes.Dummy(name="testing_stage_finish", needs=publish_image_cubes)
            graph.add(testing_stage_finish)

            # BETA stage
            beta_stage_start = dummy_cubes.Dummy(name="beta_stage_start", needs=[testing_stage_finish], manual=True)
            graph.add(beta_stage_start)

            update_beta_cube = smart_devices_cubes.UpdateToQdCube(prod_image.name, "beta", needs=[beta_stage_start])
            graph.add(update_beta_cube)

            beta_stage_finish = dummy_cubes.Dummy(name="beta_stage_finish", needs=[update_beta_cube])
            graph.add(beta_stage_finish)

            # PRODUCTION stage
            prod_stage_start = dummy_cubes.Dummy(name="prod_stage_start", needs=[testing_stage_finish], manual=True)
            graph.add(prod_stage_start)

            update_prod_cube = smart_devices_cubes.UpdateToQdCube(prod_image.name, "default", needs=[prod_stage_start])
            graph.add(update_prod_cube)

            prod_stage_finish = dummy_cubes.Dummy(name="prod_stage_finish", needs=[update_prod_cube])
            graph.add(prod_stage_finish)

            return graph

        def _build_image(self, stage_start):
            daemons_build = smart_devices_cubes.DaemonsBuildCube(platform=self.platform, needs=[stage_start])
            factory_daemons_build = smart_devices_cubes.FactoryDaemonsBuildCube(
                platform=self.platform, needs=[stage_start]
            )
            eng_image_build = SaturnImageBuild(
                type="eng", daemons_build_cube=daemons_build, factory_daemons_build_cube=factory_daemons_build
            )
            user_image_build = SaturnImageBuild(
                type="user", daemons_build_cube=daemons_build, factory_daemons_build_cube=factory_daemons_build
            )
            return [daemons_build, factory_daemons_build, user_image_build, eng_image_build]

        def _publish_image(self, stage_start, user_image_cube, eng_image_cube):
            publish_user_s3 = smart_devices_cubes.PublishImageToS3Cube(
                user_image_cube.name, "QUASAR_SATURN_OTAIMAGE", "user", needs=[stage_start, user_image_cube]
            )
            publish_user_qd = smart_devices_cubes.PublishImageToQdCube(publish_user_s3.name, "user", needs=[publish_user_s3])
            publish_eng_s3 = smart_devices_cubes.PublishImageToS3Cube(
                eng_image_cube.name, "QUASAR_SATURN_OTAIMAGE", "eng", needs=[stage_start, eng_image_cube]
            )
            publish_eng_qd = smart_devices_cubes.PublishImageToQdCube(publish_eng_s3.name, "eng", needs=[publish_eng_s3])
            return [publish_user_s3, publish_eng_s3, publish_user_qd, publish_eng_qd]

        @jg_flow.register_flow(
            title="Build Saturn factory image", description="Build Saturn factory image from release machine"
        )
        def build_saturn_factory_image(self):
            flow_start = dummy_cubes.Dummy(name="flow_start", condition=ci_conditions.CI_FIRST_TAG_ONLY)
            build_image_cubes = self._build_image(flow_start)
            publish_image_cubes = self._publish_image(flow_start, build_image_cubes[-2], build_image_cubes[-1])
            graph = jg_graph.Graph([flow_start] + build_image_cubes + publish_image_cubes)
            return graph

    class Notify(rm_configs.ReferenceCIConfig.Notify):
        class Mail(rm_configs.ReferenceCIConfig.Notify.Mail):
            mailing_list = [
                'quasar-dev@yandex-team.ru',
            ]

        class Startrek(rm_configs.ReferenceCIConfig.Notify.Startrek):
            assignee = 'slavashel'
            queue = 'QUASAR'
            dev_queue = 'QUASAR'
            release_queue = 'ALICERELEASE'
            summary_template = u'Отведение ветки saturn_factory/stable-{}'
            release_summary_template = u'Релиз платформы Saturn из ветки saturn/stable-{}'
            workflow = {}
            followers = ["slavashel", "kharcheva", "superbereza"]
            add_commiters_as_followers = False
            hide_commits_under_cut = True
            important_changes_limit = 1000

    class ChangelogCfg(rm_configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = ''
        dirs = [
            'arcadia/yandex_io',
            'arcadia/smart_devices',
            'arcadia/speechkit',
        ]
        review_groups = [
            'yandex_io',
        ]

    # Leaving it empty for now since we're leaving all release logic at smart_devices at the time
    class Releases(rm_configs.ReferenceCIConfig.Releases):
        main_release_flow_independent_stages = True

        # FIXME(slavashel): fill with OTA and DAEMONS resources
        resources_info = []

    class SvnCfg(rm_configs.ReferenceCIConfig.SvnCfg):
        start_version = 1
        use_zipatch_only = True
