# -*- coding: utf-8 -*-
import itertools

from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components import configs as rm_configs
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.configs.smart_devices._pre_release_part as sd_prerelease
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
from sandbox.projects.release_machine.components.configs.smart_devices._smart_speaker_job_graph import SmartSpeakerJobGraph
from sandbox.projects.release_machine.components.configs.smart_devices._tv_job_graph import TvJobGraph
from sandbox.projects.release_machine.components.configs.smart_devices._tv_apps_job_graph import TvAppsJobGraph
from sandbox.projects.release_machine.components.configs.smart_devices._centaur_job_graph import CentaurJobGraph

from sandbox.projects.quasar.platform import Platform

JOB_GRAPHS = {
    Platform.JBL_LINK_MUSIC: (SmartSpeakerJobGraph, dict()),
    Platform.JBL_LINK_PORTABLE: (SmartSpeakerJobGraph, dict()),
    Platform.YANDEXMICRO: (SmartSpeakerJobGraph, dict(create_new_gerrit_branch_on_prerelease=True)),
    Platform.YANDEXMIDI: (SmartSpeakerJobGraph, dict(create_new_gerrit_branch_on_prerelease=True)),
    Platform.YANDEXMINI: (SmartSpeakerJobGraph, dict()),
    Platform.YANDEXMINI_2: (SmartSpeakerJobGraph, dict(create_new_gerrit_branch_on_prerelease=True)),
    Platform.YANDEXSTATION: (SmartSpeakerJobGraph, dict()),
    Platform.YANDEXSTATION_2: (SmartSpeakerJobGraph, dict(create_new_gerrit_branch_on_prerelease=True)),
    Platform.YANDEXMODULE_2: (TvJobGraph, dict(create_new_gerrit_branch_on_prerelease=True, use_rm_app_builds=True, has_regress=True)),
    Platform.CENTAUR: (CentaurJobGraph, dict(create_new_gerrit_branch_on_prerelease=False)),
}

SMART_DEVICES_JOB_GRAPHS = [
    job_graph_class(platform=platform, component_name=rm_const.RMNames.SMART_DEVICES, **kwargs)
    for platform, (job_graph_class, kwargs) in JOB_GRAPHS.items()
]

LTS_BRANCHES = [
    41,  # jbl november release
    49,  # ym2 pvt2/mp fw, FIXME: 47 should be removed after PVT2 is released from above line
    51,  # jbl
    52,  # ym2 mp, ymicro pvt
    68,  # yandexmicro first OTA
    89,  # TV stable 1.5 ota for manufacture
    94,  # ???
    116,  # ???
]


class SmartDevicesConfig(rm_configs.ReferenceBranchedConfig):
    name = rm_const.RMNames.SMART_DEVICES
    responsible = 'vreutov'

    class Notify(rm_configs.ReferenceBranchedConfig.Notify):
        """ Notifications configuration """

        class Mail(rm_configs.ReferenceBranchedConfig.Notify.Mail):
            """Mail notifications configuration"""
            mailing_list = [
                'quasar-dev@yandex-team.ru',
            ]

        class Startrek(rm_configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = 'grmammaev'
            queue = 'QUASAR'
            dev_queue = 'QUASAR'
            release_queue = 'ALICERELEASE'
            summary_template = u'Отведение ветки smart_devices/stable-{}'
            release_summary_template = u'Релиз платформы {} из ветки smart_devices/stable-{}'
            workflow = {}
            followers = [
                'orinichev'
            ]
            add_commiters_as_followers = False
            hide_commits_under_cut = True
            important_changes_limit = 1000

    class Releases(rm_configs.ReferenceBranchedConfig.Releases):
        allow_old_releases = True

        resources_info = list(itertools.chain.from_iterable(
            device_job_graph.release_items
            for device_job_graph in SMART_DEVICES_JOB_GRAPHS
        )) + TvAppsJobGraph.release_items()

    class MergesCfg(rm_configs.ReferenceBranchedConfig.MergesCfg):
        @property
        def lts_branches(self):
            return LTS_BRANCHES

    class ChangelogCfg(rm_configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = ''
        dirs = [
            'arcadia/yandex_io',
            'arcadia/smart_devices',
            'arcadia/speechkit',
            'arcadia/voicetech/vqe/yandex_vqe/yvqelib',
            'arcadia/voicetech/spotter/embedded',
        ]
        review_groups = [
            'yandex_io',
            'quasar-app',
        ]

    class SvnCfg(rm_configs.ReferenceBranchedConfig.SvnCfg):
        use_arc = True
        use_zipatch_only = True

    class Testenv(rm_configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_task_owner = 'QUASAR'
        trunk_db = 'smart-devices-trunk'
        branch_db_template = 'smart-devices-{testenv_db_num}'

        class JobGraph(rm_configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _prerelease(self):
                pre_release_part = [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(
                        ctx={"use_previous_branch_as_baseline": True}
                    ),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrek(
                        use_arc=self._use_arc,
                        job_arrows=(
                            jg_arrows.JobTrigger(sd_prerelease.MergeSystemBranches.JOB_TYPE)
                        ),
                    )
                ]

                create_branch_arrows = []
                for config in SMART_DEVICES_JOB_GRAPHS:
                    if config.create_new_gerrit_branch_on_prerelease:
                        pre_release_part.append(sd_prerelease.CreateGerritBranch(config.platform, config.rm_branch_name, config.is_factory_rm_branch))
                        create_branch_arrows.append(jg_arrows.JobTrigger(sd_prerelease.CreateGerritBranch.JOB_TYPE, job_name_parameter=config.platform))

                pre_release_part.append(sd_prerelease.MergeSystemBranches(component_name=self.name, job_arrows=create_branch_arrows))
                return pre_release_part

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part

                branch_part.extend(SmartSpeakerJobGraph.common_branch_part())
                branch_part.extend(TvAppsJobGraph.branch_part())

                for config in SMART_DEVICES_JOB_GRAPHS:
                    branch_part.extend(config.branch_part)

                return branch_part

            @property
            def _release(self):
                release = [jg_release.JobGraphElementNewTag()]

                release.extend(TvAppsJobGraph.release_part())
                for config in SMART_DEVICES_JOB_GRAPHS:
                    release.extend(config.release_part)

                return release

        class JobPatch(rm_configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def ignore_prefix(self):
                return super(self.__class__, self).ignore_prefix + [
                    'APK_',
                    'DAEMONS_',
                    'FACTORY_',
                    'IMAGE_',
                    'NIGHTLY_',
                    'SIGNED_',
                    'SIZE_',
                    'TO_',
                    'TV_',
                    '_{}__'.format(sd_prerelease.CreateGerritBranch.JOB_TYPE),
                    '_{}__'.format(sd_prerelease.MergeSystemBranches.JOB_TYPE),
                ]

            @property
            def change_frequency(self):
                return super(self.__class__, self).change_frequency

        @staticmethod
        def db_stop_range(branch_number):
            stop_range = range(max([branch_number - 4, 1]), branch_number - 2)
            return [i for i in stop_range if i not in LTS_BRANCHES]

        @staticmethod
        def db_drop_range(branch_number):
            # type: (int) -> Iterable[int]
            if branch_number <= 10:
                drop_from = 1
            else:
                drop_from = branch_number - 10
            drop_range = range(drop_from, branch_number - 8)
            # do not drop LTS releases
            return [i for i in drop_range if i not in LTS_BRANCHES]
