import os
import json

from sandbox.common.format import ident
from sandbox.projects.release_machine.components import configs as rm_configs
from sandbox.projects.release_machine.components.configs.smart_devices import _branch_part as branch_part

import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.stages.job_graph_element as jg_element
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.quasar.platform as quasar_platform

PLATFORM_FIRMWARE_PATH = 'smart_devices/platforms/{platform}/{name_prefix}firmware'
BUILDROOT = 'buildroot'
BUILDROOT_BRANCH = 'master'

PLATFORMS = {
    quasar_platform.Platform.YANDEXSTATION_2: {
        'extensions': ('linux', 'bootloader'),
        'build_contexts': {
            '': {'sign': True, 'secureboot': True}
        }
    },
    quasar_platform.Platform.YANDEXMINI_2: {
        'build_contexts': {
            '': {'sign': True, 'secureboot': True},
            'SL_LINUX': {'sign': True, 'secureboot': True, 'build_type': 'sl_linux'}
        }
    },
}


def load_config(platform, firmware_name=''):

    # Sandbox common build does not support import outside from sandbox (with very few exceptions)
    try:
        from library.python import resource
    except ImportError:
        return {
            "tracked_branches": {},
        }

    assert platform in quasar_platform.Platform
    name_prefix = firmware_name + '_' if firmware_name else ''
    repository_config_data = resource.find(
        os.path.join(
            PLATFORM_FIRMWARE_PATH.format(platform=platform, name_prefix=name_prefix),
            'repository.json'
        )
    )
    return json.loads(repository_config_data)


def get_manifest_path(platform, branch, firmware_name=''):
    name_prefix = firmware_name + '_' if firmware_name else ''
    return os.path.join(
        PLATFORM_FIRMWARE_PATH.format(platform=platform, name_prefix=name_prefix),
        branch,
        'manifest.xml'
    )


def get_buildroot_manifest_path(platform):
    return get_manifest_path(platform, BUILDROOT_BRANCH, firmware_name=BUILDROOT)


def firmware_build_item(platform, branch='', extension='', context_name=''):
    if branch.startswith(platform):
        postfix = branch[len(platform):].strip('/-_')
    elif platform == quasar_platform.Platform.YANDEXSTATION_2 and branch == 'yandexstation_plus_develop':
        return branch
    else:
        postfix = branch
    parts = [platform]
    if extension:
        parts.append(extension)
    if context_name:
        parts.append(context_name)
    if branch:
        parts.append(postfix)
    return '_'.join(parts)


class SmartDevicesFirmwareBuildTrunk(jg_element.JobGraphElement):
    JOB_TYPE = 'BUILD_IMAGE'

    def __init__(
            self,
            task_name,
            job_params=None,
            job_arrows=(),
            build_item='',
            ctx=None,
            out=None,
            observed_paths=None,
            custom_manifest=None,
    ):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                'job_type': self.JOB_TYPE,
                'task_name': task_name,
                'ctx': ctx,
                'out': out,
                'observed_paths': observed_paths,
                'job_name_parameter': build_item,
                'should_add_to_db': jg_utils.should_add_to_db_trunk,
                'frequency': (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                'apiargs': {'hidden': False, 'requirements': {'platform': 'linux'}},
            },
        )
        super(SmartDevicesFirmwareBuildTrunk, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        if custom_manifest:
            custom_manifest = os.path.join(
                '{svn_ssh_url}',
                custom_manifest + '@{revision}'
            )
            self.job_arrows += (
                jg_arrows.ValueAndParamsData(
                    input_key='custom_manifest',
                    value=custom_manifest,
                    transform=lambda y, x, rm_config: y.format(
                        svn_ssh_url=x.svn_ssh_url,
                        revision=x.revision,
                    ),
                ),
            )


class SmartDevicesFirmwareCfg(rm_configs.ReferenceTrunkConfig):
    component_group = 'component_info_general'
    name = 'smart_devices_firmware'
    responsible = 'grmammaev'

    class Testenv(rm_configs.ReferenceTrunkConfig.Testenv):
        '''Testenv configuration'''
        trunk_db = 'smart-devices-firmware-trunk'
        trunk_task_owner = 'QUASAR'

        class JobGraph(rm_configs.ReferenceTrunkConfig.Testenv.JobGraph):
            @property
            def _trunk_part(self):
                trunk_part = super(self.__class__, self)._trunk_part
                sign = True
                for platform, platform_dict in PLATFORMS.items():
                    repository_config = load_config(platform)
                    for branch, branch_config in repository_config['tracked_branches'].items():
                        manifest_path = get_manifest_path(platform, branch)

                        build_contexts = platform_dict.get('build_contexts')
                        for context_name, context in build_contexts.items():
                            build_item = firmware_build_item(platform, branch, context_name=context_name)
                            trunk_part.append(SmartDevicesFirmwareBuildTrunk(
                                task_name='QUASAR_BUILD_{}_IMAGE'.format(ident(platform)),
                                build_item=build_item,
                                observed_paths=[manifest_path],
                                job_params={
                                    'apiargs': {
                                        'requirements': {
                                            'disk_space': 200 << 30,  # 200 Gb
                                        },
                                    },
                                },
                                ctx=context,
                                custom_manifest=manifest_path,
                            ))

                            if branch_config.get('development_branch'):
                                trunk_part.append(branch_part.PublishImageToS3Job(
                                    platform,
                                    resource_filters=dict(signed='True') if sign else None,
                                    job_name_parameter=build_item,
                                    build_image_job_name_parameter=build_item,
                                ))
                                trunk_part.append(branch_part.PublishImageToQuasmodromJob(
                                    platform,
                                    job_name_parameter=build_item,
                                    publish_to_s3_job_name_parameter=build_item,
                                    trunk=True,
                                ))

                    buildroot_manifest_path = get_buildroot_manifest_path(platform)
                    platform_extensions = platform_dict.get('extensions')
                    if platform_extensions:
                        for extension in platform_extensions:
                            trunk_part.append(SmartDevicesFirmwareBuildTrunk(
                                task_name='QUASAR_BUILD_{}_{}_IMAGE'.format(ident(platform), extension.upper()),
                                build_item=firmware_build_item(platform, extension=extension),
                                observed_paths=[buildroot_manifest_path],
                                job_params={
                                    'apiargs': {
                                        'requirements': {
                                            'disk_space': 200 << 30,  # 30 Gb
                                            'dns': 'dns64',
                                        },
                                    },
                                },
                                ctx={
                                    'secureboot': True,
                                },
                                custom_manifest=buildroot_manifest_path,
                            ))

                return trunk_part

    class Releases(rm_configs.ReferenceTrunkConfig.Releases):
        resources_info = []
