# -*- coding: utf-8 -*-

from sandbox.projects.common.constants import constants as sc
from sandbox.projects.release_machine.components import configs as cfg
from sandbox.projects.release_machine.core import const as rm_const, releasable_items as ri
import sandbox.projects.release_machine.components.job_graph.job_data as jg_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_triggers
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils

from sandbox.projects.release_machine.components.configs import smarttv

RESOURCE_NAME = "SMARTTV_UPDATER_PACKAGE"
PACKAGES = "yaphone/newpdater/docker/pkg.json"
OBSERVED_PATHS = ['yaphone/newpdater/']
TESTS_TARGETS = "yaphone/newpdater"


class SmarttvUpdaterCfg(cfg.ReferenceBranchedConfig):
    name = "smarttv_updater"
    responsible = cfg.Responsible(abc=smarttv.responsible_abc, login="alex-garmash")

    class Testenv(cfg.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "SMARTTV"

        class JobGraph(cfg.ReferenceBranchedConfig.Testenv.JobGraph):
            YA_PACKAGE_API_ARGS = {
                "kill_timeout": 15 * 60,  # 15 minutes
            }

            YA_PACKAGE_CTX = {
                sc.USE_AAPI_FUSE: True,
                sc.ALLOW_AAPI_FALLBACK: True,
                sc.USE_ARC_INSTEAD_OF_AAPI: True,
                sc.BUILD_SYSTEM_KEY: sc.SEMI_DISTBUILD_BUILD_SYSTEM,
                sc.BUILD_TYPE_KEY: sc.RELEASE_BUILD_TYPE,
                sc.YA_YT_STORE: False,
                "package_type": "tarball",
                "compress_package_archive": True,
                "do_not_remove_resources": True,

                "resource_type": RESOURCE_NAME,
                "packages": PACKAGES,
                "release_to_ya_deploy": False,
            }

            TESTS_CTX = {
                sc.ALLOW_AAPI_FALLBACK: True,
                sc.USE_ARC_INSTEAD_OF_AAPI: True,
                sc.FAILED_TESTS_CAUSE_ERROR: True,
                "targets": TESTS_TARGETS,
            }

            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrek(use_arc=self._use_arc)
                ]

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part

                branch_part += [
                    jg_build.JobGraphElementBuildBranched(
                        task_name="YA_PACKAGE",
                        ctx=self.YA_PACKAGE_CTX,
                        job_params={
                            "apiargs": self.YA_PACKAGE_API_ARGS,
                        },
                        out={
                            RESOURCE_NAME: 15,
                        },
                    ),
                    jg_test.JobGraphElementMetaTestBranched(
                        ctx=self.TESTS_CTX,
                        job_params={
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                    ),
                ]
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release

                stages = {
                    rm_const.ReleaseStatus.prestable: (jg_utils.TestFrequency.LAZY, None),
                    rm_const.ReleaseStatus.stable: (jg_utils.TestFrequency.LAZY, None),
                }

                for release_stage, frequency in stages.items():
                    release_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=release_stage,
                            task_name="RELEASE_RM_COMPONENT_2",
                            job_params={
                                "ctx": {
                                    "deploy_system": rm_const.DeploySystem.ya_deploy.name,
                                },
                            },
                            job_arrows=[
                                jg_triggers.JobTriggerBuild(
                                    parent_job_data=[
                                        jg_data.ParentDataDict(
                                            input_key="component_resources",
                                            dict_key=RESOURCE_NAME,
                                            resource_name=RESOURCE_NAME,
                                        ),
                                    ],
                                ),
                                jg_triggers.JobTriggerNewTag(
                                    parent_job_data=[
                                        jg_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                                        jg_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
                                    ],
                                ),
                            ]
                        )
                    )
                    release_part.append(
                        jg_release.JobGraphElementActionReleaseBranched(
                            job_params={
                                "frequency": frequency,
                            },
                            release_to=release_stage,
                        )
                    )

                return release_part

        class JobPatch(cfg.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def ignore_match(self):
                common_ignored = super(self.__class__, self).ignore_match
                return common_ignored + [
                    "_BUILD__SMARTTV_UPDATER__TRUNK",
                    "_META_TEST__SMARTTV_UPDATER__TRUNK",
                    "_RELEASE__SMARTTV_UPDATER__TRUNK__TESTING",
                ]

    class Notify(cfg.ReferenceBranchedConfig.Notify):
        class Startrek(cfg.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "robot-smart-tv"
            queue = "TVBACKRELEASES"
            dev_queue = "TVBACKRELEASES"
            summary_template = "Updater release {}"
            components = "updater"
            followers = cfg.PeopleGroups(
                abc_services=[smarttv.responsible_abc],
                staff_groups=None,
                logins=None
            )
            notify_on_deploy_to_feature_st_tickets = True
            nanny_reports = False

    class ChangelogCfg(cfg.ReferenceBranchedConfig.ChangelogCfg):
        dirs = [
            "arcadia/yaphone/newpdater",
        ]

        wiki_page = ""

    class SvnCfg(cfg.ReferenceBranchedConfig.SvnCfg):
        use_arc = True
        arc_tag_folder = "tags/releases"
        arc_branches_folder = "releases"
        branch_name = "smarttv/updater"
        tag_name = "smarttv/updater"

    class Releases(cfg.ReferenceBranchedConfig.Releases):
        allow_old_releases = True
        deploy_system = rm_const.DeploySystem.ya_deploy
        wait_for_deploy_time_sec = 15 * 60  # 15 minutes

        @property
        def releasable_items(self):
            deploy_infos = [
                ri.YaDeployInfo(ri.DeployService("updater-testing"), stage=rm_const.ReleaseStatus.testing),
                ri.YaDeployInfo(ri.DeployService("updater-infra"), stage=rm_const.ReleaseStatus.prestable),
                ri.YaDeployInfo(ri.DeployService("updater-production"), stage=rm_const.ReleaseStatus.stable),
            ]

            return [
                ri.ReleasableItem(
                    name=RESOURCE_NAME,
                    data=ri.SandboxResourceData(RESOURCE_NAME),
                    deploy_infos=deploy_infos,
                ),
            ]

        resources_info = [
            cfg.ReleasedResourceInfo(
                name=RESOURCE_NAME,
                resource_type=RESOURCE_NAME,
                deploy=[
                    cfg.DeployServicesInfo(
                        services=["updater-testing"],
                        level=rm_const.ReleaseStatus.testing,
                    ),
                    cfg.DeployServicesInfo(
                        services=["updater-infra"],
                        level=rm_const.ReleaseStatus.prestable,
                    ),
                    cfg.DeployServicesInfo(
                        services=["updater-production"],
                        level=rm_const.ReleaseStatus.stable,
                    ),
                ],
            ),
        ]
