# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.common import constants as sandbox_constants
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


VAULT_USERS = {
    "DEV": "robot-modadvert-test",
    "DEV2": "robot-modadvert-test",
    "STAGE": "robot-modadvert-stag",
    "TESTING": "robot-modadvert-test",
    "PROD": "robot-modadvert-yt"
}


class SupermoderationCfg(configs.ReferenceBranchedConfig):
    name = "supermoderation"
    responsible = "lkozhinov"

    class MergesCfg(configs.ReferenceBranchedConfig.MergesCfg):
        number_last_branches_to_merge = 1

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_db = "modadvert-supermoderation-trunk"
        trunk_task_owner = "MODADVERT_RELEASE"
        branch_db_template = "modadvert-supermoderation-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            DOCKER_IMAGES = [
                "MODADVERT_SM_API_BINARY",
                "MODADVERT_SM_ADMIN_BINARY",
                "MODADVERT_DIRECT_PROXY_BINARY",
            ]
            SUPERMODERATION_RESOURCES = {
                "MODADVERT_SM_CONFIG_DEV": ["modadvert/resources/supermoderation/conf/dev/package.json"],
                "MODADVERT_SM_CONFIG_DEV_2": ["modadvert/resources/supermoderation/conf/dev2/package.json"],
                "MODADVERT_SM_CONFIG_STAGE": ["modadvert/resources/supermoderation/conf/stage/package.json"],
                "MODADVERT_SM_CONFIG_TESTING": ["modadvert/resources/supermoderation/conf/testing/package.json"],
                "MODADVERT_SM_CONFIG_PROD": ["modadvert/resources/supermoderation/conf/prod/package.json"],
                "MODADVERT_SM_SERVICE_CONFIG_DEV": ["modadvert/resources/supermoderation/service_confs/dev/package.json"],
                "MODADVERT_SM_SERVICE_CONFIG_STAGE": ["modadvert/resources/supermoderation/service_confs/stage/package.json"],
                "MODADVERT_SM_SERVICE_CONFIG_TESTING": ["modadvert/resources/supermoderation/service_confs/testing/package.json"],
                "MODADVERT_SM_SERVICE_CONFIG_PROD": ["modadvert/resources/supermoderation/service_confs/prod/package.json"],
                "MODADVERT_SM_API_BINARY": ["modadvert/programs/supermoderation_api/docker_package.json"],
                "MODADVERT_SM_ADMIN_BINARY": ["modadvert/programs/supermoderation_admin/docker_package.json"],
                "MODADVERT_DIRECT_PROXY_BINARY": ["modadvert/programs/direct_proxy/docker_package.json"],
                "MODADVERT_SAAS_INDEXER_BINARY": ["modadvert/programs/saas_indexer/package.json"],
                "MODADVERT_CV_INDEXER_BINARY": ["modadvert/programs/cv_indexer/package.json"],
                "MODADVERT_CREATE_SM_DATABASE_BINARY": ["modadvert/programs/create_tables_for_testing/package.json"],
                "MODADVERT_RUN_TEST_REQUESTS_BINARY": ["modadvert/programs/sm_testing_requests_generator/package.json"],
                "MODADVERT_SAVE_SM_RESULTS_BINARY": ["modadvert/programs/save_sm_results/package.json"],
                "MODADVERT_DIFF_SM_RESULTS_BINARY": ["modadvert/programs/diff_sm_results/package.json"],
            }

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="MODADVERT_BUILD_TASK",
                        ctx={
                            "packages": ";".join([path[0] for path in self.SUPERMODERATION_RESOURCES.values()]),
                            "resource_type": ";".join([res for res in self.SUPERMODERATION_RESOURCES.keys()]),
                            "package_type": "tarball",
                            "use_new_format": True,
                        },
                        out={res: 100 for res in self.SUPERMODERATION_RESOURCES.keys()},
                    )
                )
                for resource_type in self.DOCKER_IMAGES:
                    branch_part.append(
                        jg_build.JobGraphElementYaMakeBuildBranched(
                            task_name="MODADVERT_PUSH_DOCKER_IMAGE",
                            job_params={
                                "job_name_parameter": "PUSH_{}".format(resource_type.upper()),
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="packaged_resource_id",
                                            resource_name=resource_type,
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerNewTag(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="docker_package_checkout_arcadia_from_url",
                                            output_key=sandbox_constants.ARCADIA_URL_KEY,
                                        ),
                                    ),
                                ),
                            ),
                            ctx={
                                "resource_type": "tarball",
                                "resource_type_name": resource_type,
                                "registry_login": "robot-modadvert-test",
                                "vault_item_owner": "robot-modadvert-test",
                                "vault_item_name": "registry-token",
                            }
                        )
                    )
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="MODADVERT_TEST_SUPERMODERATION_B2B",
                        job_params={
                            "job_name_parameter": "RUN_B2B",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="binary_sm_database_creator",
                                        resource_name="MODADVERT_CREATE_SM_DATABASE_BINARY",
                                    ),
                                    jg_job_data.ParentDataResource(
                                        input_key="binary_test_requests_generator",
                                        resource_name="MODADVERT_RUN_TEST_REQUESTS_BINARY",
                                    ),
                                    jg_job_data.ParentDataResource(
                                        input_key="binary_save_sm_results",
                                        resource_name="MODADVERT_SAVE_SM_RESULTS_BINARY",
                                    ),
                                    jg_job_data.ParentDataResource(
                                        input_key="binary_diff_sm_results",
                                        resource_name="MODADVERT_DIFF_SM_RESULTS_BINARY",
                                    ),
                                )
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_type=rm_const.JobTypes.ACTION_RELEASE,
                                job_name_parameter="DEPLOY_UNSTABLE_DEV",
                            ),
                        ),
                        ctx={
                            "yt_proxy_url": "markov",
                            "vault_user": "robot-modadvert-test",
                            "tokens": {
                                "YT_TOKEN": "yt-token",
                                "MODADVERT_CLIENT_ID": "modadvert-client-id",
                                "MODADVERT_CLIENT_SECRET": "modadvert-client-secret"
                            },
                        }
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                release_part.append(
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "supermoderation_api_binary",
                                        "MODADVERT_SM_API_BINARY",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="RUN_B2B",
                            )
                        )
                    )
                )
                push_arrows = []
                for res_type in self.DOCKER_IMAGES:
                    push_arrows.append(
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="PUSH_{}".format(res_type.upper()),
                        )
                    )
                for release_stage, env, force_deploy in [
                    (rm_const.ReleaseStatus.unstable, "DEV", False),
                    (rm_const.ReleaseStatus.unstable, "DEV2", False),
                    (rm_const.ReleaseStatus.prestable, "STAGE", False),
                    (rm_const.ReleaseStatus.testing, "TESTING", False),
                    (rm_const.ReleaseStatus.stable, "PROD", False),
                    (rm_const.ReleaseStatus.prestable, "STAGE", True),
                    (rm_const.ReleaseStatus.testing, "TESTING", True),
                    (rm_const.ReleaseStatus.stable, "PROD", True),
                ]:
                    release_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="MODADVERT_DEPLOY_SUPERMODERATION",
                            job_params={
                                "ctx": {
                                    "vault_user": VAULT_USERS[env],
                                    "st_token_vault_name": "st-token",
                                    "release_environment": env,
                                    "deploy_policy": "DEPLOY",  # TODO: Remove this after qloud shut down.
                                },
                                "job_type": rm_const.JobTypes.ACTION_RELEASE,
                                "job_name_parameter": "DEPLOY_{release_stage}_{env}{forced}".format(
                                    release_stage=release_stage,
                                    env=env,
                                    forced="_FORCED" if force_deploy else "",
                                ),
                            },
                            job_arrows=tuple(push_arrows) + (
                                jg_arrows.ParamsData("release_number", jg_utils.get_major_release_number),
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="saas_indexer_binaries_resource",
                                            resource_name="MODADVERT_SAAS_INDEXER_BINARY",
                                        ),
                                        jg_job_data.ParentDataResource(
                                            input_key="cv_indexer_binaries_resource",
                                            resource_name="MODADVERT_CV_INDEXER_BINARY",
                                        ),
                                        jg_job_data.ParentDataResource(
                                            input_key="supermoderation_config_resource",
                                            resource_name="MODADVERT_SM_CONFIG_DEV_2" if env == "DEV2" else "MODADVERT_SM_CONFIG_{}".format(env),
                                        ),
                                        jg_job_data.ParentDataResource(
                                            input_key="supermoderation_services_config_resource",
                                            resource_name="MODADVERT_SM_SERVICE_CONFIG_DEV" if env == "DEV2" else "MODADVERT_SM_SERVICE_CONFIG_{}".format(env),
                                        ),
                                    )
                                ),
                                jg_job_triggers.JobTriggerNewTag(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="release_tag_number",
                                            output_key="new_tag_number",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerRelease(
                                    job_name_parameter=rm_const.ReleaseStatus.stable,
                                ) if env == "PROD" else None,
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=rm_const.JobTypes.ACTION_RELEASE,
                                    job_name_parameter="DEPLOY_UNSTABLE_DEV2",
                                ) if env == "PROD" else None,
                            ),
                        )
                    )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):

            @property
            def change_frequency(self):
                uppercase_name = self.name.upper()
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "_{}_NEW_TAG".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_RUN_{}_B2B".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_PUSH_MODADVERT_SM_API_BINARY_{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_PUSH_MODADVERT_SM_ADMIN_BINARY_{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_PUSH_MODADVERT_DIRECT_PROXY_BINARY_{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="supermoderation_api_binary",
                    data=ri.SandboxResourceData("MODADVERT_SM_API_BINARY"),
                ),
            ]
        allow_robots_to_release_stable = True
        release_followers_permanent = [
            "robot-modadvert-yt",
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "robot-modadvert-yt"
            queue = "SUPERMODRELEASE"
            dev_queue = "MODDEV"
            summary_template = u"Приемка супермодерации (supermoderation) {}"
            workflow = {}
            followers = [
                "vsuvorov",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "modadvert/releases/supermoderation/"
        dirs = [
            "arcadia/modadvert/libs/supermoderation",
            "arcadia/modadvert/programs/cv_indexer",
            "arcadia/modadvert/programs/direct_proxy",
            "arcadia/modadvert/programs/saas_indexer",
            "arcadia/modadvert/programs/supermoderation_admin",
            "arcadia/modadvert/programs/supermoderation_api",
            "arcadia/modadvert/resources/supermoderation",
        ]
        review_groups = [
            "modadvert",
        ]

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "modadvert/supermoderation/"
