from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core.jg import base as jg_base
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow
from sandbox.projects.release_machine.components.config_core.jg.graph import base as graph_base
from sandbox.projects.release_machine.components.config_core.jg.cube import base as jg_cube
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import internal as internal_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import release as release_cubes
from sandbox.projects.release_machine.components.config_core.jg.preset import basic_build_presets as jg_preset

from sandbox.projects.release_machine.components.configs.supportai import _cubes as supportai_cubes
from sandbox.projects.release_machine.components.configs.supportai import _constants as constants
from sandbox.projects.release_machine.components.configs.supportai import _utils as utils


BUILD_CUBES = {
    "static": supportai_cubes.BuildCubeInfo(
        name="static",
        commands=[
            "set -e",
            "cd $PROJECT_PATH/src",
            "npm install --registry https://npm.yandex-team.ru",
            "cp ./environments/.env.$YANDEX_ENVIRONMENT ./.env",
            "echo '{{\"YANDEX_OAUTH_AUTHORIZATION_TOKEN\": \"\"}}' > ./src/secrets.json",
            "./node_modules/react-scripts/bin/react-scripts.js build",
            "mkdir $RESULT_RESOURCES_PATH/{out_folder_name}",
            "cp -R ./build/* $RESULT_RESOURCES_PATH/{out_folder_name}",
        ],
        resource_type=constants.SANDBOX_SUPPORTAI_FRONT_STATIC,
        release_system=rm_const.DeploySystem.sandbox.name,
    ),
    "configs": supportai_cubes.BuildCubeInfo(
        name="configs",
        commands=[
            "set -e",
            "cd $PROJECT_PATH",
            "mkdir $RESULT_RESOURCES_PATH/{out_folder_name}",
            "mkdir -p $RESULT_RESOURCES_PATH/{out_folder_name}/etc/nginx/sites-enabled",
            "envsubst '$YANDEX_ENVIRONMENT,$RELEASE_VERSION_STR,$WIDGET_CSP_HOSTS' "
            "< ./configs/nginx.conf.template > "
            "$RESULT_RESOURCES_PATH/{out_folder_name}/etc/nginx/sites-enabled/supportai.conf",
        ],
        resource_type=constants.SANDBOX_SUPPORTAI_FRONT_CONFIGS,
        release_system=rm_const.DeploySystem.ya_deploy.name,
    ),
    "proxy": supportai_cubes.BuildCubeInfo(
        name="proxy",
        commands=[
            "set -e",
            "cd $PROJECT_PATH/proxy",
            "npm install --registry https://npm.yandex-team.ru",
            "npm run prepare",
            "mkdir $RESULT_RESOURCES_PATH/{out_folder_name}",
            "cp -R ./* $RESULT_RESOURCES_PATH/{out_folder_name}",
        ],
        resource_type=constants.SANDBOX_SUPPORTAI_FRONT_PROXY,
        release_system=rm_const.DeploySystem.ya_deploy.name,
    ),
    "widget": supportai_cubes.BuildCubeInfo(
        name="widget",
        commands=[
            "set -e",
            "cd $PROJECT_PATH/widget",
            "npm install --registry https://npm.yandex-team.ru",
            "npm run build:widget:$YANDEX_ENVIRONMENT",
            "mkdir $RESULT_RESOURCES_PATH/{out_folder_name}",
            "cp -R ./dist/* $RESULT_RESOURCES_PATH/{out_folder_name}",
            "cp -R ./static/* $RESULT_RESOURCES_PATH/{out_folder_name}",
        ],
        resource_type=constants.SANDBOX_SUPPORTAI_FRONT_WIDGET,
        release_system=rm_const.DeploySystem.sandbox.name,
    ),
}

S3_CUBE_NAME_PATTERN = "upload_{name}_to_s3"
UPLOAD_TO_S3_BUILDS = ["static", "widget"]

FINAL_BUILD_STAGES = ["configs", "proxy"] + [S3_CUBE_NAME_PATTERN.format(name=name) for name in UPLOAD_TO_S3_BUILDS]


class SupporaiFrontendCfg(configs.ReferenceCIConfig):
    name = "supportai_frontend"
    display_name = "SupportAI Frontend"
    robot = "robot-supportai-depl"

    responsible = configs.Responsible(
        abc=configs.Abc(service_name="supportai_frontend"),
        login="nik-bes",
    )

    class JG(jg_base.BaseReleaseMachineJobGraph):
        @property
        def version(self):
            return "${context.version_info.major}-${not_null(context.version_info.minor, `0`)}"

        def _get_release_stage(self, env):
            if env == rm_const.ReleaseStatus.testing:
                return dummy_cubes.TestStageEntry()
            elif env == rm_const.ReleaseStatus.prestable:
                return dummy_cubes.ReleasePrestableStageEntry()
            elif env == rm_const.ReleaseStatus.stable:
                return dummy_cubes.ReleaseStableStageEntry(
                    attributes={
                        "manual": {
                            "enabled": True,
                            "prompt": "Approve after tests",
                            "approvers": {"service": "supportai_frontend", "scope": ["testing", "services_management"]},
                        }
                    }
                )

        def _add_release_stages(self, graph, jg_build_cubes, release_entry, env):
            component_resources = {}

            for release_item in ("proxy", "configs"):
                build = BUILD_CUBES.get(release_item)
                jg_build_cube = jg_build_cubes.get(release_item)

                component_resources[build.resource_type.lower()] = (
                    jg_build_cube.output.resources[build.resource_type].first().id
                )

            release_proxy_cube = release_cubes.ReleaseRmComponent2(
                name=utils.get_cube_name(env, "proxy"),
                title="Release proxy to {}".format(env),
                manual=False,
                where_to_release=env,
                component_name=self.component_name,
                needs=[release_entry],
                input=jg_cube.CubeInput(
                    component_resources=component_resources,
                    deploy_system=rm_const.DeploySystem.ya_deploy.name,
                    wait_for_deploy=True,
                    wait_for_deploy_time_sec=300,
                ),
            )

            graph.add(release_proxy_cube)

            return release_proxy_cube

        def _add_build_stages(self, graph, start_entry, env):
            jg_build_cubes = {}

            for build_cube in BUILD_CUBES.values():
                jg_build_cube = build_cube.get_jg_cube(
                    env=env,
                    version=self.version,
                )

                jg_build_cubes[build_cube.name] = jg_build_cube

                jg_build_cube.add_requirement(start_entry)
                graph.add(jg_build_cube)

            for build_name in UPLOAD_TO_S3_BUILDS:
                build = BUILD_CUBES.get(build_name)
                jg_build_cube = jg_build_cubes.get(build_name)

                cube_name = S3_CUBE_NAME_PATTERN.format(name=build_name)
                upload_to_s3 = supportai_cubes.get_s3_cube(
                    cube_name=cube_name,
                    tar_resource=jg_build_cube.output.resources[build.resource_type].first().id,
                    version=self.version,
                    env=env,
                )

                jg_build_cubes[cube_name] = upload_to_s3

                upload_to_s3.add_requirement(jg_build_cubes.get(build_name))
                graph.add(upload_to_s3)

            return jg_build_cubes

        @jg_flow.release_flow(
            True,
            title='SupportAI Frontend Production',
            stages=[
                jg_preset.DEFAULT_STAGE_NEW_TAG,
                jg_preset.DEFAULT_STAGE_BUILD.add_cube_name(utils.get_cube_name(constants.BUILD_STAGE, "*")),
                jg_preset.DEFAULT_STAGE_PRESTABLE_RELEASE.add_cube_name(
                    utils.get_cube_name(rm_const.ReleaseStatus.prestable, "*")
                ),
                jg_preset.DEFAULT_STAGE_STABLE_RELEASE.add_cube_name(
                    utils.get_cube_name(rm_const.ReleaseStatus.stable, "*")
                ),
            ],
        )
        def release(self):
            graph = super(SupporaiFrontendCfg.JG, self).release(self)

            jg_build_cubes = self._add_build_stages(
                graph, start_entry=graph.get(dummy_cubes.RMMainGraphEntry.NAME), env=rm_const.ReleaseStatus.stable
            )

            release_pre = self._get_release_stage(rm_const.ReleaseStatus.prestable)

            for build_name in FINAL_BUILD_STAGES:
                release_pre.add_requirement(jg_build_cubes.get(build_name))

            # Start prestable release after changelog and ST issue too
            release_pre.add_requirement(graph.get(internal_cubes.LinkFeatureTickets.NAME))
            release_pre.add_requirement(graph.get(internal_cubes.PostChangelogToStartrek.NAME))

            graph.add(release_pre)

            pre_final_cube = self._add_release_stages(
                graph,
                jg_build_cubes=jg_build_cubes,
                release_entry=release_pre,
                env=rm_const.ReleaseStatus.prestable,
            )

            release_stable = self._get_release_stage(rm_const.ReleaseStatus.stable)
            release_stable.add_requirement(pre_final_cube)
            graph.add(release_stable)

            approve_by_manager = dummy_cubes.Dummy(
                name="approve_by_manager",
                attributes={
                    "manual": {
                        "enabled": True,
                        "prompt": "Approve after tests",
                        "approvers": {"service": "supportai_frontend", "scope": "services_management"},
                    }
                },
            )

            approve_by_manager.add_requirement(release_stable)
            graph.add(approve_by_manager)

            self._add_release_stages(
                graph,
                jg_build_cubes=jg_build_cubes,
                release_entry=approve_by_manager,
                env=rm_const.ReleaseStatus.stable,
            )

            return graph

        @jg_flow.supplementary_release_flow(
            auto=True,
            title='SupportAI Frontend Testing',
            name_override='testing_{component_name}',
            stages=[
                jg_preset.DEFAULT_STAGE_BUILD.add_cube_name(utils.get_cube_name(constants.BUILD_STAGE, "*")),
                jg_preset.DEFAULT_STAGE_TEST.add_cube_name(utils.get_cube_name(rm_const.ReleaseStatus.testing, "*")),
            ],
        )
        def testing_supportai_frontend(self):
            main_entry = dummy_cubes.RMMainGraphEntry()
            graph = graph_base.Graph([main_entry])

            jg_build_cubes = self._add_build_stages(graph, start_entry=main_entry, env=rm_const.ReleaseStatus.testing)

            release_entry = self._get_release_stage(rm_const.ReleaseStatus.testing)

            for build_name in FINAL_BUILD_STAGES:
                release_entry.add_requirement(jg_build_cubes.get(build_name))

            graph.add(release_entry)

            self._add_release_stages(
                graph,
                jg_build_cubes=jg_build_cubes,
                release_entry=release_entry,
                env=rm_const.ReleaseStatus.testing,
            )

            return graph

    class CI(configs.ReferenceCIConfig.CI):
        a_yaml_dir = constants.PROJECT_PATH
        secret = constants.SECRET
        sb_owner_group = constants.SB_OWNER_GROUP
        ya_make_abs_paths_glob = [
            utils.get_path("**"),
        ]

    class Releases(configs.ReferenceCIConfig.Releases):
        @property
        def releasable_items(self):
            return [build_cube.get_releasable_item() for build_cube in BUILD_CUBES.values()]

        allow_old_releases = True
        allow_robots_to_release_stable = True

    class Notify(configs.ReferenceCIConfig.Notify):
        class Startrek(configs.ReferenceCIConfig.Notify.Startrek):
            assignee = "robot-supportai-depl"
            queue = "SUPPORTAI"
            ticket_type = "Release"
            summary_template = "SupportAI Frontend Release {}"
            followers = []
            notify_on_release_to_release_st_ticket = frozenset(
                [rm_const.ReleaseStatus.stable, rm_const.ReleaseStatus.prestable]
            )

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = None
        dirs = [
            constants.PROJECT_PATH,
        ]
