# -*- coding: utf-8 -*-

from sandbox.projects.release_machine.components import configs as cfg
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components.job_graph import (
    job_triggers as jg_job_triggers,
    job_arrows as jg_arrows,
    job_data as jg_job_data,
    utils as jg_utils,
)

REGISTRY_TAG_TEMPLATE = "registry.yandex.net/toloka-dj-basesearch-server:{maj_rel}.{min_rel}"

QLOUD_ENVIRONMENTS = {
    rm_const.ReleaseStatus.stable: "toloka.tqms.toloka-dj-production.basesearch",
    rm_const.ReleaseStatus.prestable: "toloka.tqms.toloka-dj-prestable.basesearch",
}


class ComponentNameCfg(cfg.ReferenceBranchedConfig):
    name = "toloka_dj"
    display_name = "Toloka DJ"
    responsible = "annvas"

    class MergesCfg(cfg.ReferenceBranchedConfig.MergesCfg):
        number_last_branches_to_merge = 1

    class Testenv(cfg.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "TOLOKA_ANALYTICS_DEV"
        trunk_db = "toloka_dj-trunk"

        class JobGraph(cfg.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _branch_part(self):
                """
                    Part of job graph for Testenv with jobs in branch, i.e. tests, builds.
                    :return: list with jobs
                """
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="BUILD_DOCKER_IMAGE_V6",
                        out={},
                        ctx={
                            "docker_package_json": "dj/services/toloka/docker/package.json",
                            "registry_login": "robot-toloka",
                            "vault_item_name": "robot-toloka-registry-token",
                            "release_type": "stable",
                            "use_aapi_fuse": True,
                        },
                        job_arrows=[
                            jg_job_triggers.JobTriggerNewTag([
                                jg_job_data.ParentDataOutput("docker_package_checkout_arcadia_from_url",
                                                             "checkout_arcadia_from_url"),
                            ]),
                            jg_arrows.ParentsData(
                                "registry_tags",
                                triggers=[
                                    jg_job_triggers.JobTriggerNewTag([
                                        jg_job_data.ParentDataOutput("major_release", "branch_number_for_tag"),
                                        jg_job_data.ParentDataOutput("minor_release", "new_tag_number"),
                                    ])
                                ],
                                transform=lambda major_release, minor_release: REGISTRY_TAG_TEMPLATE.format(
                                    maj_rel=major_release, min_rel=minor_release),
                                override=True,
                            )
                        ],
                    ),
                )
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="RUN_TEAMCITY_BUILD",
                        build_item="TEAMCITY_BUILD_TRUNK",
                        ctx={
                            "build_task_id": "AppliedSystems_Toloka_TolokaLibs_TolokaBackendSingleMicroservice",
                            "build_timeout": 60 * 40,  # 40 minutes
                            "teamcity_parameters": {
                                "micros-name": "tqms-toloka-pfhs"
                            }
                        },
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release

                for stage, qloud_environment in QLOUD_ENVIRONMENTS.items():
                    release_part.append(jg_release.JobGraphElementReleaseBase(
                        task_name="RELEASE_RM_COMPONENT_2",
                        release_to=stage,
                        job_arrows=(
                            jg_arrows.ParentsData(
                                "registry_name",
                                triggers=[
                                    jg_job_triggers.JobTriggerNewTag([
                                        jg_job_data.ParentDataOutput("major_release", "branch_number_for_tag"),
                                        jg_job_data.ParentDataOutput("minor_release", "new_tag_number"),
                                    ])
                                ],
                                transform=lambda major_release, minor_release: REGISTRY_TAG_TEMPLATE.format(
                                    maj_rel=major_release, min_rel=minor_release),
                                override=True,
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="registry_name",
                                        output_key="output_resource_version",
                                    )
                                )
                            ),
                            jg_job_triggers.JobTriggerNewTag([
                                jg_job_data.ParentDataOutput("major_release_num", "branch_number_for_tag"),
                                jg_job_data.ParentDataOutput("minor_release_num", "new_tag_number"),
                            ]),
                        ),
                        job_params={
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "ctx": {
                                "deploy_system": "qloud",
                                "qloud_vault_owner": "robot-toloka",
                                "qloud_vault_name": "robot-toloka-qloud-token",
                                "qloud_environment": qloud_environment,
                            }
                        },
                    ))
                    release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                        release_to=stage,
                        job_arrows=(
                            jg_job_triggers.JobTriggerRelease(job_name_parameter=stage),
                        ),
                    ))

                return release_part

    class Notify(cfg.ReferenceBranchedConfig.Notify):
        """ Notification settings """

        class Startrek(cfg.ReferenceBranchedConfig.Notify.Startrek):
            """
            Startrek notifications

            https://wiki.yandex-team.ru/releasemachine/startrekoptions/
            """
            queue = "AEA"
            summary_template = "Toloka DJ Release {}"
            assignee = "robot-toloka"

    class ChangelogCfg(cfg.ReferenceBranchedConfig.ChangelogCfg):
        """
        Changelog configuration

        https://wiki.yandex-team.ru/releasemachine/changelogs/
        """
        dirs = ["arcadia/dj/services/toloka"]
        wiki_page = "assessment/Data-Analysis-And-Research-Group/Toloka/Machine-Learning/DJ/Releases/"

    class Releases(cfg.ReferenceBranchedConfig.Releases):
        token_vault_owner = "robot-toloka"
        token_vault_name = "robot-toloka-registry-token"
        resources_info = []
